package com.bringspring.system.msgcenter.service.context;

import com.bringspring.common.base.UserInfo;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.entity.McMsgTemplateEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import lombok.Data;

import java.util.*;

/**
 * 发送消息上下文
 * <p>
 * 典型使用场景
 * 1.跨组件数据传递：在责任链、管道过滤器、策略模式中，多个处理单元需要共享数据。
 * 2.流程状态管理：记录处理过程中的中间状态（如校验结果、转换后的数据）。
 * 3.解耦组件依赖：避免处理类直接依赖具体参数，而是通过上下文接口获取数据。
 * 4.扩展性支持：动态添加元数据，支持未来功能扩展。
 * <p>
 * 核心设计要点
 * 1. 数据封装设计：基础数据通过强类型方法暴露;扩展数据通过通用 Map 存储。
 * 2. 线程安全设计：为每个请求创建独立上下文（请求级别作用域）.
 * 3. 不可变设计（可选）：构建后不可修改。
 * 4. 生命周期管理：
 * <p>
 * 关键注意事项
 * 1. 避免过度膨胀：上下文对象应尽量简单，避免包含过多数据。
 * 2. 防御性拷贝：防止外部修改内部状态。
 * 3. 空值安全：使用 Optional 包装可能为空的值。
 * 4. 序列化支持：上下文对象应实现 Serializable 接口，以便于跨线程、跨网络传递。
 */
@Data
public class SendMessageContext {
    private boolean isAgain;
    private String toType;
    private McMsgSendTemplateModel sendTemplate;
    private McTaskMsgContentEntity taskMsgContent;
    private List<McTaskMsgReceiveEntity> receiveList;
    // 防止外部修改内部状态
    public List<McTaskMsgReceiveEntity> getReceiveList() {
        return Collections.unmodifiableList(receiveList);
    }

    public void setReceiveList(List<McTaskMsgReceiveEntity> newReceiveList) {
        this.receiveList = new ArrayList<>(newReceiveList); // 拷贝新集合
    }

    private UserInfo currentUser;
    // 消息模板
    private McMsgTemplateEntity templateInfo;
    // 渠道应用授权账户
    private McMsgAccountEntity msgAccount;
    // 第三方信息映射
    private Map<String, SynThirdInfoEntity> thirdInfoMap = new HashMap<>();
    private List<McTaskMsgReceiveEntity> validList;
    private List<McTaskMsgReceiveEntity> invalidList = new ArrayList<>();
    /**
     * 动态添加元数据，支持未来功能扩展
     */
    private Map<String, Object> metadata = new HashMap<>();

    // 扩展数据通过通用 Map 存储
    public Object getAttribute(String key) {
        return metadata.get(key);
    }

    public void setAttribute(String key, Object value) {
        metadata.put(key, value);
    }

}
