package com.bringspring.system.msgcenter.service.strategy;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.DateUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.msgcenter.constant.Constants.Delimiters;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.entity.McMsgTemplateEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.enums.EnabledMarkEnum;
import com.bringspring.system.msgcenter.enums.MsgCenterCode;
import com.bringspring.system.msgcenter.enums.ToTypeEnum;
import com.bringspring.system.msgcenter.exception.SendMessageException;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.service.McBusinessService;
import com.bringspring.system.msgcenter.service.McMsgAccountService;
import com.bringspring.system.msgcenter.service.McMsgTemplateService;
import com.bringspring.system.msgcenter.service.McTaskMsgReceiveService;
import com.bringspring.system.msgcenter.service.chain.ChainChannelType;
import com.bringspring.system.msgcenter.service.chain.FilterChain;
import com.bringspring.system.msgcenter.service.chain.FilterChainFactory;
import com.bringspring.system.msgcenter.service.context.SendMessageContext;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.ListUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;

import static com.bringspring.system.msgcenter.enums.ChannelTypeEnum.getByCode;

/**
 * 定义抽象策略基类
 * 将公共方法（如getTaskMsgContent、packThirdReceive）提取到抽象类中，供具体策略复用
 */
@Slf4j
public abstract class AbstractMessageSender implements MessageSender {
    @Autowired
    protected McBusinessService mcBusinessService;
    @Autowired
    protected McMsgTemplateService mcMsgTemplateService;
    @Autowired
    protected McMsgAccountService mcMsgAccountService;
    @Autowired
    protected SynThirdInfoService synThirdInfoService;
    @Autowired
    protected McTaskMsgReceiveService mcTaskMsgReceiveService;

    @Autowired
    protected FilterChainFactory filterChainFactory;


    @Override
    public void send(boolean isAgain, String toType,
                     McMsgSendTemplateModel sendTemplate,
                     McTaskMsgContentEntity taskMsgContent,
                     List<McTaskMsgReceiveEntity> receiveList,
                     UserInfo currentUserInfo) {

        List<List<McTaskMsgReceiveEntity>> batches = ListUtils.partition(receiveList, getBatchSize());

        if (ToTypeEnum.NON_USER.getCode().equals(toType)) {
            // 接收人为非系统用户
            batches.forEach(
                    batch -> sendBatchNonSystemUser(isAgain, sendTemplate, taskMsgContent, batch, currentUserInfo)
            );
        } else {
            batches.forEach(batch -> sendBatch(isAgain, sendTemplate, taskMsgContent, batch, currentUserInfo));
        }
    }

    @Override
    public void send(SendMessageContext context) {
        executeSend(context);
    }

    // 模板方法：统一处理发送流程
    protected final void executeSend(SendMessageContext context) {

        // 1. 前置校验
        commonPreCheck(context);

        // 2. 处理链处理（黑名单/接收人/第三方信息),依次标记无法正常下发的接收人.
        processReceivers(context);

        // 3. 发送消息
        doSend(context);

        // 4. 处理结果
        // 被标记无法正常下发的接收人
        processSendResult(context);
    }

    /**
     * 处理发送结果
     */
    private void processSendResult(SendMessageContext context) {
        List<McTaskMsgReceiveEntity> updateBatch = new ArrayList<>();

        List<McTaskMsgReceiveEntity> invalidList = context.getInvalidList();
        List<McTaskMsgReceiveEntity> validList = context.getValidList();
        if (CollectionUtil.isNotEmpty(invalidList)) {
            invalidList.forEach(invalid -> updateReceiver(invalid, context.isAgain(), context.getCurrentUser()));
            updateBatch.addAll(invalidList);
        }
        if (CollectionUtil.isNotEmpty(validList)) {
            validList.forEach(valid -> updateReceiver(valid, context.isAgain(), context.getCurrentUser()));
            updateBatch.addAll(validList);
        }

        if (CollectionUtil.isNotEmpty(updateBatch)) {
            mcTaskMsgReceiveService.updateBatchById(updateBatch);
        }
    }

    /**
     * 公共更新方法
     * 统一处理实体更新逻辑
     *
     * @param receiver
     * @param isAgain
     * @param currentUser
     */
    private void updateReceiver(McTaskMsgReceiveEntity receiver, boolean isAgain, UserInfo currentUser) {
        // 防御性校验
        if (receiver == null || currentUser == null) {
            throw new IllegalArgumentException("参数不可为空");
        }

        // 设置时间戳
        if (isAgain) {
            receiver.setAgainTime(DateUtil.getNowDate());
            receiver.setAgainUserId(currentUser.getUserId());
        } else {
            receiver.setSendTime(DateUtil.getNowDate());
            receiver.setSendUserId(currentUser.getUserId());
        }

        // 设置状态描述
        Integer enabledMark = receiver.getEnabledMark();
        String prefix = getPrefix(isAgain, enabledMark);
        String userId = StringUtils.defaultString(receiver.getDescription(), "");
        receiver.setDescription(String.join(Delimiters.VERTICAL, prefix, userId));

        // 处理重试次数
        if (isAgain) {
            Integer retryCount = receiver.getAutoAgainNum();
            retryCount = (retryCount != null) ? retryCount : 0;
            receiver.setAutoAgainNum(EnabledMarkEnum.SUCCESS.getCode().equals(enabledMark) ? 0 : retryCount + 1);
        }
    }

    /**
     * 状态码映射逻辑
     *
     * @param isAgain
     * @param enabledMark
     * @return
     */
    private String getPrefix(boolean isAgain, Integer enabledMark) {
        if (EnabledMarkEnum.SUCCESS.getCode().equals(enabledMark)) {
            return isAgain ? MsgCenterCode.RESEND_SUCCESS.getDesc() : MsgCenterCode.SEND_SUCCESS.getDesc();
        } else if (EnabledMarkEnum.FAIL.getCode().equals(enabledMark)) {
            return isAgain ? MsgCenterCode.RESEND_FAILURE.getDesc() : MsgCenterCode.SEND_FAILURE.getDesc();
        }
        return "";
    }

    // 子类必须实现的抽象方法
    protected abstract void doSend(SendMessageContext context) throws SendMessageException;

    /**
     * 公共校验方法
     *
     * @param context
     */
    private void commonPreCheck(SendMessageContext context) {
        if (ObjectUtil.isEmpty(context.getTaskMsgContent())) {
            throw new SendMessageException(2, "消息内容为空");
        }

        validateParameters(context);
    }

    /**
     * 抽象方法，子类实现具体的校验逻辑，实现个性化校验。
     *
     * @param context
     */
    protected abstract void validateParameters(SendMessageContext context);

    /**
     * 发送配置消息模板校验
     * 邮箱、短信、微信、钉钉、企业微信等消息发送配置校验
     *
     * @param context
     */
    public void validateTemplate(SendMessageContext context) {
        // 消息模板
        McMsgTemplateEntity templateInfo = mcMsgTemplateService.getInfo(context.getSendTemplate().getTemplateId());
        if (ObjectUtil.isNull(templateInfo)) {
            throw new SendMessageException(2, "消息模板不存在");
        }
        context.setTemplateInfo(templateInfo);
    }

    /**
     * 发送配置应用账号校验
     * 邮箱、短信、微信、钉钉、企业微信、webhook等消息发送配置校验
     *
     * @param context
     */
    public void validateAccount(SendMessageContext context) {
        // 账号
        McMsgAccountEntity msgAccount = mcMsgAccountService.getInfo(context.getSendTemplate().getAccountConfigId());
        if (ObjectUtil.isNull(msgAccount)) {
            throw new SendMessageException(2, "应用账号不存在");
        }
        context.setMsgAccount(msgAccount);
    }

    public void processReceivers(SendMessageContext context) {
        // 构建处理链
        FilterChain chain;
        String messageType = context.getSendTemplate().getMessageType();


        final String toType = context.getToType();
        if (ToTypeEnum.NON_USER.getCode().equals(toType)) {
            // 接收人为非系统用户
            chain = filterChainFactory.createChain(ChainChannelType.NON_SYS_USER);
        } else {
            ChannelTypeEnum typeEnum = getByCode(messageType);
            switch (typeEnum) {
                case SYS_MSG:// 站内消息
                    chain = filterChainFactory.createChain(ChainChannelType.SYS_MSG);
                    break;
                case SMS_MSG: // 发送短信
                    chain = filterChainFactory.createChain(ChainChannelType.SMS);
                    break;
                case DING_MSG: // 钉钉
                    chain = filterChainFactory.createChain(ChainChannelType.DING);
                    break;
                case QY_MSG:
                case WX_MINIAPP:
                case WX_MP:
                    chain = filterChainFactory.createChain(ChainChannelType.WECHAT);
                    break;
                case WEBHOOK:
                    chain = filterChainFactory.createChain(ChainChannelType.WEBHOOK);
                    break;
                default:
                    chain = filterChainFactory.createChain(ChainChannelType.DEFAULT);
                    break;
            }
        }

        chain.process(context);
    }

    /**
     * 发送批量消息
     *
     * @param isAgain
     * @param sendTemplate
     * @param taskMsgContent
     * @param receiveList
     * @param currentUserInfo
     */
    protected abstract void sendBatch(boolean isAgain,
                                      McMsgSendTemplateModel sendTemplate,
                                      McTaskMsgContentEntity taskMsgContent,
                                      List<McTaskMsgReceiveEntity> receiveList,
                                      UserInfo currentUserInfo);

    /**
     * 发送批量消息-非系统用户
     *
     * @param isAgain
     * @param sendTemplate
     * @param taskMsgContent
     * @param receiveList
     * @param currentUserInfo
     */
    protected abstract void sendBatchNonSystemUser(boolean isAgain,
                                                   McMsgSendTemplateModel sendTemplate,
                                                   McTaskMsgContentEntity taskMsgContent,
                                                   List<McTaskMsgReceiveEntity> receiveList,
                                                   UserInfo currentUserInfo);

    /**
     * 获取批量发送的大小
     *
     * @return
     */
    protected abstract int getBatchSize();

    /**
     * 获取消息发送渠道类型
     *
     * @return
     */
    public abstract ChannelTypeEnum getChannelType();

}
