package com.bringspring.system.msgcenter.util;

import com.aliyun.dysmsapi20170525.Client;
import com.aliyun.dysmsapi20170525.models.QuerySmsTemplateRequest;
import com.aliyun.dysmsapi20170525.models.QuerySmsTemplateResponse;
import com.aliyun.dysmsapi20170525.models.SendSmsRequest;
import com.aliyun.dysmsapi20170525.models.SendSmsResponse;
import com.aliyun.teaopenapi.models.Config;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.ParameterUtil;
import com.bringspring.common.util.StringUtils;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static com.bringspring.system.msgcenter.constant.Constants.MessageConfig;

/**
 * 阿里云发送短信
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2021/4/21 11:45
 */
@Slf4j
public class SmsAliYunSentUtil {

    // 定义常量
    private static final String SMS_FAILURE_PREFIX = "发送短信失败：";
    private static final String QUERY_TEMPLATE_ERROR = "查询阿里云短信模板错误：";
    private static final String CREATE_CLIENT_ERROR = "创建阿里云短信客户端错误：";

    // 私有构造函数防止实例化
    private SmsAliYunSentUtil() {
        throw new UnsupportedOperationException("工具类不允许实例化");
    }

    private static Client createClient(String accessKeyId, String accessKeySecret, String endpoint) {
        try {
            Config config = new Config()
                    // 您的AccessKey ID
                    .setAccessKeyId(accessKeyId)
                    // 您的AccessKey Secret
                    .setAccessKeySecret(accessKeySecret);
            // 访问的域名
            config.endpoint = endpoint;
            return new Client(config);
        } catch (Exception e) {
            log.error(CREATE_CLIENT_ERROR + e.getMessage());
        }
        return null;
    }

    /**
     * 查询短信模板详情
     *
     * @param accessKeyId
     * @param accessKeySecret
     * @param endpoint
     * @param templateId
     */
    public static List<String> querySmsTemplateRequest(String accessKeyId, String accessKeySecret, String endpoint,
                                                       String templateId) {
        try {
            Client client = createClient(accessKeyId, accessKeySecret, endpoint);
            if (client == null) {
                return Collections.emptyList();
            }

            QuerySmsTemplateResponse response = client.querySmsTemplate(
                    new QuerySmsTemplateRequest().setTemplateCode(templateId)
            );

            String templateContent = response.getBody().templateContent;
            if (StringUtils.isNotEmpty(templateContent)) {
                List<String> parameters = new ArrayList<>();
                ParameterUtil.parse("${", "}", templateContent, parameters);
                return parameters;
            }
        } catch (Exception e) {
            log.error(QUERY_TEMPLATE_ERROR + e.getMessage());
        }
        return Collections.emptyList();
    }


    /**
     * 查询短信模板详情
     *
     * @param accessKeyId
     * @param accessKeySecret
     * @param endpoint
     * @param templateId
     */
    public static String querySmsTemplateContent(String accessKeyId, String accessKeySecret, String endpoint,
                                                 String templateId) {
        try {
            Client client = createClient(accessKeyId, accessKeySecret, endpoint);
            if (client == null) {
                return "";
            }

            QuerySmsTemplateResponse response = client.querySmsTemplate(
                    new QuerySmsTemplateRequest().setTemplateCode(templateId)
            );
            return response.getBody().templateContent;
        } catch (Exception e) {
            log.error(QUERY_TEMPLATE_ERROR + e.getMessage());
        }
        return "";
    }

    /**
     * 发送短信
     *
     * @param accessKeyId
     * @param accessKeySecret
     * @param endpoint
     * @param phoneNumbers
     * @param signContent
     * @param templateId
     * @param map
     * @return
     */
    public static String sentSms(String accessKeyId, String accessKeySecret, String endpoint, String phoneNumbers,
                                 String signContent, String templateId, Map<String, Object> map) {
        // 复制代码运行请自行打印 API 的返回值
        try {
            Client client = createClient(accessKeyId, accessKeySecret, endpoint);
            if (client == null) {
                return SMS_FAILURE_PREFIX + "客户端初始化失败";
            }

            SendSmsRequest request = new SendSmsRequest()
                    .setPhoneNumbers(phoneNumbers) // 接收者的号码
                    .setSignName(signContent) // 签名
                    .setTemplateCode(templateId) // 模板id
                    .setTemplateParam(JsonUtil.getObjectToString(map)); // 模板参数

            SendSmsResponse response = client.sendSms(request);
            if (!MessageConfig.SMS_OK.equalsIgnoreCase(response.body.code)) {
                String errorMsg = SMS_FAILURE_PREFIX + response.getBody().message;
                log.error(errorMsg);
                return errorMsg;
            }
            return MessageConfig.SMS_OK;
        } catch (Exception e) {
            String errorMsg = SMS_FAILURE_PREFIX + e.getMessage();
            log.error(errorMsg);
            return errorMsg;
        }
    }
}