package com.bringspring.system.base.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.model.UserLogForm;
import com.bringspring.common.util.*;
import com.bringspring.system.base.entity.LogEntity;
import com.bringspring.system.base.enums.LogSortEnum;
import com.bringspring.system.base.mapper.LogMapper;
import com.bringspring.system.base.model.BurialPoint.VisitVO;
import com.bringspring.system.base.model.logs.PaginationLogModel;
import com.bringspring.system.base.service.LogService;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 系统日志
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Service
public class LogServiceImpl extends ServiceImpl<LogMapper, LogEntity> implements LogService {

    @Autowired
    private UserProvider userProvider;
    @Autowired
    private LogMapper logMapper;
    private List<DateTime> dateTimes;

    @Override
    public List<LogEntity> getList(int category, PaginationLogModel paginationTime) {
        UserInfo userInfo = userProvider.get();
        QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(LogEntity::getCategory, category);
        //日期范围（近7天、近1月、近3月、自定义）
        String startTime = paginationTime.getStartTime() != null ? paginationTime.getStartTime() : null;
        String endTime = paginationTime.getEndTime() != null ? paginationTime.getEndTime() : null;
        if (!StringUtils.isEmpty(startTime) && !StringUtils.isEmpty(endTime)) {
            Date startTimes = DateUtil.stringToDate(DateUtil.daFormatYmd(Long.parseLong(startTime)) + " 00:00:00");
            Date endTimes = DateUtil.stringToDate(DateUtil.daFormatYmd(Long.parseLong(endTime)) + " 23:59:59");
            queryWrapper.lambda().ge(LogEntity::getCreatorTime, startTimes).le(LogEntity::getCreatorTime, endTimes);
        }
        //关键字（用户、IP地址、功能名称）
        String keyWord = paginationTime.getKeyword() != null ? paginationTime.getKeyword() : null;
        if (!StringUtils.isEmpty(keyWord)) {
            queryWrapper.lambda().and(
                    t -> t.like(LogEntity::getUserName, keyWord)
                            .or().like(LogEntity::getIpAddress, keyWord)
                            .or().like(LogEntity::getModuleName, keyWord)
                            .or().like(LogEntity::getRequestUrl, keyWord)
            );
        }
        //用户Id
        String userId = userInfo.getUserId() != null ? userInfo.getUserId() : null;
        String userAccount = userInfo.getUserAccount() != null ? userInfo.getUserAccount() : null;
        if (!StringUtils.isEmpty(userId) && !StringUtils.isEmpty(userAccount)) {
            if (!userInfo.getIsAdministrator()) {
                queryWrapper.lambda().and(
                        t -> t.eq(LogEntity::getUserId, userId)
                                .or().eq(LogEntity::getUserId, userAccount)
                );
            }
        }
        // 操作ip
        if (StringUtils.isNotEmpty(paginationTime.getIpaddress())) {
            queryWrapper.lambda().like(LogEntity::getIpAddress, paginationTime.getIpaddress());
        }
        // 操作模块
        if (StringUtils.isNotEmpty(paginationTime.getModuleName())) {
            queryWrapper.lambda().like(LogEntity::getModuleName, paginationTime.getModuleName());
        }
        // 操作类型
        if (StringUtils.isNotEmpty(paginationTime.getRequestMethod())) {
            queryWrapper.lambda().eq(LogEntity::getRequestMethod, paginationTime.getRequestMethod());
        }
        //排序
        if (StringUtils.isEmpty(paginationTime.getSidx())) {
            queryWrapper.lambda().orderByDesc(LogEntity::getCreatorTime);
        } else {
            try {
                String sidx = paginationTime.getSidx();
                LogEntity logEntity = new LogEntity();
                Field declaredField = logEntity.getClass().getDeclaredField(sidx);
                declaredField.setAccessible(true);
                String value = declaredField.getAnnotation(TableField.class).value();
                queryWrapper = "asc".equals(paginationTime.getSort().toLowerCase()) ?
                        queryWrapper.orderByAsc(value) : queryWrapper.orderByDesc(value);
            } catch (NoSuchFieldException e) {
                e.printStackTrace();
            }
        }
        Page<LogEntity> page = new Page<>(paginationTime.getCurrentPage(), paginationTime.getPageSize());
        IPage<LogEntity> userPage = this.page(page, queryWrapper);
        return paginationTime.setData(userPage.getRecords(), page.getTotal());
    }

    @Override
    public List<LogEntity> getList(UserLogForm userLogForm) {
        UserInfo userInfo = userProvider.get();
        QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(LogEntity::getCategory, userLogForm.getCategory());
        //日期范围（近7天、近1月、近3月、自定义）
        if (!StringUtils.isEmpty(userLogForm.getStartTime()) && !StringUtils.isEmpty(userLogForm.getEndTime())) {
            Date startTimes =
                    DateUtil.stringToDate(DateUtil.daFormatYmd(Long.parseLong(userLogForm.getStartTime())) + " 00:00" +
                            ":00");
            Date endTimes = DateUtil.stringToDate(DateUtil.daFormatYmd(Long.parseLong(userLogForm.getEndTime())) + " " +
                    "23:59:59");
            queryWrapper.lambda().ge(LogEntity::getCreatorTime, startTimes).le(LogEntity::getCreatorTime, endTimes);
        }
        //关键字（用户、IP地址、功能名称）
        String keyWord = userLogForm.getKeyword();
        if (!StringUtils.isEmpty(keyWord)) {
            queryWrapper.lambda().and(
                    t -> t.like(LogEntity::getUserName, keyWord)
                            .or().like(LogEntity::getIpAddress, keyWord)
                            .or().like(LogEntity::getModuleName, keyWord)
            );
        }
        //用户Id
        String userId = userInfo.getUserId() != null ? userInfo.getUserId() : null;
        String userAccount = userInfo.getUserAccount() != null ? userInfo.getUserAccount() : null;
        if (!StringUtils.isEmpty(userId) && !StringUtils.isEmpty(userAccount)) {
            queryWrapper.lambda().and(
                    t -> t.eq(LogEntity::getUserId, userId)
                            .or().eq(LogEntity::getUserId, userAccount)
            );
        }
        //排序
        if (StringUtils.isEmpty(userLogForm.getSidx())) {
            queryWrapper.lambda().orderByDesc(LogEntity::getCreatorTime);
        } else {
            queryWrapper = "asc".equals(userLogForm.getSort().toLowerCase()) ?
                    queryWrapper.orderByAsc(userLogForm.getSidx()) : queryWrapper.orderByDesc(userLogForm.getSidx());
        }
        Page<LogEntity> page = new Page<>(userLogForm.getCurrentPage(), userLogForm.getPageSize());
        IPage<LogEntity> userPage = this.page(page, queryWrapper);
        return userLogForm.setData(userPage.getRecords(), page.getTotal());
    }

    @Override
    public LogEntity getInfo(String id) {
        QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(LogEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    @DSTransactional
    public boolean delete(String[] ids) {
        if (ids.length > 0) {
            QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(LogEntity::getId, ids);
            return this.remove(queryWrapper);
        }
        return false;
    }

    @Override
    public void writeLogAsync(String userId, String userName, String abstracts) {
        LogEntity entity = new LogEntity();
        entity.setId(RandomUtil.uuId());
        entity.setUserId(userId);
        entity.setUserName(userName);
        entity.setAbstracts(abstracts);
        entity.setRequestUrl(ServletUtils.getServletPath());
        entity.setRequestMethod(ServletUtils.getRequest().getMethod());
        entity.setIpAddress(IpUtil.getIpAddr());
        entity.setPlatForm(ServletUtils.getUserAgent());
        entity.setCategory(LogSortEnum.Login.getCode());
        this.save(entity);
    }

    @Override
    public void writeLogAsync(LogEntity entity) {
        entity.setId(RandomUtil.uuId());
        this.save(entity);
    }

    @Override
    public void deleteHandleLog(String type) {
        QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(LogEntity::getCategory, type);
        this.remove(queryWrapper);
    }

    @Override
    public void autoDeleteLog() {
        // 1-登录日志 3-操作日志 4-异常日志 5-请求日志
        Integer[] integers = {LogSortEnum.Login.getCode(), LogSortEnum.Request.getCode()};
        QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().in(LogEntity::getCategory, integers);
        queryWrapper.lambda().lt(LogEntity::getCreatorTime, DateUtil.offsetMonth(new Date(), -1));
        this.remove(queryWrapper);
    }

    @Override
    public Set<String> queryList() {
        QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(LogEntity::getCategory, 3);
        return this.list(queryWrapper).size() > 0 ?
                this.list(queryWrapper).stream().map(t -> t.getModuleName()).collect(Collectors.toSet()) :
                new HashSet<>(16);
    }

    private QueryWrapper<LogEntity> getQueryWrapper(String category, PaginationLogModel logModel){
        QueryWrapper<LogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(LogEntity::getCategory, category);
        String startTime = logModel.getStartTime() != null ? logModel.getStartTime() : null;
        String endTime = logModel.getEndTime() != null ? logModel.getEndTime() : null;
        if (!StringUtils.isEmpty(startTime) && !StringUtils.isEmpty(endTime)) {
            Date startTimes = DateUtil.stringToDate(startTime + ":00");
            Date endTimes = DateUtil.stringToDate(endTime + ":00");
            queryWrapper.lambda().ge(LogEntity::getCreatorTime, startTimes).le(LogEntity::getCreatorTime, endTimes);
            dateTimes = DateUtil.rangeToList(startTimes, endTimes, DateField.MINUTE);
        } else {
            DateTime end = DateUtil.parse(DateUtil.getmmNow());
            Date start = DateUtil.dateAddMinutes(end, -29);
            queryWrapper.lambda().ge(LogEntity::getCreatorTime, start).le(LogEntity::getCreatorTime, end);
            dateTimes = DateUtil.rangeToList(start, end, DateField.MINUTE);
        }
        return queryWrapper;
    }

    @Override
    public List<Map<String, Object>> selectUserGroupByCreatorTime(String category, PaginationLogModel logModel) {
        List<Map<String, Object>> resultMap = new ArrayList<>();
        QueryWrapper<LogEntity> queryWrapper = this.getQueryWrapper(category, logModel);
        List<Map<String, Object>> maps = logMapper.selectUserGroupByCreatorTime(queryWrapper);
        for (DateTime dateTime : dateTimes) {
            String dateStr = DateUtil.getDateString(dateTime, "HH:mm");
            Map<String, Object> sendTime =
                    maps.stream().filter(t -> t.get("creator_time").equals(dateStr)).findFirst().orElse(null);
            if (CollectionUtil.isEmpty(sendTime)) {
                sendTime = new HashMap<>();
                sendTime.put("creator_time", dateStr);
                sendTime.put("scount", 0);
            }
            resultMap.add(sendTime);
        }
        return resultMap;
    }

    @Override
    public List<Map<String, Object>> selectModuleGroupByCreatorTime(String category, PaginationLogModel logModel) {
        List<Map<String, Object>> resultMap = new ArrayList<>();
        QueryWrapper<LogEntity> queryWrapper = this.getQueryWrapper(category, logModel);
        List<Map<String, Object>> maps = logMapper.selectModuleGroupByCreatorTime(queryWrapper);
        for (DateTime dateTime : dateTimes) {
            String dateStr = DateUtil.getDateString(dateTime, "HH:mm");
            Map<String, Object> sendTime =
                    maps.stream().filter(t -> t.get("creator_time").equals(dateStr)).findFirst().orElse(null);
            if (CollectionUtil.isEmpty(sendTime)) {
                sendTime = new HashMap<>();
                sendTime.put("creator_time", dateStr);
                sendTime.put("scount", 0);
            }
            resultMap.add(sendTime);
        }
        return resultMap;
    }

    @Override
    public List<Map<String, Object>> selectGroupByCreatorTime(String category, PaginationLogModel logModel) {
        List<Map<String, Object>> resultMap = new ArrayList<>();
        QueryWrapper<LogEntity> queryWrapper = this.getQueryWrapper(category, logModel);
        List<Map<String, Object>> maps = logMapper.selectGroupByCreatorTime(queryWrapper);
        for (DateTime dateTime : dateTimes) {
            String dateStr = DateUtil.getDateString(dateTime, "HH:mm");
            Map<String, Object> sendTime =
                    maps.stream().filter(t -> t.get("creator_time").equals(dateStr)).findFirst().orElse(null);
            if (CollectionUtil.isEmpty(sendTime)) {
                sendTime = new HashMap<>();
                sendTime.put("creator_time", dateStr);
                sendTime.put("scount", 0);
            }
            resultMap.add(sendTime);
        }
        return resultMap;
    }

    @Override
    public List<Map<String, Object>> selectMixedGroupByCreatorTime(String category, PaginationLogModel logModel) {
        List<Map<String, Object>> resultMap = new ArrayList<>();
        QueryWrapper<LogEntity> queryWrapper = this.getQueryWrapper(category, logModel);
        List<Map<String, Object>> userMaps = logMapper.selectUserGroupByCreatorTime(queryWrapper);
        List<Map<String, Object>> moduleMaps = logMapper.selectModuleGroupByCreatorTime(queryWrapper);
        List<Map<String, Object>> reqMaps = logMapper.selectGroupByCreatorTime(queryWrapper);
        for (DateTime dateTime : dateTimes) {
            String dateStr = DateUtil.getDateString(dateTime, "HH:mm");
            Map<String, Object> createTimes = new HashMap<>();
            createTimes.put("creator_time", dateStr);
            Map<String, Object> userMap =
                    userMaps.stream().filter(t -> t.get("creator_time").equals(dateStr)).findFirst().orElse(null);
            createTimes.put("usercount", CollectionUtil.isEmpty(userMap)? 0 : userMap.get("scount"));
            Map<String, Object> moduleMap =
                    moduleMaps.stream().filter(t -> t.get("creator_time").equals(dateStr)).findFirst().orElse(null);
            createTimes.put("modulecount", CollectionUtil.isEmpty(moduleMap)? 0 : moduleMap.get("scount"));
            Map<String, Object> reqMap =
                    reqMaps.stream().filter(t -> t.get("creator_time").equals(dateStr)).findFirst().orElse(null);
            createTimes.put("reqcount", CollectionUtil.isEmpty(reqMap)? 0 : reqMap.get("scount"));

            resultMap.add(createTimes);
        }
        return resultMap;
    }

    @Override
    public List<Map<String, Object>> selectGroupByModule(String category, PaginationLogModel logModel) {
        QueryWrapper<LogEntity> queryWrapper = this.getQueryWrapper(category, logModel);
        List<Map<String, Object>> resultMap = logMapper.selectGroupByModule(queryWrapper);
        return resultMap;
    }

    @Override
    public List<VisitVO> userVisit(VisitVO visitVO) {
        List<VisitVO> resultMap = new ArrayList<>();
        if (ObjectUtil.isEmpty(visitVO.getCreatorTime())) {
            List<String> creatorTime = new ArrayList<>();
            creatorTime.add(DateUtil.daFormat(DateUtil.getBeginDayOfWeek()));
            creatorTime.add(DateUtil.getmmNow());
            visitVO.setCreatorTime(creatorTime);
        }
        //  构建排序字符串
        String orderBy = "";
        if (ObjectUtil.isNotEmpty(visitVO.getSidx())) {
            orderBy = visitVO.getSidx() + " " + visitVO.getSort();
        }
        PageHelper.startPage((int) visitVO.getCurrentPage(), (int) visitVO.getPageSize(), orderBy);
        resultMap = logMapper.userVisit(visitVO);
        PageInfo<VisitVO> pageInfo = new PageInfo<>(resultMap);
        return visitVO.setData(pageInfo.getList(), pageInfo.getTotal());
    }

    @Override
    public List<VisitVO> functionVisit(VisitVO visitVO) {
        List<VisitVO> resultMap = new ArrayList<>();
        if (ObjectUtil.isEmpty(visitVO.getCreatorTime())) {
            List<String> creatorTime = new ArrayList<>();
            creatorTime.add(DateUtil.daFormat(DateUtil.getBeginDayOfWeek()));
            creatorTime.add(DateUtil.getmmNow());
            visitVO.setCreatorTime(creatorTime);
        }
        //  构建排序字符串
        String orderBy = "";
        if (ObjectUtil.isNotEmpty(visitVO.getSidx())) {
            orderBy = visitVO.getSidx() + " " + visitVO.getSort();
        }
        PageHelper.startPage((int) visitVO.getCurrentPage(), (int) visitVO.getPageSize(), orderBy);
        resultMap = logMapper.functionVisit(visitVO);
        PageInfo<VisitVO> pageInfo = new PageInfo<>(resultMap);
        return visitVO.setData(pageInfo.getList(), pageInfo.getTotal());
    }

    @Override
    public List<VisitVO> functionUserVisit(VisitVO visitVO) {
        List<VisitVO> resultMap = new ArrayList<>();
        if (ObjectUtil.isEmpty(visitVO.getCreatorTime())) {
            List<String> creatorTime = new ArrayList<>();
            creatorTime.add(DateUtil.daFormat(DateUtil.getBeginDayOfWeek()));
            creatorTime.add(DateUtil.getmmNow());
            visitVO.setCreatorTime(creatorTime);
        }
        resultMap = logMapper.functionUserVisit(visitVO);
        return resultMap;
    }

    @Override
    public List<VisitVO> functionUserCountVisit(VisitVO visitVO) {
        List<VisitVO> resultMap = new ArrayList<>();
        if (ObjectUtil.isEmpty(visitVO.getCreatorTime())) {
            List<String> creatorTime = new ArrayList<>();
            creatorTime.add(DateUtil.daFormat(DateUtil.getBeginDayOfWeek()));
            creatorTime.add(DateUtil.getmmNow());
            visitVO.setCreatorTime(creatorTime);
        }
        resultMap = logMapper.functionUserCountVisit(visitVO);
        return resultMap;
    }

    @Override
    public List<VisitVO> functionList(VisitVO visitVO) {
        List<VisitVO> resultMap = new ArrayList<>();
        resultMap = logMapper.functionList();
        return resultMap;
    }
}
