package com.bringspring.system.external.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.extra.emoji.EmojiUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.DeleteMarkEnum;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.base.listener.DIngStream;
import com.bringspring.system.external.bean.WxCpDepartTreeNode;
import com.bringspring.system.external.service.WxCpCorpService;
import com.bringspring.system.external.util.WxCpCorpValidateUtil;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.message.util.SynThirdConsts;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.entity.PositionEntity;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.OrganizeService;
import com.bringspring.system.permission.service.PositionService;
import com.bringspring.system.permission.service.UserService;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.error.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.bean.WxCpDepart;
import me.chanjar.weixin.cp.bean.WxCpUser;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

import static com.bringspring.system.external.constant.WxCpSysConfigConsts.*;
import static com.bringspring.system.message.util.SynThirdConsts.*;

/**
 * 企业相关接口实现类
 *
 * @since 30/1/2024 9:57 PMs
 */
@Slf4j
@Component
public class WxCpCorpServiceImpl implements WxCpCorpService {

    @Autowired
    private WxCpCorpValidateUtil wxCpCorpValidateUtil;
    @Autowired
    private SynThirdInfoService synThirdInfoService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    PositionService positionService;
    @Autowired
    private UserService userService;

    @Autowired
    private DIngStream dIngStream;

    /**
     * 保存部门树形结构信息到本地
     *
     * @param treeList 部门树形结构
     * @param corpId   企业ID
     * @return
     */
    @Override
    public List<SynThirdInfoEntity> saveOrUpdateOrganizeList(List<WxCpDepartTreeNode> treeList, String corpId,
                                                             SynThirdInfoEntity parentThird) {
        List<OrganizeEntity> organizeList = new ArrayList<>();
        List<SynThirdInfoEntity> synThirdInfoList = new ArrayList<>();
        if (ObjectUtil.isNotEmpty(parentThird)) {
            synThirdInfoList.add(parentThird);
        }
        if (CollectionUtil.isNotEmpty(treeList)) {
            wxCpCorpValidateUtil.initData(corpId);
            /**
             * 递归封装本地数据
             */
            this.wxCpDepartToOrganizeList(treeList, corpId, organizeList, synThirdInfoList);

            if (organizeList.size() != 0 && synThirdInfoList.size() != 0) {
                /**
                 * 本地有的编辑，本地没有的新增 saveBatch
                 * 批量新增或编辑
                 */
                organizeService.saveOrUpdateBatch(organizeList);
                synThirdInfoService.saveOrUpdateBatch(synThirdInfoList);
                /**
                 * 机构编码为空的 设置机构编码
                 */
                // dIngStream.setOrganizeCode();
                /**
                 * 设置祖宗组织id
                 */
                dIngStream.setOrganizedTree(corpId);
            }
        }
        return synThirdInfoList;
    }

    /**
     * 企业微信部门树形数据转化为本地持久化数据
     * 递归处理
     *
     * @param treeList         企业微信部门树形数据
     * @param corpId           企业微信
     * @param organizeList     本地部门数据
     * @param synThirdInfoList 第三方同步记录表数据
     */
    @Override
    public void wxCpDepartToOrganizeList(List<WxCpDepartTreeNode> treeList, String corpId,
                                         List<OrganizeEntity> organizeList, List<SynThirdInfoEntity> synThirdInfoList) {
        treeList.stream().forEach(item -> {
            // 企业微信多例配置 系统表base_organize主键id不可以直接返回的部门id
            String thirdObjid = item.getId().toString();
            // 父部门id
            String pid = item.getParentId().toString();
            // 是否顶级父id
            boolean isTopPid = String.valueOf(TOP_PID).equals(pid);
            // 判断父级部门是否已经拉取
            SynThirdInfoEntity pThirdInfo = wxCpCorpValidateUtil.getThirdCorpOrgByThirdId(pid);
            String parentId = "-1";
            if (ObjectUtil.isEmpty(pThirdInfo)) {
                if (CollectionUtil.isNotEmpty(synThirdInfoList) && !isTopPid) {
                    pThirdInfo =
                            synThirdInfoList.stream().filter(p -> p.getThirdObjectId().equals(pid)).collect(Collectors.toList()).get(0);
                    parentId = ObjectUtil.isNotEmpty(pThirdInfo) ? pThirdInfo.getSystemObjectId() : parentId;
                }
            } else {
                parentId = pThirdInfo.getSystemObjectId();
            }
            //组织数据
            OrganizeEntity organizeEntity = new OrganizeEntity();
            organizeEntity.setEnabledMark(1); // 0:禁用；1:正常
            organizeEntity.setDeleteMark(DeleteMarkEnum.NOT_DELETE.getCode());
            // 父部门id。根部门为0 替换为 根部门为-1
            organizeEntity.setParentId(isTopPid ? "-1" : parentId);
            // 部门名称
            organizeEntity.setFullName(item.getName());
            // 排序
            organizeEntity.setSortCode(1000000000 - Long.valueOf(item.getOrder()));
            // 判断是否拉取新增，THIRD_TYPE_QY：第三方工具类型(1:企业微信;2:钉钉)、DATA_TYPE_ORG：数据类型(1:组织(公司与部门);2:用户)
            SynThirdInfoEntity synThirdInfoEntity = wxCpCorpValidateUtil.getThirdCorpOrgByThirdId(thirdObjid);
            if (ObjectUtil.isNotEmpty(synThirdInfoEntity)) {// 第三方数据变更(已拉取过)
                // system_object_id系统对象ID
                String systemObjectId = synThirdInfoEntity.getSystemObjectId();
                organizeEntity.setId(systemObjectId);
            } else {
                // 第三方数据新增(未拉取过)
                synThirdInfoEntity = wxCpCorpValidateUtil.newSynThirdInfoForWxDept(corpId, thirdObjid);
                // 新增的部门id
                organizeEntity.setId(RandomUtil.uuId());
                // 机构分类。根据父部门id 为0或1的机构分类为公司
                organizeEntity.setCategory(isTopPid ? "company" : "department");
            }
            organizeList.add(organizeEntity);

            synThirdInfoEntity.setSystemObjectId(organizeEntity.getId()); // 系统对象ID
            synThirdInfoEntity.setSynState(SynThirdConsts.SYN_STATE_OK); // 同步状态(0:未同步;1:同步成功;
            // 2:同步失败)
            synThirdInfoList.add(synThirdInfoEntity);

            /**
             * 递归
             */
            if (CollectionUtil.isNotEmpty(item.getChildren())) {
                wxCpDepartToOrganizeList(item.getChildren(), corpId, organizeList, synThirdInfoList);
            }
        });
    }

    /**
     * 部门列表树形结构转化
     *
     * @param departList 部门列表
     * @param parentId   父级ID
     * @return
     */
    @Override
    public List<WxCpDepartTreeNode> deptListToTree(List<WxCpDepartTreeNode> departList, Long parentId) {
        List<WxCpDepartTreeNode> list = departList.stream()
                // 过滤父节点
                .filter(parent -> parent.getParentId().equals(parentId))
                // 把父节点children递归赋值成为子节点
                .map(child -> {
                    child.setChildren(deptListToTree(departList, child.getId()));
                    return child;
                }).collect(Collectors.toList());
        return list;
    }

    @Override
    public void deleteOrganize(String corpId, String deptId) {
        // THIRD_TYPE_QY：第三方工具类型(1:企业微信;2:钉钉)、DATA_TYPE_ORG：数据类型(1:组织(公司与部门);2:用户)
        SynThirdInfoEntity synThirdInfoEntity =
                synThirdInfoService.getInfoByThirdObjId(THIRD_TYPE_QY, DATA_TYPE_ORG, deptId, corpId);
        if (ObjectUtil.isNotEmpty(synThirdInfoEntity)) {
            String systemObjectId = synThirdInfoEntity.getSystemObjectId();
            // 删除组织部门信息
            ActionResult<String> actionResult = organizeService.delete(systemObjectId, WXCP_DELETE);
            if (MsgCode.SU003.get().equals(actionResult.getCode())) {
                // 删除同步表记录
                synThirdInfoService.delete(synThirdInfoEntity);
            } else {
                String msg = actionResult.getMsg();
                synThirdInfoEntity.setDescription(msg);
            }
        }
    }


    @Override
    public List<UserEntity> saveOrUpdateUserList(List<WxCpUser> users, WxCpService wxCpService) throws WxErrorException {
        List<UserEntity> userBatch = new ArrayList<>();
        List<SynThirdInfoEntity> synThirdInfoBatch = new ArrayList<>();
        List<PositionEntity> positionBatch = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(users)) {
            String corpId = wxCpService.getWxCpConfigStorage().getCorpId();
            wxCpCorpValidateUtil.initDataByUser(corpId);
            /**
             * 递归封装本地数据
             */
            this.wxCpUserToUserList(users, wxCpService, userBatch, synThirdInfoBatch, positionBatch);
            if (StringUtils.isNotEmpty(userBatch) && StringUtils.isNotEmpty(synThirdInfoBatch)) {

                for (UserEntity entity : userBatch) {
                    // 组织 岗位 角色
                    userService.saveOrUpdateCommon(entity.getId(), entity);
                }
                positionService.saveOrUpdateBatch(positionBatch);
                userService.saveOrUpdateBatch(userBatch);
                synThirdInfoService.saveOrUpdateBatch(synThirdInfoBatch);
            }
        }
        return userBatch;
    }

    /**
     * 企业微信用户数据转化为本地用户表数据
     * 递归处理
     *
     * @param users
     * @param wxCpService
     * @param userBatch
     * @param synThirdInfoBatch
     * @param positionBatch
     * @throws WxErrorException
     */
    private void wxCpUserToUserList(List<WxCpUser> users, WxCpService wxCpService, List<UserEntity> userBatch,
                                    List<SynThirdInfoEntity> synThirdInfoBatch, List<PositionEntity> positionBatch) throws WxErrorException {

        String corpId = wxCpService.getWxCpConfigStorage().getCorpId();

        Map<String, UserEntity> updateUserMap = new HashMap<>();
        Map<String, UserEntity> saveUserMap = new HashMap<>();
        Map<String, SynThirdInfoEntity> updateThirdMap = new HashMap<>();
        Map<String, SynThirdInfoEntity> saveThirdMap = new HashMap<>();
        Map<String, PositionEntity> savePositionMap = new HashMap<>();

        for (WxCpUser wxCpUser : users) {
            String userid = wxCpUser.getUserId();
            // 判断是不是为空,需要挨个判断
            if (StringUtils.isEmpty(userid)) {
                // 结束进入下一次循环
                continue;
            }

            String mobile = wxCpUser.getMobile();
            String email = wxCpUser.getEmail();

            UserEntity userEntity; // 系统用户
            SynThirdInfoEntity synThirdInfoEntity; // 第三方记录表
            /**
             * 登录账户是否新增,默认false
             */
            boolean userIsNew = false;
            /*
            third_object_id第三对象ID
            注意：这里约定，多例企业配置用user，互联企业配置用‘/’拼接corpid/userid.
             */
//            String thirdObjid = corpId + TARGET_CHAR + userid;
            String thirdObjid = userid;
            synThirdInfoEntity = wxCpCorpValidateUtil.getThirdCorpUserByThirdId(thirdObjid);
            if (ObjectUtil.isNotEmpty(synThirdInfoEntity)) {// 第三方数据已同步
                synThirdInfoEntity.setThirdObjectId(thirdObjid);
                // system_object_id系统对象ID
                String systemObjectId = synThirdInfoEntity.getSystemObjectId();

                userEntity = wxCpCorpValidateUtil.getUserEntityByUserid(systemObjectId);
                if (ObjectUtil.isEmpty(userEntity)) {// 系统用户ID不存在
                    // 判断account或phone或email是否存在
                    userEntity = wxCpCorpValidateUtil.getUserEntity(userid, mobile, email);
                    if (ObjectUtil.isEmpty(userEntity)) {
                        /**
                         *  登录账户是新增
                         *  系统用户ID 和 账号 都不存在
                         */
                        userIsNew = true;
                        userEntity = new UserEntity();
                        userEntity.setId(RandomUtil.uuId());
                        userEntity.setAccount(userid);
                    }
                }
            } else {
                // 第三方数据未同步
                synThirdInfoEntity = wxCpCorpValidateUtil.newSynThirdInfoForWxUser(thirdObjid);

                // 判断account或phone或email是否存在
                userEntity = wxCpCorpValidateUtil.getUserEntity(userid, mobile, email);
                if (ObjectUtil.isEmpty(userEntity)) {
                    /**
                     *  登录账户是新增
                     *  系统用户ID 和 账号 都不存在
                     */
                    userIsNew = true;
                    userEntity = new UserEntity();
                    userEntity.setId(RandomUtil.uuId());
                    userEntity.setAccount(userid);
                }
            }

            synThirdInfoEntity.setCropId(corpId);
            synThirdInfoEntity.setSystemObjectId(userEntity.getId());

            // 姓名
            if (StringUtils.isNotEmpty(wxCpUser.getName())) {
                // 去掉除汉字以外的字符
                String name = StringUtils.stringFilter(EmojiUtil.removeAllEmojis(wxCpUser.getName()));
                userEntity.setRealName(name);
            }
            // 头像
            if (StringUtils.isNotEmpty(wxCpUser.getAvatar())) {
                userEntity.setHeadIcon(wxCpUser.getAvatar());
            } else if (StringUtils.isEmpty(userEntity.getHeadIcon())) {
                // 头像默认  微信企业没传
                userEntity.setHeadIcon("/api/file/Image/userAvatar/001.png");
            }
            // 手机
            if (StringUtils.isNotEmpty(mobile)) {
                userEntity.setMobilePhone(mobile);
            }
            // 电话
            if (StringUtils.isNotEmpty(wxCpUser.getTelephone())) {
                userEntity.setTelePhone(wxCpUser.getTelephone());
            }
            // 邮箱
            if (StringUtils.isNotEmpty(email)) {
                userEntity.setEmail(email);
            }
            // 岗位
            boolean positionNotEmpty = false;
            if (StringUtils.isNotEmpty(wxCpUser.getPosition())) {
                positionNotEmpty = true;
            }
            // 部门, 找部门，确保本地拥有部门信息
            Long[] departIds = wxCpUser.getDepartIds();
            List<SynThirdInfoEntity> deptThirdInfoList = departIdsConvertSynThirdInfo(corpId, wxCpService, departIds);
            if (CollectionUtil.isNotEmpty(deptThirdInfoList)) {
                int deptNum = deptThirdInfoList.size();
                String[] organizeId = new String[deptNum];
                String[] positionId = new String[deptNum];
                for (int i = 0; i < deptNum; i++) {
                    String sysDeptId = deptThirdInfoList.get(i).getSystemObjectId();
                    organizeId[i] = sysDeptId;
                    if (positionNotEmpty) {
                        // 岗位不为空
                        positionId[i] = wxCpUser.getPosition() + TARGET_CHAR + sysDeptId;
                    }
                }
                userEntity.setOrganizeId(ArrayUtil.join(organizeId, REGEX_CHAR));
                userEntity.setPositionId(positionNotEmpty ? ArrayUtil.join(positionId, REGEX_CHAR) : null);
            }
            // 工号
            userEntity.setCode(wxCpCorpValidateUtil.getCode(wxCpUser));

            if (userIsNew) {
                /**
                 * 登录账户是新增,设置属性默认值
                 */
                // 角色默认
                userEntity.setRoleId(wxCpCorpValidateUtil.getDefaultRoleId());
                // 性别默认 （1:男；3:女；3:保密）
                userEntity.setGender(3);
                // 有效标志默认
                userEntity.setEnabledMark(1);
                // 删除标志默认
                userEntity.setDeleteMark(DeleteMarkEnum.NOT_DELETE.getCode());
                // 秘钥默认
                userEntity.setSecretkey(RandomUtil.uuId());
                // 是否管理员默认 0:否；1:是
                userEntity.setIsAdministrator(0);
                // 账户密码默认
                userEntity.setPassword(userService.getDefaultPassword(userEntity.getSecretkey()));
            }

            if (userIsNew) {
                /**
                 * 登录账户是新增
                 */
                if (saveUserMap.containsKey(userEntity.getAccount())) {
                    // 待保存的Map中存在, 合并到一个user
                    UserEntity aUser = saveUserMap.get(userEntity.getAccount());
                    // 合并组织
                    wxCpCorpValidateUtil.mergeDistinctOrganize(aUser, userEntity);
                    // 合并岗位
                    wxCpCorpValidateUtil.mergeDistinctPosition(aUser, userEntity);

                    // 设置同步到系统用户表ID
                    synThirdInfoEntity.setSystemObjectId(aUser.getId());
                } else {
                    // 放入待保存的Map
                    saveUserMap.put(userEntity.getAccount(), userEntity);
                }

                // 设置同步状态值
                synThirdInfoEntity.setSynState(SynThirdConsts.SYN_STATE_OK);
                // 放入待保存的Map
                saveThirdMap.put(thirdObjid, synThirdInfoEntity);
            } else {
                /**
                 * 登录账户否新增
                 */
                // 系统用户systemObjectId或登录账户account已存在
                if (updateUserMap.containsKey(userEntity.getAccount())) {
                    // 待更新的Map中存在
                    UserEntity aUser = updateUserMap.get(userEntity.getAccount());
                    // 合并组织
                    wxCpCorpValidateUtil.mergeDistinctOrganize(aUser, userEntity);
                    // 合并岗位
                    wxCpCorpValidateUtil.mergeDistinctPosition(aUser, userEntity);

                    // 设置同步到系统用户表ID
                    synThirdInfoEntity.setSystemObjectId(aUser.getId());
                } else {
                    // 放入待更新的Map
                    updateUserMap.put(userEntity.getAccount(), userEntity);
                }

                // 设置同步状态值
                synThirdInfoEntity.setSynState(SynThirdConsts.SYN_STATE_OK);
                // 放入待保存的Map
                updateThirdMap.put(thirdObjid, synThirdInfoEntity);
            }
        }
        log.debug("===========拉取企业微信部门成员反馈===========");

        // 检查该 HashMap 是否含有元素
        if (!saveUserMap.isEmpty()) {
            Iterator<String> iterator = saveUserMap.keySet().iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                UserEntity userEntity = saveUserMap.get(key);
                // 岗位名称转换positionId
                wxCpCorpValidateUtil.convertPosition(savePositionMap, userEntity);

                // 将新增用户 加入 批量保存或更新数据库列表
                userBatch.add(userEntity);
            }
        }
        if (!updateUserMap.isEmpty()) {
            Iterator<String> iterator = updateUserMap.keySet().iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                UserEntity userEntity = updateUserMap.get(key);
                // 保留用户关联信息
                wxCpCorpValidateUtil.setUserRelation(userEntity);

                // 岗位名称转换positionId
                wxCpCorpValidateUtil.convertPosition(savePositionMap, userEntity);

                // 将更新用户 加入 批量保存或更新数据库列表
                userBatch.add(userEntity);
            }
        }
        if (!saveThirdMap.isEmpty()) {
            Iterator<String> iterator = saveThirdMap.keySet().iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                synThirdInfoBatch.add(saveThirdMap.get(key));
            }
        }
        if (!updateThirdMap.isEmpty()) {
            Iterator<String> iterator = updateThirdMap.keySet().iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                synThirdInfoBatch.add(updateThirdMap.get(key));
            }
        }
        if (!savePositionMap.isEmpty()) {
            Iterator<String> iterator = savePositionMap.keySet().iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                positionBatch.add(savePositionMap.get(key));
            }
        }

    }

    public List<SynThirdInfoEntity> departIdsConvertSynThirdInfo(String corpId, WxCpService wxCpService,
                                                                 Long[] departIds) throws WxErrorException {
        List<SynThirdInfoEntity> deptSynThirdInfoList = new ArrayList<>();
        for (int i = 0; i < departIds.length; i++) {
            String deptId = String.valueOf(departIds[i]);
            // 递归获取当前部门第三方表信息,或其父级部门第三方部门信息
            SynThirdInfoEntity synThirdInfo = this.recursionExistSynThirdInfo(wxCpService, deptId);
            boolean notEmpty = ObjectUtil.isNotEmpty(synThirdInfo);
            if (notEmpty && deptId.equals(synThirdInfo.getThirdObjectId())) {
                // 获取当前部门存在
                deptSynThirdInfoList.add(synThirdInfo);
            } else {
                // 获取当前部门不存在，获取父级(祖宗)部门。则需要拉取部门信息到本地。
//                String corpId = synThirdInfo.getCropId();
                Long pid = notEmpty ? Long.valueOf(synThirdInfo.getThirdObjectId()) : TOP_PID;
                List<WxCpDepart> departList = wxCpService.getDepartmentService().list(pid);
                // 需要拉取部门信息到本地
                List<WxCpDepartTreeNode> treeList = JsonUtil.getJsonToList(departList, WxCpDepartTreeNode.class);
                List<WxCpDepartTreeNode> wxCpDepartTreeNodes = this.deptListToTree(treeList, pid);
                /**
                 * 保存到本地
                 */
                List<SynThirdInfoEntity> synThirdInfos = saveOrUpdateOrganizeList(wxCpDepartTreeNodes, corpId, null);
                if (CollectionUtil.isNotEmpty(synThirdInfos)) {
                    List<SynThirdInfoEntity> collect =
                            synThirdInfos.stream().filter(d -> d.getThirdObjectId().equals(deptId)).collect
                                    (Collectors.toList());
                    deptSynThirdInfoList.addAll(collect);
                    // 保存成功，刷新部门第三方组织同步记录
                    wxCpCorpValidateUtil.initData(corpId);
                }
            }
        }
        return deptSynThirdInfoList;
    }

    /**
     * 递归获取当前部门第三方表信息,或其父级部门第三方部门信息
     * 当前部门不存在时，则向上获取存在的父级(祖宗)部门
     *
     * @param wxCpService
     * @param departId    当前部门ID
     * @return 当前部门或父级(祖宗)部门
     * @throws WxErrorException
     */
    private SynThirdInfoEntity recursionExistSynThirdInfo(WxCpService wxCpService, String departId) throws
            WxErrorException {
        if (StringUtils.isNotEmpty(departId) && !String.valueOf(TOP_PID).equals(departId)) {
            // 企业微信默认顶级节点的父级ID为0，顶级节点都没有说明，所有组织都没同步。
            SynThirdInfoEntity synThirdInfoEntity = wxCpCorpValidateUtil.getThirdCorpOrgByThirdId(departId);
            if (ObjectUtil.isEmpty(synThirdInfoEntity)) {
                List<WxCpDepart> departList = wxCpService.getDepartmentService().list(Long.parseLong(departId));
                if (CollectionUtil.isNotEmpty(departList)) {
                    final WxCpDepart wxCpDepart =
                            departList.stream().filter(v -> departId.equals(String.valueOf(v.getId()))).collect(Collectors.toList()).get(0);
                    return recursionExistSynThirdInfo(wxCpService, String.valueOf(wxCpDepart.getParentId()));
                }
            } else {
                return synThirdInfoEntity;
            }
        }
        return null;
    }

    @Override
    public void deleteUser(String corpId, String userId) {
        // THIRD_TYPE_QY：第三方工具类型(1:企业微信;2:钉钉)、DATA_TYPE_USER：数据类型(1:组织(公司与部门);2:用户)
//        String thirdObjid = corpId + TARGET_CHAR + userId;
        String thirdObjid = userId;
        SynThirdInfoEntity synThirdInfoEntity =
                synThirdInfoService.getInfoByThirdObjId(THIRD_TYPE_QY, DATA_TYPE_USER, thirdObjid, corpId);
        if (ObjectUtil.isNotEmpty(synThirdInfoEntity)) {
            String systemObjectId = synThirdInfoEntity.getSystemObjectId();
            UserEntity entity = userService.getInfo(systemObjectId);
            if (entity != null) {
                // 删除用户信息
                try {
                    userService.delete(entity);
                    // 删除同步表记录
                    synThirdInfoService.delete(synThirdInfoEntity);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }
}
