package com.bringspring.system.msgcenter.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.annotation.NotRecordedLog;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.model.mctaskmsg.McTaskMsgInfoVO;
import com.bringspring.system.msgcenter.model.mctaskmsgcontent.McTaskMsgContentInfoVO;
import com.bringspring.system.msgcenter.model.mctaskmsgreceive.*;
import com.bringspring.system.msgcenter.service.McMsgSendTemplateService;
import com.bringspring.system.msgcenter.service.McTaskMsgContentService;
import com.bringspring.system.msgcenter.service.McTaskMsgReceiveService;
import com.bringspring.system.msgcenter.service.McTaskMsgService;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static com.bringspring.system.msgcenter.constant.DictConsts.MSG_TYPE;

/**
 * 业务场景消息接收记录表
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2024-06-14
 */
@Slf4j
@RestController
@Api(tags = "业务场景消息接收记录表", value = "mcTaskMsgReceive")
@RequestMapping("/api/msgCenter/mcTaskMsgReceive")
public class McTaskMsgReceiveController {

    @Autowired
    private BaseDataUtil baseDataUtil;

    @Autowired
    private McTaskMsgReceiveService mcTaskMsgReceiveService;
    @Autowired
    private McTaskMsgContentService mcTaskMsgContentService;
    @Autowired
    private McTaskMsgService mcTaskMsgService;

    @Autowired
    private McMsgSendTemplateService mcMsgSendTemplateService;

    /**
     * 列表
     *
     * @param mcTaskMsgReceivePagination
     * @return
     */
    @PostMapping("/getList")
    public ActionResult list(@RequestBody McTaskMsgReceivePagination mcTaskMsgReceivePagination) throws IOException {
        List<McTaskMsgReceiveEntity> list = mcTaskMsgReceiveService.getList(mcTaskMsgReceivePagination);
        List<McTaskMsgReceiveListVO> listVO = JsonUtil.getJsonToList(list, McTaskMsgReceiveListVO.class);
        for (McTaskMsgReceiveListVO vo : listVO) {
            //处理id字段转名称，若无需转或者为空可删除
//            vo.setChannelType(baseDataUtil.getDictName(vo.getChannelType(), MSG_TYPE));
            vo.setSysUserId(baseDataUtil.userSelectValues(vo.getSysUserId()));
            vo.setSendUserId(baseDataUtil.userSelectValue(vo.getSendUserId()));
            vo.setAgainUserId(baseDataUtil.userSelectValue(vo.getAgainUserId()));
//            vo.setIsRead(baseDataUtil.switchSelectValue(vo.getIsRead(), "是", "否"));
            vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));

            McTaskMsgEntity taskMsgEntity = mcTaskMsgService.getInfo(vo.getMsgId());
            if (ObjectUtil.isNotNull(taskMsgEntity)) {
                vo.setTaskCode(taskMsgEntity.getTaskCode());
                vo.setSendCode(taskMsgEntity.getSendCode());
                vo.setBusinessKey(taskMsgEntity.getBusinessKey());
            }
        }

        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        PaginationVO page = JsonUtil.getJsonToBean(mcTaskMsgReceivePagination, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }


    /**
     * 站内信列表
     *
     * @param mcTaskMsgReceivePagination
     * @return
     */
    @PostMapping("/getSysMsgList")
    public ActionResult getSysMsgList(@RequestBody McTaskMsgReceivePagination mcTaskMsgReceivePagination) throws IOException {
        List<McTaskMsgReceiveEntity> list = mcTaskMsgReceiveService.getList(mcTaskMsgReceivePagination);
        List<McTaskMsgReceiveSysListVO> listVO = JsonUtil.getJsonToList(list, McTaskMsgReceiveSysListVO.class);
        for (McTaskMsgReceiveSysListVO vo : listVO) {
            vo.setSysUserId(baseDataUtil.userSelectValues(vo.getSysUserId()));
            vo.setSendUserId(baseDataUtil.userSelectValue(vo.getSendUserId()));
            vo.setAgainUserId(baseDataUtil.userSelectValue(vo.getAgainUserId()));
            McTaskMsgContentEntity contentEntity = mcTaskMsgContentService.getInfo(vo.getContentId());
            if (ObjectUtil.isNotNull(contentEntity)) {
                McTaskMsgContentInfoVO content = JsonUtil.getJsonToBean(contentEntity, McTaskMsgContentInfoVO.class);
                vo.setContentType(content.getContentType());
                vo.setTitle(content.getTitle());
                vo.setContent(content.getContent());
                vo.setLinkUrl(content.getLinkUrl());
                vo.setPicUrl(content.getPicUrl());
            }
            McTaskMsgEntity taskMsgEntity = mcTaskMsgService.getInfo(vo.getMsgId());
            if (ObjectUtil.isNotNull(taskMsgEntity)) {
                vo.setTaskCode(taskMsgEntity.getTaskCode());
                vo.setSendCode(taskMsgEntity.getSendCode());
                vo.setCustomKey(taskMsgEntity.getCustomKey());
            }
        }

        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        PaginationVO page = JsonUtil.getJsonToBean(mcTaskMsgReceivePagination, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }

    /**
     * 创建
     *
     * @param mcTaskMsgReceiveCrForm
     * @return
     */
    @PostMapping
    public ActionResult create(@RequestBody @Valid McTaskMsgReceiveCrForm mcTaskMsgReceiveCrForm) throws DataException {

        mcTaskMsgReceiveService.create(mcTaskMsgReceiveCrForm);
        return ActionResult.success("创建成功");
    }


    /**
     * 信息
     *
     * @param id
     * @return
     */
    @GetMapping("/{id}")
    public ActionResult<McTaskMsgReceiveInfoVO> info(@PathVariable("id") String id) {
        McTaskMsgReceiveEntity entity = mcTaskMsgReceiveService.getInfo(id);
        McTaskMsgReceiveInfoVO vo = JsonUtil.getJsonToBean(entity, McTaskMsgReceiveInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 表单信息(详情页)
     *
     * @param id
     * @return
     */
    @GetMapping("/detail/{id}")
    public ActionResult<McTaskMsgReceiveInfoVO> detailInfo(@PathVariable("id") String id) {
        McTaskMsgReceiveEntity entity = mcTaskMsgReceiveService.getInfo(id);
        McTaskMsgReceiveInfoVO vo = JsonUtil.getJsonToBean(entity, McTaskMsgReceiveInfoVO.class);

        McTaskMsgEntity taskMsgEntity = mcTaskMsgService.getInfo(vo.getMsgId());
        if (ObjectUtil.isNotNull(taskMsgEntity)) {
            McTaskMsgInfoVO taskMsgInfo = JsonUtil.getJsonToBean(taskMsgEntity, McTaskMsgInfoVO.class);
            vo.setTaskMsgInfo(taskMsgInfo);
        }
        McMsgSendTemplateModel model = mcMsgSendTemplateService.getModel(vo.getTemplateId());
        if (ObjectUtil.isNotNull(model)) {
            vo.setSendTemplateModel(model);
        }
        McTaskMsgContentEntity contentEntity = mcTaskMsgContentService.getInfo(vo.getContentId());
        if (ObjectUtil.isNotNull(contentEntity)) {
            McTaskMsgContentInfoVO content = JsonUtil.getJsonToBean(contentEntity, McTaskMsgContentInfoVO.class);
            vo.setTaskMsgContentInfo(content);
        }

        vo.setChannelType(baseDataUtil.getDictName(vo.getChannelType(), MSG_TYPE));
        vo.setSysUserId(baseDataUtil.userSelectValues(vo.getSysUserId()));
        vo.setSendUserId(baseDataUtil.userSelectValue(vo.getSendUserId()));
        vo.setAgainUserId(baseDataUtil.userSelectValue(vo.getAgainUserId()));
//        vo.setIsRead(baseDataUtil.switchSelectValue(vo.getIsRead(), "是", "否"));
        vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));

        return ActionResult.success(vo);
    }


    /**
     * 更新
     *
     * @param id
     * @return
     */
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id,
                               @RequestBody @Valid McTaskMsgReceiveUpForm mcTaskMsgReceiveUpForm) throws DataException {
        McTaskMsgReceiveEntity entity = mcTaskMsgReceiveService.getInfo(id);
        if (entity != null) {
            mcTaskMsgReceiveService.update(id, mcTaskMsgReceiveUpForm);
            return ActionResult.success("更新成功");
        } else {
            return ActionResult.fail("更新失败，数据不存在");
        }
    }


    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @DeleteMapping("/{deleteMark}/{ids}")
    public ActionResult delete(@PathVariable("deleteMark") Integer deleteMark, @PathVariable("ids") String[] ids) {
        if (ObjectUtil.isNull(deleteMark)) {
            return ActionResult.fail("参数deleteMark为空");
        }
        if (ObjectUtil.isEmpty(ids)) {
            return ActionResult.fail("参数ids为空");
        }
        mcTaskMsgReceiveService.delete(deleteMark, ids);
        return ActionResult.success("操作成功");
    }


    /**
     * 重发
     *
     * @param ids
     * @return
     */
    @GetMapping("/againSend/{msgId}/{ids}")
    public ActionResult againSend(@PathVariable("msgId") String msgId, @PathVariable("ids") String[] ids) {
        if (StringUtils.isEmpty(msgId)) {
            return ActionResult.fail("参数msgId为空");
        }
        McTaskMsgEntity taskMsgInfo = mcTaskMsgService.getInfo(msgId);
        if (ObjectUtil.isEmpty(taskMsgInfo)) {
            return ActionResult.fail("消息内容表不存在");
        }
        if (ObjectUtil.isEmpty(ids)) {
            return ActionResult.fail("参数ids为空");
        }
        McTaskMsgReceiveListQuery listQuery = new McTaskMsgReceiveListQuery();
        listQuery.setMsgId(taskMsgInfo.getId()); // 消息id
        listQuery.setIds(Arrays.asList(ids));
        List<McTaskMsgReceiveEntity> list = mcTaskMsgReceiveService.getList(listQuery);
        if (CollectionUtil.isEmpty(list)) {
            return ActionResult.fail("接收人记录不存在");
        }

        try {
            mcTaskMsgReceiveService.againSendMessage(taskMsgInfo, list);
            return ActionResult.success("重发成功");
        } catch (Exception e) {
            e.printStackTrace();
            final String message = "发送失败：" + e.getMessage();
            log.error(message);
            return ActionResult.fail(message);
        }
    }


    /**
     * 已读
     *
     * @param isAll 是否全部已读：0否 1是
     * @param ids   1或多条已读ids
     * @return
     */
    @NotRecordedLog
    @GetMapping("/isRead/{isAll}/{ids}")
    public ActionResult isRead(@PathVariable("isAll") String isAll, @PathVariable("ids") String[] ids) {
        if (StringUtils.isNotEmpty(isAll) || ObjectUtil.isNotEmpty(ids)) {
            try {
                mcTaskMsgReceiveService.isRead(isAll, ids);
                return ActionResult.success("已读成功");
            } catch (Exception e) {
                e.printStackTrace();
                final String message = "已读失败：" + e.getMessage();
                log.error(message);
                return ActionResult.fail(message);
            }
        }
        return ActionResult.success("已读成功");
    }

    /**
     * 已读 统计
     *
     * @return
     */
    @NotRecordedLog
    @GetMapping("/isReadCount")
    public ActionResult isReadCount() {
        List<Map<String, Object>> vo = mcTaskMsgReceiveService.isReadCount();
        return ActionResult.success(vo);
    }

    // ============================== 门户统计 ==============================
    /**
     * 门户面板数据集
     *
     * @return
     */
    @ApiOperation("门户面板数据集")
    @PostMapping("/portalPanelData")
    public ActionResult portalPanelData(@RequestBody McTaskMsgReceivePagination pagination) {
        if (StringUtils.isEmpty(pagination.getMenuId())) {
            return ActionResult.fail("菜单id不可为空");
        }

        ActionResult result = new ActionResult();
        Map<String, Object> data = mcTaskMsgReceiveService.selectDataPanel(pagination);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }

    @ApiOperation("按消息渠道分类统计")
    @PostMapping("/countGroupByChannelType")
    public ActionResult countGroupByChannelType(@RequestBody McTaskMsgReceivePagination pagination) {
        if (StringUtils.isEmpty(pagination.getMenuId())) {
            return ActionResult.fail("菜单id不可为空");
        }

        ActionResult result = new ActionResult();
        List<Map<String, Object>> data = mcTaskMsgReceiveService.selectGroupByChannelType(pagination);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }

    @ApiOperation("按发送时间分类统计")
    @PostMapping("/countGroupBySendTime")
    public ActionResult countGroupBySendTime(@RequestBody McTaskMsgReceivePagination pagination) {
        if (StringUtils.isEmpty(pagination.getMenuId())) {
            return ActionResult.fail("菜单id不可为空");
        }

        ActionResult result = new ActionResult();
        List<Map<String, Object>> data = mcTaskMsgReceiveService.selectGroupBySendTime(pagination, 7);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }
}
