package com.bringspring.system.base.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.base.entity.MessageTemplateEntity;
import com.bringspring.system.base.entity.SmsTemplateEntity;
import com.bringspring.system.base.exception.BaseException;
import com.bringspring.system.base.model.messagetemplate.*;
import com.bringspring.common.base.SmsModel;
import com.bringspring.system.base.service.MessageTemplateService;
import com.bringspring.system.base.service.SmsTemplateService;
import com.bringspring.common.sms.util.message.SmsUtil;
import com.bringspring.system.message.model.message.SentMessageForm;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.UserService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Map;

/**
 * 消息模板控制类
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021-12-08
 */
@Tag(name = "消息模板控制类")
@RestController
@RequestMapping("/api/system/MessageTemplate")
public class MessageTemplateController {

    @Autowired
    private MessageTemplateService messageTemplateService;
    @Autowired
    private SmsTemplateService smsTemplateService;
    @Autowired
    private UserService userService;


    /**
     * 消息模板列表
     *
     * @param pagination
     * @return
     */
    @Operation(summary="消息模板列表")
    @GetMapping
    public ActionResult<PageListVO<MessageTemplateListVO>> list(Pagination pagination) {
        List<MessageTemplateEntity> list = messageTemplateService.getList(pagination);
        List<MessageTemplateListVO> listVO = JsonUtil.getJsonToList(list, MessageTemplateListVO.class);
        for (MessageTemplateListVO messageTemplateListVO : listVO) {
            StringBuffer noticeMethod = new StringBuffer();
            if (messageTemplateListVO.getIsDingTalk() == 1) {
                noticeMethod.append("、阿里钉钉");
            }
            if (messageTemplateListVO.getIsEmail() == 1) {
                noticeMethod.append("、电子邮箱");
            }
            if (messageTemplateListVO.getIsSms() == 1) {
                noticeMethod.append("、短信");
            }
            if (messageTemplateListVO.getIsStationLetter() == 1) {
                noticeMethod.append("、站内信");
            }
            if (messageTemplateListVO.getIsWecom() == 1) {
                noticeMethod.append("、企业微信");
            }
            if (noticeMethod.length() > 0) {
                messageTemplateListVO.setNoticeMethod(noticeMethod.toString().replaceFirst("、", ""));
            }
            if ("1".equals(messageTemplateListVO.getCategory())) {
                messageTemplateListVO.setCategory("普通");
            } else if ("2".equals(messageTemplateListVO.getCategory())) {
                messageTemplateListVO.setCategory("重要");
            } else if ("3".equals(messageTemplateListVO.getCategory())) {
                messageTemplateListVO.setCategory("紧急");
            }
            UserEntity entity = userService.getInfo(messageTemplateListVO.getCreatorUserId());
            messageTemplateListVO.setCreatorUserId(entity != null ? entity.getRealName() : null);
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(listVO, paginationVO);
    }

    /**
     * 消息模板下拉框
     *
     * @return
     */
    @Operation(summary="消息模板下拉框")
    @GetMapping("/Selector")
    public ActionResult<PageListVO<MessageTemplateSelector>> selector(Pagination pagination) {
        List<MessageTemplateEntity> list = messageTemplateService.getSelector(pagination);
        for (MessageTemplateEntity entity : list) {
            if ("1".equals(entity.getCategory())) {
                entity.setCategory("普通");
            } else if ("2".equals(entity.getCategory())) {
                entity.setCategory("重要");
            } else if ("3".equals(entity.getCategory())) {
                entity.setCategory("紧急");
            }
        }
        List<MessageTemplateSelector> jsonToList = JsonUtil.getJsonToList(list, MessageTemplateSelector.class);
        ListVO<MessageTemplateSelector> listVO = new ListVO<>();
        listVO.setList(jsonToList);
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(jsonToList, paginationVO);
    }

    /**
     * 获取消息模板
     *
     * @param id
     * @return
     */
    @Operation(summary="获取消息模板")
    @GetMapping("/{id}")
    public ActionResult<MessageTemplateVO> info(@PathVariable("id") String id) {
        MessageTemplateEntity entity = messageTemplateService.getInfo(id);
        MessageTemplateVO vo = JsonUtil.getJsonToBean(entity, MessageTemplateVO.class);
        SmsTemplateEntity info = smsTemplateService.getInfo(vo.getSmsId());
        vo.setSmsTemplateName(info != null ? info.getFullName() : null);
        return ActionResult.success(vo);
    }

    /**
     * 获取消息模板参数
     *
     * @param id
     * @return
     */
    @Operation(summary="获取消息模板参数")
    @GetMapping("/getTemplate/{id}")
    public ActionResult<?> getParameter(@PathVariable("id") String id) {
        MessageTemplateEntity entity = messageTemplateService.getInfo(id);
        if (entity == null) {
            return ActionResult.fail("模板不存在");
        }
        // 获取参数
        String templateJson = entity.getTemplateJson();
        Map<String, Object> map = JsonUtil.stringToMap(templateJson);
        // 如果是短信，获取短信模板参数
        if (entity.getIsSms() == 1) {
            SmsModel smsModel = smsTemplateService.getSmsConfig();
            String smsId = entity.getSmsId();
            SmsTemplateEntity info = smsTemplateService.getInfo(smsId);
            List<String> list = SmsUtil.querySmsTemplateRequest(info.getCompany(), smsModel, info.getEndpoint(), info.getRegion(), info.getTemplateId());
            for (String key : list) {
                map.put(key, null);
            }
        }
        return ActionResult.success(map);
    }

    /**
     * 新建
     *
     * @return
     */
    @Operation(summary="新建")
    @PostMapping
    public ActionResult<String> create(@RequestBody @Valid MessageTemplateCrForm messageTemplateCrForm) {
        MessageTemplateEntity entity = JsonUtil.getJsonToBean(messageTemplateCrForm, MessageTemplateEntity.class);
        if (messageTemplateService.isExistByFullName(entity.getFullName(), entity.getId())) {
            return ActionResult.fail("新建失败，名称不能重复");
        }
        if (messageTemplateService.isExistByEnCode(entity.getEnCode(), entity.getId())) {
            return ActionResult.fail("新建失败，编码不能重复");
        }
        messageTemplateService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 修改
     *
     * @return
     */
    @Operation(summary="修改")
    @PutMapping("/{id}")
    public ActionResult<String> update(@PathVariable("id") String id, @RequestBody @Valid MessageTemplateUpForm messageTemplateUpForm) {
        MessageTemplateEntity entity = JsonUtil.getJsonToBean(messageTemplateUpForm, MessageTemplateEntity.class);
        if (entity != null) {
            if (messageTemplateService.isExistByFullName(entity.getFullName(), id)) {
                return ActionResult.fail("更新失败，名称不能重复");
            }
            if (messageTemplateService.isExistByEnCode(entity.getEnCode(), id)) {
                return ActionResult.fail("更新失败，编码不能重复");
            }
            boolean flag = messageTemplateService.update(id, entity);
            if (!flag) {
                return ActionResult.fail(MsgCode.FA002.get());
            }
            return ActionResult.success(MsgCode.SU004.get());
        }
        return ActionResult.fail(MsgCode.FA002.get());
    }

    /**
     * 删除
     *
     * @return
     */
    @Operation(summary="删除")
    @DeleteMapping("/{id}")
    public ActionResult<String> delete(@PathVariable("id") String id) {
        MessageTemplateEntity entity = messageTemplateService.getInfo(id);
        if (entity == null) {
            return ActionResult.fail(MsgCode.FA003.get());
        }
        messageTemplateService.delete(entity);
        return ActionResult.success(MsgCode.SU003.get());
    }

    /**
     * 修改状态
     *
     * @return
     */
    @Operation(summary="修改状态")
    @PutMapping("/{id}/Actions/State")
    public ActionResult<String> update(@PathVariable("id") String id) {
        MessageTemplateEntity entity = messageTemplateService.getInfo(id);
        if (entity != null) {
            if (entity.getEnabledMark() == 0) {
                entity.setEnabledMark(1);
            } else {
                entity.setEnabledMark(0);
            }
            boolean flag = messageTemplateService.update(id, entity);
            if (!flag) {
                return ActionResult.fail(MsgCode.FA002.get());
            }
            return ActionResult.success(MsgCode.SU004.get());
        }
        return ActionResult.success(MsgCode.FA002.get());
    }

    /**
     * 发送信息
     *
     * @return
     */
    @Operation(summary="测试发送消息")
    @PostMapping("/sendMessage")
    public ActionResult<String> sendMessage(@RequestBody @Valid SentMessageForm messageTemplateCrForm) throws Exception {
        messageTemplateService.testSendMessage(messageTemplateCrForm);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 获取消息模板
     */
    @GetMapping("/getMessageTemplateByEncode/{enCode}")
    public ActionResult getMessageTemplateByEncode(@PathVariable("enCode") String enCode) throws BaseException {
        MessageTemplateEntity messageTemplateByEncode = messageTemplateService.getMessageTemplateByEncode(enCode);
        return ActionResult.success(messageTemplateByEncode);
    }

}
