package com.bringspring.system.base.controller;

import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.DownloadVO;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.model.login.ModuleModel;
import com.bringspring.common.model.login.UserMenuModel;
import com.bringspring.common.util.*;
import com.bringspring.common.util.enums.ModuleTypeEnum;
import com.bringspring.common.util.treeutil.ListToTreeUtil;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.TreeViewModel;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.ModuleEntity;
import com.bringspring.system.base.model.module.*;
import com.bringspring.system.base.service.ModuleService;
import com.bringspring.system.base.util.JsonUtilEx;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static com.bringspring.common.constant.CacheConsts.LOGIN_CACHE;

/**
 * 系统功能
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Tag(name = "极系统菜单")
@RestController
@RequestMapping("/api/system/Menu")
public class ModuleController {

    @Autowired
    private ModuleService moduleService;
    @Autowired
    private CacheUtil cacheUtil;
    @Autowired
    private CacheKeyUtil cacheKeyUtil;
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private ConfigValueUtil configValueUtil;

    /**
     * 获取菜单列表
     *
     * @param paginationMenu 分页参数
     * @return ignore
     */
    @Operation(summary="获取系统列表")
    @GetMapping("/systemList")
    public ActionResult<ListVO<MenuListVO>> systemList(PaginationMenu paginationMenu) {
        List<ModuleEntity> data = moduleService.getList(paginationMenu.getCategory(), paginationMenu.getKeyword(), paginationMenu.getType(), null, null);
        List<UserMenuModel> list = JsonUtil.getJsonToList(data, UserMenuModel.class);
        List<SumTree<UserMenuModel>> menuList = TreeDotUtils.convertListToTreeDot(list);
        List<ModuleModel> menuvo = JsonUtil.getJsonToList(menuList, ModuleModel.class);
        ListVO vo = new ListVO();
        vo.setList(menuvo);
        return ActionResult.success(vo);
    }

    /**
     * 获取菜单列表
     *
     * @param paginationMenu 分页参数
     * @return ignore
     */
    @Operation(summary="获取菜单列表")
    @GetMapping
    public ActionResult<ListVO<MenuListVO>> list(PaginationMenu paginationMenu) {
        List<ModuleEntity> data = moduleService.getList(paginationMenu.getCategory(), paginationMenu.getKeyword(), paginationMenu.getType(), null,paginationMenu.getParentId());

        List<UserMenuModel> list = JsonUtil.getJsonToList(data, UserMenuModel.class);
        List<SumTree<UserMenuModel>> menuList = TreeDotUtils.convertListToTreeDot(list);
        List<MenuListVO> menuvo = JsonUtil.getJsonToList(menuList, MenuListVO.class);
        List<MenuListVO> resultMenuList =new ArrayList<>();
        if (StringUtils.isNotEmpty(paginationMenu.getParentId())) {
            for (MenuListVO menuListVO : menuvo) {
                //如果是父节点，则添加子集到结果中
                if (menuListVO.getId().equals(paginationMenu.getParentId())&&StringUtils.isNotEmpty(menuListVO.getChildren())) {
                    resultMenuList.addAll(menuListVO.getChildren());
                }else if (menuListVO.getParentId().equals(paginationMenu.getParentId())) {
                    resultMenuList.add(menuListVO);

                    //多层子节点直接添加
                }else if (!menuListVO.getParentId().equals(paginationMenu.getParentId())) {
                    resultMenuList.add(menuListVO);
                }
            }
        }
        ListVO vo = new ListVO();
        vo.setList(resultMenuList);
        return ActionResult.success(vo);
    }

    /**
     * 获取菜单列表(下拉框)
     *
     * @param category 分类
     * @param id       主键
     * @return ignore
     */
    @Operation(summary="获取菜单列表(下拉框)")
    @GetMapping("/Selector/{id}")
    public ActionResult<ListVO<MenuSelectVO>> treeView(@RequestParam(value = "category", required = false) String category, @PathVariable("id") String id) {
        List<ModuleEntity> moduleList = moduleService.getList(category, null, "1", null, null);
        List<ModuleEntity> systemModuleList = moduleService.getList(null, null, "0", null, null);
        systemModuleList.addAll(moduleList);
        if (!"0".equals(id)) {
            systemModuleList.remove(moduleService.getInfo(id));
        }
        List<UserMenuModel> list = JsonUtil.getJsonToList(systemModuleList, UserMenuModel.class);
        List<SumTree<UserMenuModel>> menuList = TreeDotUtils.convertListToTreeDotFilter(list);
        List<MenuSelectVO> menuvo = JsonUtil.getJsonToList(menuList, MenuSelectVO.class);
        ListVO vo = new ListVO();
        vo.setList(menuvo);
        return ActionResult.success(vo);
    }

    /**
     * 获取菜单列表(下拉框)
     *
     * @param category 分类
     * @return ignore
     */
    @Operation(summary="获取菜单列表下拉框")
    @GetMapping("/Selector/All")
    public ActionResult<ListVO<MenuSelectVO>> menuSelect(String category) {
        List<ModuleEntity> data = moduleService.getList(category, null, null, "1", null);
        List<UserMenuModel> list = JsonUtil.getJsonToList(data, UserMenuModel.class);
        List<SumTree<UserMenuModel>> menuList = TreeDotUtils.convertListToTreeDot(list);
        List<MenuSelectAllVO> menuvo = JsonUtil.getJsonToList(menuList, MenuSelectAllVO.class);
        ListVO vo = new ListVO();
        vo.setList(menuvo);
        return ActionResult.success(vo);
    }


    /**
     * 系统功能类别树形
     *
     * @return ignore
     */
    @Operation(summary="系统功能类别树形")
    @GetMapping("/TreeView")
    public ActionResult treeView() {
        List<ModuleEntity> moduleList = moduleService.getList(null, null, null, null, "0");
        List<TreeViewModel> treeList = new ArrayList<>();
        TreeViewModel treeViewModel = new TreeViewModel();
        treeViewModel.setId("apply");
        treeViewModel.setText("软件开发平台");
        treeViewModel.setParentId("0");
        treeViewModel.setImg("fa fa-windows apply");
        treeList.add(treeViewModel);
        for (ModuleEntity entity : moduleList) {
            TreeViewModel treeModel = new TreeViewModel();
            treeModel.setId(entity.getId());
            treeModel.setText(entity.getFullName());
            treeModel.setParentId("apply");
            treeModel.setImg("fa fa-tags");
            treeList.add(treeModel);
        }
        return ActionResult.success(ListToTreeUtil.toTreeView(treeList));
    }


    /**
     * 获取菜单信息
     *
     * @param id 主键值
     * @return ignore
     * @throws DataException ignore
     */
    @Operation(summary="获取菜单信息")
    @GetMapping("/{id}")
    public ActionResult info(@PathVariable("id") String id) throws DataException {
        ModuleEntity entity = moduleService.getInfo(id);
        ModuleInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, ModuleInfoVO.class);
        return ActionResult.success(vo);
    }


    /**
     * 新建系统功能
     *
     * @param moduleCrForm 实体对象
     * @return ignore
     */
    @Operation(summary="新建系统功能")
    @PostMapping
    public ActionResult create(@RequestBody @Valid ModuleCrForm moduleCrForm) {
        ModuleEntity entity = JsonUtil.getJsonToBean(moduleCrForm, ModuleEntity.class);
        if (entity.getUrlAddress() != null) {
            entity.setUrlAddress(entity.getUrlAddress().trim());
        }
        if (moduleService.isExistByFullName(entity, moduleCrForm.getCategory())) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        if (moduleService.isExistByEnCode(entity, moduleCrForm.getCategory())) {
            return ActionResult.fail(MsgCode.EXIST002.get());
        }
        moduleService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新系统功能
     *
     * @param id           主键值
     * @param moduleUpForm 实体对象
     * @return ignore
     */
    @Operation(summary="更新系统功能")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid ModuleUpForm moduleUpForm) {
        ModuleEntity entity = JsonUtil.getJsonToBean(moduleUpForm, ModuleEntity.class);
        //判断如果是目录则不能修改类型
        ModuleEntity moduleEntity = moduleService.getInfo(id);
        if (moduleEntity != null && moduleEntity.getType() == 1 && entity.getType() != 1 && moduleService.getList(moduleEntity.getId()).size() > 0) {
            return ActionResult.fail("当前目录存在数据,不能修改类型");
        }
        if (entity.getUrlAddress() != null) {
            entity.setUrlAddress(entity.getUrlAddress().trim());
        }
        if (moduleService.isExistByFullName(entity, moduleUpForm.getCategory())) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        if (moduleService.isExistByEnCode(entity, moduleUpForm.getCategory())) {
            return ActionResult.fail(MsgCode.EXIST002.get());
        }
        boolean flag = moduleService.update(id, entity);
        if (!flag) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除系统功能
     *
     * @param id 主键值
     * @return ignore
     */
    @Operation(summary="删除系统功能")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        ModuleEntity entity = moduleService.getInfo(id);
        if (entity != null) {
            List<ModuleEntity> list = moduleService.getList().stream().filter(t -> t.getParentId().equals(entity.getId())).collect(Collectors.toList());
            if (list.size() > 0) {
                return ActionResult.fail("删除失败，请先删除子菜单。");
            }
            moduleService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    /**
     * 更新菜单状态
     *
     * @param id 主键值
     * @return ignore
     */
    @Operation(summary="更新菜单状态")
    @PutMapping("/{id}/Actions/State")
    public ActionResult upState(@PathVariable("id") String id) {
        ModuleEntity entity = moduleService.getInfo(id);
        if (entity != null) {
            if (entity.getEnabledMark() == null || "1".equals(String.valueOf(entity.getEnabledMark()))) {
                entity.setEnabledMark(0);
            } else {
                entity.setEnabledMark(1);
            }
            moduleService.update(id, entity);
            //清除redis权限
            String cacheKey = cacheKeyUtil.getUserAuthorize() + userProvider.get().getUserId();
            if (cacheUtil.exists(LOGIN_CACHE, cacheKey)) {
                cacheUtil.remove(LOGIN_CACHE, cacheKey);
            }
            return ActionResult.success(MsgCode.SU004.get());
        }
        return ActionResult.fail(MsgCode.FA002.get());
    }

    /**
     * 系统菜单导出功能
     *
     * @param id 接口id
     * @return ignore
     */
    @Operation(summary="导出系统菜单数据")
    @GetMapping("/{id}/Action/Export")
    public ActionResult exportFile(@PathVariable("id") String id) {
        DownloadVO downloadVO = moduleService.exportData(id);
        return ActionResult.success(downloadVO);
    }

    /**
     * 系统菜单导入功能
     *
     * @param multipartFile 文件
     * @return ignore
     */
    @Operation(summary="系统菜单导入功能")
    @PostMapping(value = "/Action/Import", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public ActionResult importFile(@RequestPart("file") MultipartFile multipartFile) throws DataException {
        //判断是否为.json结尾
        if (FileUtil.existsSuffix(multipartFile, ModuleTypeEnum.SYSTEM_MODULE.getTableName())) {
            return ActionResult.fail(MsgCode.IMP002.get());
        }
        //读取文件内容
        String fileContent = FileUtil.getFileContent(multipartFile, configValueUtil.getTemporaryFilePath());
        try {
            //转model后导入
            ModuleExportModel exportModel = JsonUtil.getJsonToBean(fileContent, ModuleExportModel.class);
            ModuleEntity moduleEntity = exportModel.getModuleEntity();
            //判断父级节点是否存在
            if (!"-1".equals(moduleEntity.getParentId()) && moduleService.getInfo(moduleEntity.getParentId()) == null) {
                return ActionResult.fail("导入失败,查询不到上级分类");
            }
            boolean isExists = moduleService.importData(exportModel);
            if (isExists) {
                return ActionResult.fail(MsgCode.IMP003.get());
            }
        } catch (Exception e) {
            throw new DataException(MsgCode.IMP004.get());
        }
        return ActionResult.success(MsgCode.IMP001.get());
    }

}
