package com.bringspring.system.msgcenter.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.msgcenter.constant.Constants.EnableStatus;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.entity.McMsgSendEntity;
import com.bringspring.system.msgcenter.entity.McMsgSendTemplateEntity;
import com.bringspring.system.msgcenter.entity.McMsgTemplateEntity;
import com.bringspring.system.msgcenter.model.mcbusiness.McBusinessInfoVO;
import com.bringspring.system.msgcenter.model.mcmsgsend.*;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.service.*;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.io.IOException;
import java.util.List;

import static com.bringspring.system.msgcenter.constant.DictType.MSG_TYPE;
import static com.bringspring.system.msgcenter.constant.DictType.TASK_CODE;

/**
 * 消息发送配置表
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2024-06-12
 */
@Slf4j
@RestController
@Tag(name = "消息发送配置表")
@RequestMapping("/api/msgCenter/mcMsgSend")
public class McMsgSendController {

    @Autowired
    private BaseDataUtil baseDataUtil;

    @Autowired
    private McMsgSendService mcMsgSendService;

    @Autowired
    private McMsgSendTemplateService mcMsgSendTemplateService;

    @Autowired
    private McBusinessService mcBusinessService;
    @Autowired
    private McMsgTemplateService mcMsgTemplateService;
    @Autowired
    private McMsgAccountService mcMsgAccountService;


    /**
     * 列表
     *
     * @param mcMsgSendPagination
     * @return
     */
    @PostMapping("/getList")
    public ActionResult list(@RequestBody McMsgSendPagination mcMsgSendPagination) throws IOException {
        List<McMsgSendEntity> list = mcMsgSendService.getList(mcMsgSendPagination);
        List<McMsgSendListVO> listVO = JsonUtil.getJsonToList(list, McMsgSendListVO.class);
        //处理id字段转名称，若无需转或者为空可删除
        for (McMsgSendListVO vo : listVO) {
            // 业务系统
            vo.setTaskCode(baseDataUtil.getDictName(vo.getTaskCode(), TASK_CODE.getCode()));
            // 场景配置
            McBusinessInfoVO business = mcBusinessService.getInfoByEnCode(vo.getMessageSource());
            vo.setMessageSource(business != null ? business.getFullName() : null);
//            vo.setEnabledMark(baseDataUtil.switchSelectValue(vo.getEnabledMark(), EnableStatus.ENABLED.getDisplay(), EnableStatus.DISABLED.getDisplay()));
            vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
            vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));
            vo.setDeleteUserId(baseDataUtil.userSelectValue(vo.getDeleteUserId()));
            //子表
            List<McMsgSendTemplateEntity> sendTempList = mcMsgSendService.getMcMsgSendTemplateList(vo.getId(), 1);
            List<McMsgSendTemplateModel> modelList = JsonUtil.getJsonToList(sendTempList, McMsgSendTemplateModel.class);
            for (McMsgSendTemplateModel templateEntity : modelList) {
                templateEntity.setMessageTypeName(baseDataUtil.getDictName(templateEntity.getMessageType(), MSG_TYPE.getCode()));
            }
            vo.setSendTemplateList(modelList);
        }

        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        PaginationVO page = JsonUtil.getJsonToBean(mcMsgSendPagination, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }


    /**
     * 创建
     *
     * @param mcMsgSendCrForm
     * @return
     */
    @PostMapping
    public ActionResult create(@RequestBody @Valid McMsgSendCrForm mcMsgSendCrForm) throws DataException {
        // 业务系统的code 验证
        if (StringUtils.isEmpty(mcMsgSendCrForm.getTaskCode())) {
            return ActionResult.fail("taskCode不能为空");
        } else {
            if (StringUtils.isEmpty(baseDataUtil.getDictName(mcMsgSendCrForm.getTaskCode(), TASK_CODE.getCode()))) {
                return ActionResult.fail("无效的taskCode");
            }
        }
        // 校验编码是否存在重复
        if (!mcMsgSendService.checkEnCodeUnique(mcMsgSendCrForm.getEnCode(), null)) {
            return ActionResult.fail("编码已存在");
        }
        // 校验名称是否存在重复
        if (!mcMsgSendService.checkFullNameUnique(mcMsgSendCrForm.getFullName(), null)) {
            return ActionResult.fail("名称已存在");
        }
        // 校验templateId和accountConfigId重复
        if (!mcMsgSendTemplateService.checkTemplateUnique(mcMsgSendCrForm.getSendTemplateList())) {
            return ActionResult.fail("模板+账号重复");
        }

        mcMsgSendService.create(mcMsgSendCrForm);

        return ActionResult.success("创建成功");
    }


    /**
     * 信息
     *
     * @param id
     * @return
     */
    @GetMapping("/{id}")
    public ActionResult<McMsgSendInfoVO> info(@PathVariable("id") String id) {
        McMsgSendEntity entity = mcMsgSendService.getInfo(id);
        McMsgSendInfoVO vo = JsonUtil.getJsonToBean(entity, McMsgSendInfoVO.class);
        McBusinessInfoVO business = mcBusinessService.getInfoByEnCode(vo.getMessageSource());
        vo.setMessageSourceName(business != null ? business.getFullName() : null);
        //子表
        List<McMsgSendTemplateEntity> mcMsgSendTemplateList = mcMsgSendService.getMcMsgSendTemplateList(id);
        List<McMsgSendTemplateModel> modelList = JsonUtil.getJsonToList(mcMsgSendTemplateList,
                McMsgSendTemplateModel.class);
        for (McMsgSendTemplateModel templateEntity : modelList) {
            McMsgTemplateEntity templateInfo = mcMsgTemplateService.getInfo(templateEntity.getTemplateId());
            templateEntity.setTemplateName(templateInfo != null ? templateInfo.getFullName() : null);
            templateEntity.setTemplateEnCode(templateInfo != null ? templateInfo.getEnCode() : null);
            McMsgAccountEntity accountInfo = mcMsgAccountService.getInfo(templateEntity.getAccountConfigId());
            templateEntity.setAccountName(accountInfo != null ? accountInfo.getFullName() : null);
            templateEntity.setAccountEnCode(accountInfo != null ? accountInfo.getEnCode() : null);
            templateEntity.setMessageTypeName(baseDataUtil.getDictName(templateEntity.getMessageType(), MSG_TYPE.getCode()));
        }
        vo.setSendTemplateList(modelList);
        return ActionResult.success(vo);
    }

    /**
     * 信息
     *
     * @param enCode
     * @return
     */
    @GetMapping("/getInfoByEnCode/{enCode}")
    public ActionResult<McMsgSendInfoVO> infoByEnCode(@PathVariable("enCode") String enCode) {
        McMsgSendEntity entity = mcMsgSendService.getEntityByEnCode(enCode);
        if (ObjectUtil.isNotNull(entity)) {
            McMsgSendInfoVO vo = JsonUtil.getJsonToBean(entity, McMsgSendInfoVO.class);
            McBusinessInfoVO business = mcBusinessService.getInfoByEnCode(vo.getMessageSource());
            vo.setMessageSourceName(business != null ? business.getFullName() : null);
            //子表
            List<McMsgSendTemplateEntity> mcMsgSendTemplateList = mcMsgSendService.getMcMsgSendTemplateList(vo.getId());
            List<McMsgSendTemplateModel> modelList = JsonUtil.getJsonToList(mcMsgSendTemplateList,
                    McMsgSendTemplateModel.class);
            for (McMsgSendTemplateModel templateEntity : modelList) {
                McMsgTemplateEntity templateInfo = mcMsgTemplateService.getInfo(templateEntity.getTemplateId());
                templateEntity.setTemplateName(templateInfo != null ? templateInfo.getFullName() : null);
                templateEntity.setTemplateEnCode(templateInfo != null ? templateInfo.getEnCode() : null);
                McMsgAccountEntity accountInfo = mcMsgAccountService.getInfo(templateEntity.getAccountConfigId());
                templateEntity.setAccountName(accountInfo != null ? accountInfo.getFullName() : null);
                templateEntity.setAccountEnCode(accountInfo != null ? accountInfo.getEnCode() : null);
                templateEntity.setMessageTypeName(baseDataUtil.getDictName(templateEntity.getMessageType(), MSG_TYPE.getCode()));
            }
            vo.setSendTemplateList(modelList);
            return ActionResult.success(vo);
        } else {
            return ActionResult.fail("发送配置[enCode]不存在");
        }
    }

    /**
     * 表单信息(详情页)
     *
     * @param id
     * @return
     */
    @GetMapping("/detail/{id}")
    public ActionResult<McMsgSendInfoVO> detailInfo(@PathVariable("id") String id) {
        McMsgSendEntity entity = mcMsgSendService.getInfo(id);
        McMsgSendInfoVO vo = JsonUtil.getJsonToBean(entity, McMsgSendInfoVO.class);

        //子表数据转换
        List<McMsgSendTemplateEntity> mcMsgSendTemplateList = mcMsgSendService.getMcMsgSendTemplateList(id);
        List<McMsgSendTemplateModel> mcMsgSendTemplateModelList = JsonUtil.getJsonToList(mcMsgSendTemplateList,
                McMsgSendTemplateModel.class);
        for (McMsgSendTemplateModel mcMsgSendTemplateModel : mcMsgSendTemplateModelList) {
        }
        vo.setSendTemplateList(mcMsgSendTemplateModelList);

        //附表数据转换

        // 业务系统
        vo.setTaskCode(baseDataUtil.getDictName(vo.getTaskCode(), TASK_CODE.getCode()));
        // 场景配置
        McBusinessInfoVO business = mcBusinessService.getInfoByEnCode(vo.getMessageSource());
        vo.setMessageSource(business != null ? business.getFullName() : null);
        vo.setEnabledMark(baseDataUtil.switchSelectValue(vo.getEnabledMark(), EnableStatus.ENABLED.getDisplay(), EnableStatus.DISABLED.getDisplay()));
        vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
        vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));
        vo.setDeleteUserId(baseDataUtil.userSelectValue(vo.getDeleteUserId()));

        return ActionResult.success(vo);
    }


    /**
     * 更新
     *
     * @param id
     * @return
     */
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid McMsgSendUpForm mcMsgSendUpForm) throws DataException {
        // 业务系统的code 验证
        if (StringUtils.isEmpty(mcMsgSendUpForm.getTaskCode())) {
            return ActionResult.fail("taskCode不能为空");
        } else {
            if (StringUtils.isEmpty(baseDataUtil.getDictName(mcMsgSendUpForm.getTaskCode(), TASK_CODE.getCode()))) {
                return ActionResult.fail("无效的taskCode");
            }
        }
        // 校验编码是否存在重复
        if (!mcMsgSendService.checkEnCodeUnique(mcMsgSendUpForm.getEnCode(), id)) {
            return ActionResult.fail("编码已存在");
        }
        // 校验名称是否存在重复
        if (!mcMsgSendService.checkFullNameUnique(mcMsgSendUpForm.getFullName(), id)) {
            return ActionResult.fail("名称已存在");
        }
        // 校验templateId和accountConfigId重复
        if (!mcMsgSendTemplateService.checkTemplateUnique(mcMsgSendUpForm.getSendTemplateList())) {
            return ActionResult.fail("模板+账号重复");
        }

        McMsgSendEntity entity = mcMsgSendService.getInfo(id);
        if (entity != null) {
            mcMsgSendService.update(id, mcMsgSendUpForm);
            return ActionResult.success("更新成功");
        } else {
            return ActionResult.fail("更新失败，数据不存在");
        }
    }


    /**
     * 删除
     *
     * @param id
     * @return
     */
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        McMsgSendEntity entity = mcMsgSendService.getInfo(id);
        if (entity != null) {
            mcMsgSendService.delete(entity);

        }
        return ActionResult.success("删除成功");
    }


    /**
     * 发送配置分页选择框
     *
     * @return
     */
    @Operation(summary="发送配置分页选择框")
    @GetMapping("/Selector")
    public ActionResult<PageListVO<McMsgSendSelector>> selector(McMsgSendPagination pagination) {
        List<McMsgSendEntity> list = mcMsgSendService.getList(pagination);
        List<McMsgSendSelector> listSelector = JsonUtil.getJsonToList(list, McMsgSendSelector.class);
        if (CollectionUtil.isNotEmpty(listSelector)) {
            for (McMsgSendSelector selector : listSelector) {
                McBusinessInfoVO mcBusinessInfoVO = mcBusinessService.getInfoByEnCode(selector.getMessageSource());
                selector.setBusinessInfoVO(mcBusinessInfoVO);
            }
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(listSelector, paginationVO);
    }

}
