package com.bringspring.system.msgcenter.service.chain;

import cn.hutool.core.collection.CollUtil;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.EnabledMarkEnum;
import com.bringspring.system.msgcenter.enums.MsgCenterCode;
import com.bringspring.system.msgcenter.service.context.FilterResult;
import com.bringspring.system.msgcenter.service.context.SendMessageContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 业务消息接收人是否有效过滤器
 * 第三方信息校验过滤器
 *
 * 此处策略针对第三方信息表：是在系统用户依赖第三方信息表的情况下，判断系统用户是否绑定有效的第三方应用。
 * 注意：此处是对消息中心接收人记录过滤后的扩展，即接收人拥有系统用户且给用户第三方应用发送消息的情况。
 * 适用渠道：企业微信、钉钉、微信公众号、微信小程序等适用此处策略。
 *
 */
@Component
public class ThirdInfoFilter implements FilterProcessor {
    @Autowired
    private SynThirdInfoService synThirdInfoService;

    @Override
    public FilterResult process(SendMessageContext context) {
        /**
         * 获取有效的第三方信息列表，存储到上下文中
         * 注意：责任链上一个节点已经将无效的记录过滤掉了，所以此处获取context.getValidList()
         */
        List<String> thirdIds =
                context.getValidList().stream().map(McTaskMsgReceiveEntity::getReceiveUserId).collect(Collectors.toList());
        List<SynThirdInfoEntity> thirdInfoList = synThirdInfoService.listByIds(thirdIds);
        Map<String, SynThirdInfoEntity> infoMap = thirdInfoList.stream()
                .collect(Collectors.toMap(SynThirdInfoEntity::getId, Function.identity()));
        context.setThirdInfoMap(infoMap);

        Map<Boolean, List<McTaskMsgReceiveEntity>> partitioned =
                context.getValidList().stream().collect(Collectors.partitioningBy(r -> infoMap.containsKey(r.getReceiveUserId())));


        List<McTaskMsgReceiveEntity> invalidList = partitioned.get(false);
        if (CollUtil.isNotEmpty(invalidList)) {
            // 分离有效和无效记录，并标注原因
            invalidList.stream().forEach(e -> {
                e.setEnabledMark(EnabledMarkEnum.FAIL.getCode());
                e.setDescription(MsgCenterCode.THIRD_PARTY_INFO_MISSING.getDesc() + "：" + e.getReceiveUserId());
            });
        }

        return new FilterResult(partitioned.get(true), invalidList);
    }
}
