package com.bringspring.system.msgcenter.service.strategy;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.msgcenter.constant.Constants.MessageConfig;
import com.bringspring.system.msgcenter.constant.Constants.ReadStatus;
import com.bringspring.system.msgcenter.entity.*;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.model.mcbusiness.McBusinessInfoVO;
import com.bringspring.system.msgcenter.model.mctaskmsgcontent.ContentData;
import com.bringspring.system.msgcenter.service.McBusinessService;
import com.bringspring.system.msgcenter.service.McMsgAccountService;
import com.bringspring.system.msgcenter.service.McMsgTemplateService;
import com.bringspring.system.msgcenter.service.context.TaskMsgReceiveMetadata;
import com.bringspring.system.msgcenter.util.SensitiveWordUtil;
import com.bringspring.system.msgcenter.util.UrlUtils;
import com.bringspring.system.permission.entity.UserEntity;
import org.apache.commons.text.StringSubstitutor;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static com.bringspring.system.msgcenter.enums.ChannelTypeEnum.getByCode;

/**
 * 定义抽象策略基类
 * 将公共方法（如getTaskMsgContent、packThirdReceive）提取到抽象类中，供具体策略复用
 */
public abstract class AbstractMessageChannelStrategy implements MessageChannelStrategy {
    @Autowired
    protected McBusinessService mcBusinessService;
    @Autowired
    private McMsgTemplateService mcMsgTemplateService;
    @Autowired
    protected McMsgAccountService mcMsgAccountService;
    @Autowired
    protected SynThirdInfoService synThirdInfoService;

    private final String PIC_URL_KEY = "picUrl";
    private final String PENDING_STATUS_DESC = "创建，设置下发状态为‘待下发’";
    /**
     * 获取消息发送渠道类型
     *
     * @return
     */
    public abstract ChannelTypeEnum getChannelType();

    /**
     * 公共方法：生成消息内容实体
     * 模板占位符替换
     *
     * @param templateId
     * @param channelType
     * @param taskMsg
     * @param msgTemplate
     * @param account
     * @return
     */
    protected McTaskMsgContentEntity getTaskMsgContent(
            String templateId, String channelType, McTaskMsgEntity taskMsg,
            McMsgTemplateEntity msgTemplate, McMsgAccountEntity account
    ) {
        String taskMsgId = taskMsg.getId();
        McBusinessInfoVO businessInfoVO = mcBusinessService.getInfoByEnCode(msgTemplate.getMessageSource());
        Map<String, Object> businessKeyMap = JsonUtil.stringToMap(taskMsg.getBusinessKey());

        // 处理模板占位符替换和链接生成
        ContentData contentData = processTemplateData(msgTemplate, businessKeyMap, account, taskMsgId, channelType);

        // 构建消息内容实体
        return buildContentEntity(contentData, taskMsg, channelType, templateId, account, businessInfoVO, msgTemplate);
    }

    /**
     * 处理模板数据的替换和链接生成
     */
    private ContentData processTemplateData(
            McMsgTemplateEntity msgTemplate, Map<String, Object> businessKeyMap,
            McMsgAccountEntity account, String taskMsgId, String channelType
    ) {
        ContentData data = new ContentData();
        if (MapUtil.isEmpty(businessKeyMap)) return data;

        // 替换标题和内容
        data.setTitle(replacePlaceholder(msgTemplate.getTitle(), businessKeyMap, "title"));
        data.setContent(replacePlaceholder(msgTemplate.getContent(), businessKeyMap, "content"));

        // 处理摘要、链接和图片
        data.setDescription(Optional.ofNullable(businessKeyMap.get("desc"))
                .map(Object::toString)
                .orElseGet(() -> truncateContent(data.getContent())));
        data.setLinkUrl(packLinkUrl(channelType, businessKeyMap, account, taskMsgId));
        data.setPicUrl(Optional.ofNullable(businessKeyMap.get(PIC_URL_KEY)).map(Object::toString).orElse(null));

        return data;
    }

    /**
     * 替换占位符的通用方法
     */
    private String replacePlaceholder(String template, Map<String, Object> params, String fallbackKey) {
        if (StringUtils.isNotEmpty(template)) {
            return new StringSubstitutor(params, "{", "}").replace(template);
        } else {
            return String.valueOf(params.get(fallbackKey));
        }
    }

    /**
     * 截取内容作为摘要
     */
    private String truncateContent(String content) {
        return StringUtils.isEmpty(content) ? null : StringUtils.substring(content, 0, 200);
    }

    /**
     * 构建消息内容实体
     */
    private McTaskMsgContentEntity buildContentEntity(
            ContentData contentData, McTaskMsgEntity taskMsg, String channelType,
            String templateId, McMsgAccountEntity account, McBusinessInfoVO businessInfoVO,
            McMsgTemplateEntity msgTemplate
    ) {
        McTaskMsgContentEntity entity = new McTaskMsgContentEntity();
        entity.setId(RandomUtil.uuId());
        entity.setMsgId(taskMsg.getId());
        entity.setChannelType(channelType);
        entity.setTemplateId(templateId);
        entity.setMsgTemplateId(msgTemplate.getId());
        entity.setMsgAccountId(ObjectUtil.isEmpty(account) ? null : account.getId());
        entity.setContentType(businessInfoVO.getCategory());

        // 设置敏感词过滤后的字段
        setFilteredField(contentData.getTitle(), 50, entity::setTitle);
        setFilteredField(contentData.getContent(), null, entity::setContent);
        setFilteredField(contentData.getDescription(), null, entity::setDescription);
        Optional.ofNullable(contentData.getLinkUrl()).ifPresent(entity::setLinkUrl);
        Optional.ofNullable(contentData.getPicUrl()).ifPresent(entity::setPicUrl);

        return entity;
    }

    /**
     * 通用方法：敏感词过滤并设置字段
     */
    private void setFilteredField(String value, Integer maxLength, Consumer<String> setter) {
        if (StringUtils.isEmpty(value)) return;

        String filtered = SensitiveWordUtil.sensitiveFilter(value, true);
        if (maxLength != null && filtered.length() > maxLength) {
            filtered = StringUtils.substring(filtered, 0, maxLength);
        }
        setter.accept(filtered);
    }

    /**
     * 公共方法：生成消息内容实体
     * 模板变量替换
     *
     * @param templateId
     * @param channelType
     * @param taskMsg
     * @param msgTemplate
     * @param account
     * @return
     */
    protected McTaskMsgContentEntity getTaskMsgContentKey(String templateId, String channelType,
                                                          McTaskMsgEntity taskMsg, McMsgTemplateEntity msgTemplate,
                                                          McMsgAccountEntity account) {
        String taskMsgId = taskMsg.getId();
        // 消息场景
        McBusinessInfoVO businessInfoVO = mcBusinessService.getInfoByEnCode(msgTemplate.getMessageSource());
        // 业务消息关键词
        String businessKey = taskMsg.getBusinessKey();
        Map<String, Object> businessKeyMap = JsonUtil.stringToMap(businessKey);

        McTaskMsgContentEntity taskMsgContentEntity = new McTaskMsgContentEntity();
        // 模板变量
        List<McMsgTemplateFieldEntity> fieldList = mcMsgTemplateService.getMcMsgTemplateFieldList(msgTemplate.getId());
        if (CollUtil.isNotEmpty(fieldList) && StringUtils.isNotEmpty(businessKeyMap)) {
            HashMap<Object, Object> fieldMap = new HashMap<>();
            for (McMsgTemplateFieldEntity field : fieldList) {
                fieldMap.put(field.getThirdField(), businessKeyMap.get(field.getKeywordCode()));
                if (field.getIsTitle() == 1) {
                    taskMsgContentEntity.setTitle(businessKeyMap.get(field.getKeywordCode()).toString());
                }
            }
            // 内容
            String content = JSON.toJSONString(fieldMap);
            if (StringUtils.isNotEmpty(content)) {
                content = SensitiveWordUtil.sensitiveFilter(content, true);
                taskMsgContentEntity.setContent(content);
            }

            // 消息链接
            taskMsgContentEntity.setLinkUrl(this.packLinkUrl(channelType, businessKeyMap, account, taskMsgId));
            // 图片链接
            if (businessKeyMap.containsKey(PIC_URL_KEY) && ObjectUtil.isNotEmpty(businessKeyMap.get(PIC_URL_KEY))) {
                taskMsgContentEntity.setLinkUrl(String.valueOf(businessKeyMap.get(PIC_URL_KEY)));
            }
        }
        taskMsgContentEntity.setId(RandomUtil.uuId());
        taskMsgContentEntity.setMsgId(taskMsgId);
        taskMsgContentEntity.setChannelType(channelType);
        taskMsgContentEntity.setTemplateId(templateId);
        taskMsgContentEntity.setMsgTemplateId(msgTemplate.getId());
        taskMsgContentEntity.setMsgAccountId(ObjectUtil.isEmpty(account) ? null : account.getId());
        taskMsgContentEntity.setContentType(businessInfoVO.getCategory());
        return taskMsgContentEntity;
    }

    // 公共方法：生成第三方接收人
    protected List<McTaskMsgReceiveEntity> packThirdReceive(
            List<UserEntity> sysList,
            List<SynThirdInfoEntity> thirdList,
            TaskMsgReceiveMetadata metadata,
            List<McTaskMsgReceiveEntity> receiveList) {

        return CollUtil.isEmpty(receiveList)
                ? createNewReceives(sysList, thirdList, metadata)
                : updateExistingReceives(thirdList, metadata, receiveList);
    }

    /**
     * 创建新的接收人列表（当 receiveList 为空时）
     */
    private List<McTaskMsgReceiveEntity> createNewReceives(
            List<UserEntity> sysList,
            List<SynThirdInfoEntity> thirdList,
            TaskMsgReceiveMetadata metadata
    ) {
        List<McTaskMsgReceiveEntity> receiveEntities = new ArrayList<>();
        for (UserEntity user : sysList) {
            McTaskMsgReceiveEntity baseReceive = createBaseReceive(metadata, user);
            processThirdInfoForUser(thirdList, user, receiveEntities, baseReceive);
        }
        return receiveEntities;
    }

    /**
     * 创建基础接收人实体（提取公共属性设置）
     */
    private McTaskMsgReceiveEntity createBaseReceive(
            TaskMsgReceiveMetadata metadata,
            UserEntity user
    ) {
        McTaskMsgReceiveEntity entity = new McTaskMsgReceiveEntity();
        entity.setMsgId(metadata.getTaskMsgId());
        entity.setTemplateId(metadata.getTemplateId());
        entity.setMsgTemplateId(metadata.getMsgTemplateId());
        entity.setMsgAccountId(metadata.getMsgAccountId());
        entity.setContentId(metadata.getContentId());
        entity.setChannelType(metadata.getChannelType());
        entity.setSysUserId(user.getId());
        entity.setIsRead(ReadStatus.UNREAD.getCode());
        return entity;
    }

    /**
     * 处理用户关联的第三方信息（减少嵌套）
     */
    private void processThirdInfoForUser(
            List<SynThirdInfoEntity> thirdList, UserEntity user,
            List<McTaskMsgReceiveEntity> receiveEntities, McTaskMsgReceiveEntity baseReceive
    ) {
        List<SynThirdInfoEntity> userThirdInfos = thirdList.stream()
                .filter(s -> user.getId().equals(s.getSystemObjectId()))
                .collect(Collectors.toList());

        if (CollUtil.isEmpty(userThirdInfos)) {
            baseReceive.setEnabledMark(0);
            baseReceive.setDescription(PENDING_STATUS_DESC);
            baseReceive.setId(RandomUtil.uuId());
            receiveEntities.add(baseReceive);
        } else {
            userThirdInfos.forEach(s -> {
                McTaskMsgReceiveEntity cloned = cloneAndSetThirdInfo(baseReceive, s);
                receiveEntities.add(cloned);
            });
        }
    }

    /**
     * 克隆实体并设置第三方信息
     */
    private McTaskMsgReceiveEntity cloneAndSetThirdInfo(McTaskMsgReceiveEntity source, SynThirdInfoEntity thirdInfo) {
        McTaskMsgReceiveEntity cloned = new McTaskMsgReceiveEntity();
        BeanUtils.copyProperties(source, cloned);
        cloned.setReceiveUserId(thirdInfo.getId());
        cloned.setEnabledMark(0);
        cloned.setDescription(PENDING_STATUS_DESC);
        cloned.setId(RandomUtil.uuId());
        return cloned;
    }

    /**
     * 更新现有接收人列表（当 receiveList 不为空时）
     */
    private List<McTaskMsgReceiveEntity> updateExistingReceives(
            List<SynThirdInfoEntity> thirdList,
            TaskMsgReceiveMetadata metadata,
            List<McTaskMsgReceiveEntity> receiveList
    ) {
        for (SynThirdInfoEntity thirdInfo : thirdList) {
            receiveList.stream()
                    .filter(r -> thirdInfo.getSystemObjectId().equals(r.getSysUserId()))
                    .findFirst()
                    .ifPresent(existing -> updateOrCloneReceive(existing, thirdInfo, metadata, receiveList));
        }
        return receiveList;
    }

    /**
     * 更新或克隆接收人实体
     */
    private void updateOrCloneReceive(
            McTaskMsgReceiveEntity existing, SynThirdInfoEntity thirdInfo,
            TaskMsgReceiveMetadata metadata,
            List<McTaskMsgReceiveEntity> receiveList
    ) {
        if (StringUtils.isEmpty(existing.getReceiveUserId())) {
            existing.setReceiveUserId(thirdInfo.getId());
            existing.setTemplateId(metadata.getTemplateId());
            existing.setMsgTemplateId(metadata.getMsgTemplateId());
            existing.setMsgAccountId(metadata.getMsgAccountId());
            existing.setContentId(metadata.getContentId());
        } else {
            McTaskMsgReceiveEntity cloned = cloneAndUpdateReceive(existing, thirdInfo, metadata);
            receiveList.add(cloned);
        }
    }

    /**
     * 克隆并更新接收人实体属性
     */
    private McTaskMsgReceiveEntity cloneAndUpdateReceive(
            McTaskMsgReceiveEntity source, SynThirdInfoEntity thirdInfo,
            TaskMsgReceiveMetadata metadata
    ) {
        McTaskMsgReceiveEntity cloned = new McTaskMsgReceiveEntity();
        BeanUtils.copyProperties(source, cloned);
        cloned.setId(RandomUtil.uuId());
        cloned.setReceiveUserId(thirdInfo.getId());
        cloned.setTemplateId(metadata.getTemplateId());
        cloned.setMsgTemplateId(metadata.getMsgTemplateId());
        cloned.setMsgAccountId(metadata.getMsgAccountId());
        cloned.setContentId(metadata.getContentId());
        return cloned;
    }

    // 公共方法：生成第三方接收人（非系统用户，第三方唯一标识直接发送）
    protected List<McTaskMsgReceiveEntity> packNonSystemReceive(Set<String> userSet, String taskMsgId,
                                                                String templateId, String channelType,
                                                                String msgTemplateId, String msgAccountId,
                                                                String contentId
    ) {
        List<McTaskMsgReceiveEntity> receiveEntities = new ArrayList<>();
        for (String openid : userSet) {
            McTaskMsgReceiveEntity thirdReceive = new McTaskMsgReceiveEntity();
            thirdReceive.setId(RandomUtil.uuId());
            thirdReceive.setMsgId(taskMsgId);
            // 发送配置模板id
            thirdReceive.setTemplateId(templateId);
            // 消息模板Id
            thirdReceive.setMsgTemplateId(msgTemplateId);
            // 授权应用Id
            thirdReceive.setMsgAccountId(msgAccountId);
            // 消息内容Id
            thirdReceive.setContentId(contentId);
            // 消息接收人渠道
            thirdReceive.setChannelType(channelType);
            // 下发渠道接收人：base_user主键id。非系统用户，第三方唯一标识直接发送，不用关联系统用户
            thirdReceive.setSysUserId(null);
            thirdReceive.setReceiveUserId(openid); // 第三方用户唯一标识
            thirdReceive.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
            thirdReceive.setIsRead(ReadStatus.UNREAD.getCode()); // 是否阅读，0否、1是
            thirdReceive.setDescription(PENDING_STATUS_DESC);
            receiveEntities.add(thirdReceive);
        }
        return receiveEntities;
    }

    // 公共方法：生成消息链接
    protected String packLinkUrl(String channelType, Map<String, Object> businessKeyMap,
                                 McMsgAccountEntity account, String taskMsgId) {
        ChannelTypeEnum typeEnum = getByCode(channelType);
        String urlKey = getUrlKeyByChannel(typeEnum);      // 动态获取 URL 键名
        Object linkUrl = (urlKey != null) ? businessKeyMap.get(urlKey) : null;

        // 后备逻辑：如果未找到 URL，尝试从账户对象获取
        if (ObjectUtil.isEmpty(linkUrl) && ObjectUtil.isNotEmpty(account)) {
            linkUrl = account.getAddress();
        }

        // 构建最终带参数的 URL
        return buildUrlWithParams(linkUrl, taskMsgId);
    }

    /**
     * 根据消息渠道类型映射对应的 URL 键名（如 linkUrl1, linkUrl2...）
     */
    private String getUrlKeyByChannel(ChannelTypeEnum channel) {
        if (channel == null) return null;
        Map<ChannelTypeEnum, String> keyMapping = new EnumMap<>(ChannelTypeEnum.class);
        keyMapping.put(ChannelTypeEnum.SYS_MSG, "linkUrl");
        keyMapping.put(ChannelTypeEnum.MAIL_MSG, "linkUrl2");
        keyMapping.put(ChannelTypeEnum.SMS_MSG, "linkUrl3");
        keyMapping.put(ChannelTypeEnum.DING_MSG, "linkUrl4");
        keyMapping.put(ChannelTypeEnum.QY_MSG, "linkUrl5");
        keyMapping.put(ChannelTypeEnum.WX_MINIAPP, "linkUrl6");
        keyMapping.put(ChannelTypeEnum.WX_MP, "linkUrl7");
        keyMapping.put(ChannelTypeEnum.WEBHOOK, "linkUrl8");
        return keyMapping.get(channel);
    }

    /**
     * 拼接参数并生成最终 URL
     */
    private String buildUrlWithParams(Object linkUrl, String taskMsgId) {
        if (ObjectUtil.isEmpty(linkUrl)) return null;
        Map<String, Object> params = new HashMap<>();
        params.put(MessageConfig.MSG_ID, taskMsgId);
        return UrlUtils.addParameter(String.valueOf(linkUrl), params);
    }
}
