package com.bringspring.system.base.exception;

import com.alibaba.fastjson.JSON;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.ActionResultCode;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.*;
import com.bringspring.system.base.entity.LogEntity;
import com.bringspring.system.base.enums.LogSortEnum;
import com.bringspring.system.base.service.LogService;
import com.bringspring.system.permission.exception.PermissionException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

import java.nio.file.AccessDeniedException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/16 10:10
 */
@Slf4j
@Controller
@ControllerAdvice
public class ResultException {

    @Autowired
    private UserProvider userProvider;
    @Autowired
    private LogService logService;
    @Autowired
    private ConfigValueUtil configValueUtil;

    @ResponseBody
    @ExceptionHandler(value = LoginException.class)
    public ActionResult loginException(LoginException e) {
        log.error("系统异常:" + e.getMessage(), e);
        ActionResult result = ActionResult.fail(ActionResultCode.Fail.getCode(), e.getMessage());
        return result;
    }

    @ResponseBody
    @ExceptionHandler(value = AccessDeniedException.class)
    public ActionResult accessException(AccessDeniedException e) {
        log.error("系统异常:" + e.getMessage(), e);
        ActionResult result = ActionResult.fail(ActionResultCode.ValidateError.getCode(), "无权访问接口");
        return result;
    }


    /**
     * 自定义异常内容返回
     *
     * @param e
     * @return
     */
    @ResponseBody
    @ExceptionHandler(value = DataException.class)
    public ActionResult dataException(DataException e) {
        log.error("系统异常:" + e.getMessage(), e);
        ActionResult result = ActionResult.fail(ActionResultCode.Fail.getCode(), e.getMessage());
        return result;
    }


    @ResponseBody
    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    public ActionResult methodArgumentNotValidException(MethodArgumentNotValidException e) {
        log.error("系统异常:" + e.getMessage(), e);
        Map<String, String> map = new HashMap<>(16);
        List<ObjectError> allErrors = e.getBindingResult().getAllErrors();
        for (int i = 0; i < allErrors.size(); i++) {
            String s = allErrors.get(i).getCodes()[0];
            //用分割的方法得到字段名
            String[] parts = s.split("\\.");
            String part1 = parts[parts.length - 1];
            map.put(part1, allErrors.get(i).getDefaultMessage());
        }
        String json = JSON.toJSONString(map);
        ActionResult result = ActionResult.fail(ActionResultCode.ValidateError.getCode(), json);
        printLog(e, "字段验证异常");
        return result;
    }

    @ResponseBody
    @ExceptionHandler(value = WorkFlowException.class)
    public ActionResult workFlowException(WorkFlowException e) {
        log.error("系统异常:" + e.getMessage(), e);
        return ActionResult.fail(e.getMessage());
    }

    @ResponseBody
    @ExceptionHandler(value = WxErrorException.class)
    public ActionResult wxErrorException(WxErrorException e) {
        log.error("系统异常:" + e.getMessage(), e);
        return ActionResult.fail(e.getError().getErrorCode(), "操作过于频繁");
    }
    @ResponseBody
    @ExceptionHandler(value = BaseException.class)
    public ActionResult baseExceptionException(BaseException e) {
        log.error("系统异常:" + e.getMessage(), e);
        printLog(e, "系统异常");
        return ActionResult.fail(ActionResultCode.Fail.getCode(), e.getMessage());
    }

    @ResponseBody
    @ExceptionHandler(value = PermissionException.class)
    public ActionResult permissionException(PermissionException e) {
        log.error("系统异常:" + e.getMessage(), e);
        printLog(e, "系统异常");
        return ActionResult.fail(ActionResultCode.Fail.getCode(), e.getMessage());
    }
    @ResponseBody
    @ExceptionHandler(value = Exception.class)
    public ActionResult exception(Exception e) {
        log.error("系统异常:" + e.getMessage(), e);
        printLog(e, "系统异常");
        return ActionResult.fail(ActionResultCode.Fail.getCode(), "系统异常");
    }

    private void printLog(Exception e, String module) {
        UserInfo userInfo = userProvider.get();
        LogEntity entity = new LogEntity();
        entity.setId(RandomUtil.uuId());
        entity.setCategory(LogSortEnum.Operate.getCode());
        entity.setUserId(userInfo.getUserId());
        String userName = StringUtils.isNotEmpty(userInfo.getUserName()) ? userInfo.getUserName()+"/"+userInfo.getUserAccount() : userInfo.getUserAccount();
        entity.setUserName(userName);
        if (!ServletUtils.getIsMobileDevice()) {
            String modelName = module;
            entity.setModuleName(modelName);
        }
        StringBuffer sb = new StringBuffer();
        sb.append(e.toString() + "\n");
        StackTraceElement[] stackArray = e.getStackTrace();
        for (int i = 0; i < stackArray.length; i++) {
            StackTraceElement element = stackArray[i];
            sb.append(element.toString() + "\n");
        }
        entity.setJsons(sb.toString());
        entity.setRequestUrl(ServletUtils.getRequest().getServletPath());
        entity.setRequestMethod(ServletUtils.getRequest().getMethod());
        entity.setCategory(4);
        entity.setUserId(userInfo.getUserId());
        entity.setIpAddress(IpUtil.getIpAddr());
        entity.setCreatorTime(new Date());
        entity.setPlatForm(ServletUtils.getUserAgent());
        if (configValueUtil.isMultiTenancy() && StringUtils.isEmpty(userInfo.getTenantId())) {
            log.error("请求异常， 无登陆租户：" + ReflectionUtil.toString(entity), e);
        }else{
            logService.save(entity);
        }
    }

    private void printLoginLog(Exception e, String module) {
        if (!configValueUtil.isMultiTenancy()) {
            UserInfo userInfo = userProvider.get();
            LogEntity entity = new LogEntity();
            entity.setId(RandomUtil.uuId());
            entity.setCategory(LogSortEnum.Operate.getCode());
            entity.setUserId(userInfo.getUserId());
            String userName = StringUtils.isNotEmpty(userInfo.getUserName()) ? userInfo.getUserName()+"/"+userInfo.getUserAccount() : userInfo.getUserAccount();
            entity.setUserName(userName);
            if (!ServletUtils.getIsMobileDevice()) {
                String modelName = module;
                entity.setModuleName(modelName);
            }
            entity.setAbstracts("登陆失败");
            entity.setJsons(e.getMessage());
            entity.setRequestUrl(ServletUtils.getRequest().getServletPath());
            entity.setRequestMethod(ServletUtils.getRequest().getMethod());
            entity.setCategory(1);
            entity.setUserId(userInfo.getUserId());
            entity.setIpAddress(IpUtil.getIpAddr());
            entity.setCreatorTime(new Date());
            entity.setPlatForm(ServletUtils.getUserAgent());
            logService.save(entity);
        }

    }

}
