package com.bringspring.system.external.config.mutil;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.base.entity.SysConfigEntity;
import com.bringspring.system.base.exception.BaseException;
import com.bringspring.system.base.service.SysConfigService;
import com.bringspring.system.external.bean.WeComModel;
import com.bringspring.system.external.handler.*;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.service.McMsgAccountService;
import com.google.common.collect.Maps;
import lombok.val;
import me.chanjar.weixin.common.api.WxConsts;
import me.chanjar.weixin.common.error.WxRuntimeException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.api.impl.WxCpServiceImpl;
import me.chanjar.weixin.cp.config.impl.WxCpDefaultConfigImpl;
import me.chanjar.weixin.cp.constant.WxCpConsts;
import me.chanjar.weixin.cp.message.WxCpMessageRouter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.bringspring.system.external.constant.WxCpSysConfigConsts.WECOM_KEY_NAME;
import static com.bringspring.system.msgcenter.enums.ChannelTypeEnum.QY_MSG;

/**
 * 多实例配置
 *
 * @author <a href="https://github.com/0katekate0">Wang_Wong</a>
 */
@Component
public class WxCpConfiguration {
    @Autowired
    private LogHandler logHandler;
    @Autowired
    private NullHandler nullHandler;
    @Autowired
    private LocationHandler locationHandler;
    @Autowired
    private MenuHandler menuHandler;
    @Autowired
    private MsgHandler msgHandler;
    @Autowired
    private ContactChangeHandler contactChangeHandler;
    @Autowired
    private UnsubscribeHandler unsubscribeHandler;
    @Autowired
    private SubscribeHandler subscribeHandler;

    @Autowired
    private SysConfigService sysConfigService;
    @Autowired
    private McMsgAccountService mcMsgAccountService;

    private Map<String, WxCpMessageRouter> routers = Maps.newHashMap();
    private Map<String, WeComModel> weComModels = Maps.newHashMap();
    private Map<String, WxCpService> corpServices = Maps.newHashMap();
    private Map<String, WxCpService> cpServices = Maps.newHashMap();

    public Map<String, WxCpMessageRouter> getRouters() {
        return routers;
    }

    public Map<String, WeComModel> getWeComModels() {
        return weComModels;
    }

    /**
     * 判断是互联企业配置 或是 多例企业配置
     * true为互联企业配置，false为多例企业配置
     *
     * @return
     */
    public boolean getIsLinkedCorp() {
        if (CollectionUtil.isNotEmpty(weComModels) && weComModels.size() == 1) {
            // 企业配置不为空 且仅配置了一个
            final Optional<WeComModel> firstValue = weComModels.values().stream().findFirst();
            if (firstValue.isPresent() && firstValue.get().getIsLinkedCorp()) {
                return firstValue.get().getIsLinkedCorp();
            }
        }
        return false;
    }

    public WxCpService getCpService(String corpId, Integer agentId) {
        String key = corpId + agentId;
        WxCpService cpService = cpServices.get(key);
        if (ObjectUtil.isEmpty(cpService)) {
            initServices();
            cpService = cpServices.get(key);
        }
        return cpService;
    }

    public WeComModel getWeComModel(String corpId) {
        WeComModel weComModel = weComModels.get(corpId);
        if (ObjectUtil.isEmpty(weComModel)) {
            initServices();
            weComModel = weComModels.get(corpId);
        }
        return weComModel;
    }

    public WxCpService getCorpService(String corpId) {
        String key = corpId;
        WxCpService corpService = corpServices.get(key);
        if (ObjectUtil.isEmpty(corpService)) {
            initServices();
            corpService = corpServices.get(key);
        }
        return corpService;
    }

    public void initServices() {
        try {
            List<McMsgAccountEntity> account = mcMsgAccountService.getAccountByCategory(QY_MSG.getCode());
            if (CollectionUtil.isEmpty(account)) {
                throw new WxRuntimeException("未添加企业微信相关应用配置！");
            }
            List<WeComModel> listModel = new ArrayList<>();
            account.stream().forEach(a -> {
                WeComModel weComModel = new WeComModel();
                weComModel.setAccountConfigId(a.getId()); // 应用配置ID
                weComModel.setName(a.getFullName()); // 名称
                weComModel.setQyhCorpId(a.getEnterpriseId()); // 企业id
                weComModel.setQyhCorpSecret(a.getAppKey()); // 通讯录Secret
                weComModel.setQyhAgentId(a.getAgentId()); // 应用id
                weComModel.setQyhAgentSecret(a.getBearer()); // 应用Secret
                weComModel.setToken(a.getAppId()); // 回调token
                weComModel.setEncodingAESKey(a.getAppSecret()); // 回调encodingAESKey
                weComModel.setIsLinkedCorp(Boolean.parseBoolean(a.getProgramState())); // 是否互联
                weComModel.setQyhMobileUrl(a.getAddress()); // 应用访问地址
                listModel.add(weComModel);
            });

            SysConfigEntity weComInfo = sysConfigService.getConfigByKeyName(WECOM_KEY_NAME);
            if (ObjectUtil.isNotEmpty(weComInfo) && StringUtils.isNotEmpty(weComInfo.getKeyValue())) {
                String keyValue = weComInfo.getKeyValue();
                List<WeComModel> sysConfig = JsonUtil.getJsonToList(keyValue, WeComModel.class);
                sysConfig.stream().forEach(a -> {
                    listModel.stream().forEach(b -> {
                        if (b.getAccountConfigId().equals(a.getAccountConfigId())) {
                            b.setQyhIsSynOrg(a.getQyhIsSynOrg()); // 启用同步组织
                            b.setQyhIsSynUser(a.getQyhIsSynUser()); // 启用同步用户
                            b.setOrgParent(a.getOrgParent()); // 所属组织
                        }
                    });
                });
            }

            cpServices = listModel.stream().map(a -> {
                String qyhCorpId = a.getQyhCorpId();
                Integer agentId = Integer.valueOf(a.getQyhAgentId());
                val configStorage = new WxCpDefaultConfigImpl();
                configStorage.setCorpId(qyhCorpId);
                configStorage.setAgentId(agentId);
                configStorage.setCorpSecret(a.getQyhAgentSecret());
                configStorage.setToken(a.getToken());
                configStorage.setAesKey(a.getEncodingAESKey());

                val service = new WxCpServiceImpl();
                service.setWxCpConfigStorage(configStorage);

                routers.put(qyhCorpId + agentId, this.newRouter(service));
                return service;
            }).collect(Collectors.toMap(service -> service.getWxCpConfigStorage().getCorpId() + service.getWxCpConfigStorage().getAgentId(), a -> a));

            corpServices = listModel.stream().map(a -> {
                String qyhCorpId = a.getQyhCorpId();
                Integer agentId = Integer.valueOf(a.getQyhAgentId());
                val configStorage = new WxCpDefaultConfigImpl();
                configStorage.setCorpId(qyhCorpId);
                configStorage.setCorpSecret(a.getQyhCorpSecret());
                configStorage.setAgentId(agentId);
                configStorage.setToken(a.getToken());
                configStorage.setAesKey(a.getEncodingAESKey());

                val service = new WxCpServiceImpl();
                service.setWxCpConfigStorage(configStorage);

                routers.put(qyhCorpId, this.newRouter(service));

                weComModels.put(qyhCorpId, a);
                return service;
            }).collect(Collectors.toMap(service -> service.getWxCpConfigStorage().getCorpId(), a -> a));

        } catch (BaseException e) {
            e.printStackTrace();
        }
    }

    private WxCpMessageRouter newRouter(WxCpService wxCpService) {
        final val newRouter = new WxCpMessageRouter(wxCpService);

        // 记录所有事件的日志 （异步执行）
        newRouter.rule().handler(this.logHandler).next();

        // 自定义菜单事件
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxConsts.MenuButtonType.CLICK).handler(this.menuHandler).end();

        // 点击菜单链接事件（这里使用了一个空的处理器，可以根据自己需要进行扩展）
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxConsts.MenuButtonType.VIEW).handler(this.nullHandler).end();

        // 关注事件
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxConsts.EventType.SUBSCRIBE).handler(this.subscribeHandler)
                .end();

        // 取消关注事件
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxConsts.EventType.UNSUBSCRIBE)
                .handler(this.unsubscribeHandler).end();

        // 上报地理位置事件
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxConsts.EventType.LOCATION).handler(this.locationHandler)
                .end();

        // 接收地理位置消息
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.LOCATION)
                .handler(this.locationHandler).end();

        // 扫码事件（这里使用了一个空的处理器，可以根据自己需要进行扩展）
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxConsts.EventType.SCAN).handler(this.nullHandler).end();

        // 成员、部门变更事件
        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxCpConsts.EventType.CHANGE_CONTACT).handler(this.contactChangeHandler).end();

        newRouter.rule().async(false).msgType(WxConsts.XmlMsgType.EVENT)
                .event(WxCpConsts.EventType.ENTER_AGENT).handler(new EnterAgentHandler()).end();

        // 默认
        newRouter.rule().async(false).handler(this.msgHandler).end();

        return newRouter;
    }

}
