package com.bringspring.system.external.controller;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.listener.DIngStream;
import com.bringspring.system.external.bean.WeComModel;
import com.bringspring.system.external.bean.WxCpDepartTreeNode;
import com.bringspring.system.external.config.mutil.WxCpConfiguration;
import com.bringspring.system.external.service.SynPushToThirdQyService;
import com.bringspring.system.external.service.WxCpCorpService;
import com.bringspring.system.external.util.WxCpCorpValidateUtil;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.model.message.OraganizeListVO;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.message.util.SynQyWebChatUtil;
import com.bringspring.system.message.util.SynThirdTotal;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.model.organize.OrganizeModel;
import com.bringspring.system.permission.service.OrganizeService;
import com.bringspring.system.permission.service.PositionService;
import com.bringspring.system.permission.service.UserService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.bean.WxCpDepart;
import me.chanjar.weixin.cp.bean.WxCpUser;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static com.bringspring.system.message.util.SynThirdConsts.*;

/**
 * 企业微信-企业-获取接口信息
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2024/1/31 9:30
 */
@Slf4j
@Tag(name = "企业微信-企业")
@RestController
@RequestMapping("/api/external/corp/{corpId}/{agentId}")
public class WxCpCorpController {
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private SynThirdInfoService synThirdInfoService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    PositionService positionService;
    @Autowired
    private UserService userService;

    @Autowired
    private DIngStream dIngStream;

    @Autowired
    WxCpConfiguration wxCpConfiguration;

    @Autowired
    WxCpCorpValidateUtil wxCpCorpValidateUtil;

    @Autowired
    private WxCpCorpService wxCpCorpService;

    @Autowired
    private SynPushToThirdQyService synPushToThirdQyService;


    /**
     * 获取企业部门列表
     * 第一步：获取部门列表
     * <p>
     * 第二步：与第三方同步表比对，存在则更新 不存在则新增
     * <p>
     * 第三步：数据批量持久化
     *
     * @return
     */
    @Operation(summary="企业微信【企业】获取企业部门列表")
    @GetMapping("/department/deptList")
    @DSTransactional
    @CacheEvict(value = "OrganizeCache", allEntries = true)
    public ActionResult corpDepartmentList(@PathVariable String corpId, @PathVariable Integer agentId) {
        UserInfo userInfo = userProvider.get();
        if (ObjectUtil.isEmpty(userInfo)) {
            return ActionResult.fail("获取当前登录用户信息失败");
        }
        final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, agentId);
        if (wxCpService == null) {
            return ActionResult.fail(String.format("未找到对应corpId=[%s]、agentId=[%d]的配置，请核实！", corpId, agentId));
        }
        try {
            /**
             * 第一步：获取部门列表
             */
            // 获取部门列表
            List<WxCpDepart> departList = wxCpService.getDepartmentService().list(null);
            if (departList == null || departList.size() == 0) {
                return ActionResult.fail("获取获取部门列表接口数据为空");
            }
            Long parentid = 0L; // 企业微信默认顶级节点不允许删除，且顶级节点的父级ID为0.
            WeComModel weComModel = wxCpConfiguration.getWeComModel(corpId);
            SynThirdInfoEntity parentThird = null;
            if (StringUtils.isNotEmpty(weComModel.getOrgParent())) {
                // 组织上级，配置的手动维护好的企业上级组织部门。
                parentThird = synThirdInfoService.getInfoBySysObjId(THIRD_TYPE_QY, DATA_TYPE_ORG,
                        weComModel.getOrgParent());
                if (ObjectUtil.isNotEmpty(parentThird) && StringUtils.isNotEmpty(parentThird.getThirdObjectId())) {
                    parentid = Long.valueOf(parentThird.getThirdObjectId());
                }
            }
            List<WxCpDepartTreeNode> treeList = JsonUtil.getJsonToList(departList, WxCpDepartTreeNode.class);
            List<WxCpDepartTreeNode> wxCpDepartTreeNodes = wxCpCorpService.deptListToTree(treeList, parentid);
            /**
             * 保存到本地
             */
            wxCpCorpService.saveOrUpdateOrganizeList(wxCpDepartTreeNodes, corpId, parentThird);

            // 获取结果
            SynThirdTotal synTotal = synThirdInfoService.getCorpDataTypeSynTotal(THIRD_TYPE_QY, DATA_TYPE_ORG, corpId);
            return ActionResult.success(synTotal);
        } catch (Exception e) {
            e.printStackTrace();
            return ActionResult.fail("系统运行时异常：" + e);
        }
    }


    /**
     * 获取本地部门列表推送push到企业微信
     *
     * @return
     */
    @Operation(summary="本地部门列表推送企业部门列表")
    @GetMapping("/department/pushBatch")
    @DSTransactional
    public ActionResult pushBatchDeptList(@PathVariable String corpId, @PathVariable Integer agentId) {
        UserInfo userInfo = userProvider.get();
        if (ObjectUtil.isEmpty(userInfo)) {
            return ActionResult.fail("获取当前登录用户信息失败");
        }
        try {
            List<SynThirdInfoEntity> synThirdInfoList = synThirdInfoService.getListByCrop(THIRD_TYPE_QY,
                    DATA_TYPE_ORG, corpId);
            List<String> idList =
                    synThirdInfoList.stream().map(SynThirdInfoEntity::getSystemObjectId).collect(Collectors.toList());
            List<OrganizeEntity> organizeList = organizeService.getListAll(idList, null);

            // 部门进行树结构化,固化上下层级序列化
            List<OrganizeModel> organizeModelList = JsonUtil.getJsonToList(organizeList, OrganizeModel.class);
            List<SumTree<OrganizeModel>> trees = TreeDotUtils.convertListToTreeDot(organizeModelList);
            List<OraganizeListVO> listVO = JsonUtil.getJsonToList(trees, OraganizeListVO.class);

            // 转化成为按上下层级顺序排序的列表数据
            List<OrganizeEntity> listByOrder = new ArrayList<>();
            for (OraganizeListVO organizeVo : listVO) {
                OrganizeEntity entity =
                        organizeList.stream().filter(t -> t.getId().equals(organizeVo.getId())).findFirst().orElse(null);
                listByOrder.add(entity);
                SynQyWebChatUtil.getOrganizeTreeToList(organizeVo, organizeList, listByOrder);
            }

            // 根据同步表、部门表进行比较，判断不存的执行删除
            for (SynThirdInfoEntity synThirdInfoEntity : synThirdInfoList) {
                if (organizeList.stream().filter(t -> t.getId().equals(synThirdInfoEntity.getSystemObjectId())).count() == 0 ? true : false) {
                    //执行删除操作
                    synPushToThirdQyService.deleteDepartmentSysToQy(userInfo, true,
                            synThirdInfoEntity.getSystemObjectId(), null);
                }
            }

            // 根据公司表、同步表进行比较，决定执行创建、还是更新
            for (OrganizeEntity organizeEntity : listByOrder) {
                if (synThirdInfoList.stream().filter(t -> t.getSystemObjectId().equals(organizeEntity.getId())).count() > 0 ? true : false) {
                    // 执行更新功能
                    synPushToThirdQyService.updateDepartmentSysToQy(userInfo, true, organizeEntity, null);
                } else {
                    // 执行创建功能
                    synPushToThirdQyService.createDepartmentSysToQy(userInfo, true, organizeEntity, null);
                }
            }

        } catch (Exception e) {
            ActionResult.fail(e.toString());
        }

        //获取结果
        SynThirdTotal synTotal = synThirdInfoService.getCorpDataTypeSynTotal(THIRD_TYPE_QY, DATA_TYPE_ORG, corpId);
        return ActionResult.success(synTotal);
    }


    /**
     * 获取企业部门成员详情
     * 第一步：获取第三方同步表中corpId的deptId
     * 第二步：遍历第一步获取的数据，依次调用接口获取部门成员详情
     * 第三步：判断系统用户新增或更新，封装用户信息
     * 第四步：数据批量持久化
     *
     * @return ignore
     */
    @Operation(summary="企业微信【企业】获取企业部门成员详情")
    @GetMapping("/user/userList")
    @DSTransactional
    @CacheEvict(value = "UserCache", allEntries = true)
    public ActionResult corpUserList(@PathVariable String corpId, @PathVariable Integer agentId) {
        UserInfo userInfo = userProvider.get();
        if (ObjectUtil.isEmpty(userInfo)) {
            return ActionResult.fail("获取当前登录用户信息失败");
        }

        final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, agentId);
        if (wxCpService == null) {
            return ActionResult.fail(String.format("未找到对应corpId=[%s]、agentId=[%d]的配置，请核实！", corpId, agentId));
        }

        try {

            /**
             * 获取部门成员详情
             * WxJava接口WxCpUserService的listByDepartment方法有参数：
             * departId - 必填。部门id
             * fetchChild - 非必填。1/0：是否递归获取子部门下面的成员
             * status - 非必填。0获取全部员工，1获取已关注成员列表，2获取禁用成员列表，4获取未关注成员列表。status可叠加
             *
             * 所以利用企业微信通讯录“不允许删除根部门”且“根部门ID为1”的特点，获取全部员工。
             */
            Long departId = 1L; // 根部门ID
            boolean fetchChild = true; // 递归获取子部门下面的成员
            Integer status = 0; // 获取全部员工
            List<WxCpUser> users = wxCpService.getUserService().listByDepartment(departId, fetchChild, status);
            if (users == null || users.size() == 0) {
                return ActionResult.fail("企业微信通讯录员工信息为空");
            }

            wxCpCorpService.saveOrUpdateUserList(users, wxCpService);
            //获取结果
            SynThirdTotal synTotal = synThirdInfoService.getCorpDataTypeSynTotal(THIRD_TYPE_QY, DATA_TYPE_USER, corpId);
            return ActionResult.success(synTotal);
        } catch (Exception e) {
            e.printStackTrace();
            return ActionResult.fail("系统运行时异常：" + e);
        }
    }

    /**
     * 获取本地用户列表推送push到企业微信
     *
     * @return
     */
    @Operation(summary="本地用户列表推送企业部门成员列表")
    @GetMapping("/user/pushBatch")
    public ActionResult pushBatchUserList(@PathVariable String corpId, @PathVariable Integer agentId) {
        UserInfo userInfo = userProvider.get();
        if (ObjectUtil.isEmpty(userInfo)) {
            return ActionResult.fail("获取当前登录用户信息失败");
        }
        try {
            // 当前指定数据类型的数据列表 1:企业微信;2:用户;企业ID(corpId)
            List<SynThirdInfoEntity> synThirdInfoList = synThirdInfoService.getListByCrop(THIRD_TYPE_QY,
                    DATA_TYPE_USER, corpId, false, false);

            List<String> idList =
                    synThirdInfoList.stream().map(SynThirdInfoEntity::getSystemObjectId).collect(Collectors.toList());
            List<UserEntity> userList = userService.getUserByIds(idList);

            // 根据同步表、用户表进行比较，判断不存的执行删除
            for (SynThirdInfoEntity synThirdInfoEntity : synThirdInfoList) {
                if (userList.stream().filter(t -> t.getId().equals(synThirdInfoEntity.getSystemObjectId())).count() == 0 ? true : false) {
                    //执行删除操作
                    synPushToThirdQyService.deleteUserSysToQy(userInfo, true, synThirdInfoEntity.getSystemObjectId(),
                            null);
                }
            }
            List<OrganizeEntity> list = organizeService.getList();
            // 根据用户表、同步表进行比较，决定执行创建、还是更新
            for (UserEntity userEntity : userList) {
                OrganizeEntity info = organizeService.getInfo(userEntity.getOrganizeId());
                if (synThirdInfoList.stream().filter(t -> userEntity.getId().equals(t.getSystemObjectId())).count() > 0 ? true : false) {
                    // 执行更新功能
                    synPushToThirdQyService.updateUserSysToQy(userInfo, true, userEntity, null);
                } else {
                    // 执行创建功能
                    synPushToThirdQyService.createUserSysToQy(userInfo, true, userEntity, null);
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
            return ActionResult.fail(e.getMessage());
        }

        //获取结果
        SynThirdTotal synTotal = synThirdInfoService.getCorpDataTypeSynTotal(THIRD_TYPE_QY, DATA_TYPE_USER, corpId);
        return ActionResult.success(synTotal);
    }


    /**
     * 获取第三方企业微信的组织与用户同步统计信息
     *
     * @param corpId 企业ID
     * @return ignore
     */
    @Operation(summary="获取第三方企业的组织与用户同步统计信息")
    @GetMapping("/getWxCpSynThirdInfoTotal")
    public ActionResult getWxCpSynThirdInfoTotal(@PathVariable String corpId, @PathVariable Integer agentId) {
        return ActionResult.success(synThirdInfoService.getCorpSynTotal(THIRD_TYPE_QY, corpId, false));
    }
}
