package com.bringspring.system.msgcenter.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.system.msgcenter.constant.Constants.Delimiters;
import com.bringspring.system.msgcenter.constant.Constants.MessageConfig;
import com.bringspring.system.msgcenter.entity.McMsgSendEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgEntity;
import com.bringspring.system.msgcenter.model.mctaskmsg.*;
import com.bringspring.system.msgcenter.service.McMsgSendService;
import com.bringspring.system.msgcenter.service.McTaskMsgReceiveService;
import com.bringspring.system.msgcenter.service.McTaskMsgService;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.OrganizeService;
import com.bringspring.system.permission.service.UserService;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import static com.bringspring.system.msgcenter.constant.DictType.TASK_CODE;

/**
 * 业务场景消息表
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2024-06-14
 */
@Slf4j
@RestController
@Tag(name = "业务场景消息表")
@RequestMapping("/api/msgCenter/mcTaskMsg")
public class McTaskMsgController {

    @Autowired
    private UserProvider userProvider;

    @Autowired
    private BaseDataUtil baseDataUtil;

    @Autowired
    private McTaskMsgService mcTaskMsgService;
    @Autowired
    private UserService userService;
    @Autowired
    private OrganizeService organizeService;

    @Autowired
    private McMsgSendService mcMsgSendService;

    @Autowired
    private McTaskMsgReceiveService mcTaskMsgReceiveService;


    /**
     * 列表
     *
     * @param mcTaskMsgPagination
     * @return
     */
    @PostMapping("/getList")
    public ActionResult list(@RequestBody McTaskMsgPagination mcTaskMsgPagination) throws IOException {
        List<McTaskMsgEntity> list = mcTaskMsgService.getList(mcTaskMsgPagination);
        List<McTaskMsgListVO> listVO = JsonUtil.getJsonToList(list, McTaskMsgListVO.class);
        for (McTaskMsgListVO vo : listVO) {
            //处理id字段转名称，若无需转或者为空可删除
            String toUserIds = vo.getToUserIds();
            if (StringUtils.isNotEmpty(toUserIds) && !MessageConfig.ALL_USERS.equals(toUserIds)) {
                // 指定用户
                List<String> toUserArr = Arrays.asList(toUserIds.split(Delimiters.COMMA));
                List<UserEntity> userList = userService.getUserByIds(toUserArr);
                if (CollectionUtil.isNotEmpty(userList)) {
                    vo.setToUserIds(userList.stream().map(UserEntity::getRealName).collect(Collectors.joining(Delimiters.COMMA)));
                }
            }
            String toDeptIds = vo.getToDeptIds();
            if (StringUtils.isNotEmpty(toDeptIds)) {
                // 指定用户
                List<String> toDeptArr = Arrays.asList(toDeptIds.split(Delimiters.COMMA));
                List<OrganizeEntity> deptList = organizeService.getListAll(toDeptArr, null);
                if (CollectionUtil.isNotEmpty(deptList)) {
                    vo.setToDeptIds(deptList.stream().map(OrganizeEntity::getFullName).collect(Collectors.joining(Delimiters.COMMA)));
                }
            }
            vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
            vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));
            McMsgSendEntity msgSendEntity = mcMsgSendService.getEntityByEnCode(vo.getSendCode());
            if (ObjectUtil.isNotNull(msgSendEntity)) {
                vo.setSendName(msgSendEntity.getFullName());
            }
            String taskCode = baseDataUtil.getDictName(vo.getTaskCode(), TASK_CODE.getCode());
            if (StringUtils.isNotEmpty(taskCode)) {
                vo.setTaskCode(taskCode);
            }
        }

        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        PaginationVO page = JsonUtil.getJsonToBean(mcTaskMsgPagination, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }


    /**
     * 创建
     *
     * @param mcTaskMsgCrForm
     * @return
     */
    @PostMapping
    public ActionResult create(@RequestBody @Valid McTaskMsgCrForm mcTaskMsgCrForm) throws DataException {
        try {
            mcTaskMsgService.addTaskMsg(mcTaskMsgCrForm);
            return ActionResult.success("创建成功");
        } catch (Exception e) {
            return ActionResult.fail("创建失败：" + e.getMessage());
        }
    }


    /**
     * 信息
     *
     * @param id
     * @return
     */
    @GetMapping("/{id}")
    public ActionResult<McTaskMsgInfoVO> info(@PathVariable("id") String id) {
        McTaskMsgEntity entity = mcTaskMsgService.getInfo(id);
        McTaskMsgInfoVO vo = JsonUtil.getJsonToBean(entity, McTaskMsgInfoVO.class);
        McMsgSendEntity msgSendEntity = mcMsgSendService.getEntityByEnCode(vo.getSendCode());
        if (ObjectUtil.isNotNull(msgSendEntity)) {
            vo.setSendName(msgSendEntity.getFullName());
        }

        /**
         * 根据消息id和用户id更新消息阅读状态
         * 解决：当用户多个消息渠道接收到消息后，一端点击了获取了消息详情，则设置当前用户的其他渠道消息为已读。
         */
        mcTaskMsgReceiveService.updateIsReadByMsgIdAndUserId(id, userProvider.get().getUserId());
        return ActionResult.success(vo);
    }

    /**
     * 表单信息(详情页)
     *
     * @param id
     * @return
     */
    @GetMapping("/detail/{id}")
    public ActionResult<McTaskMsgInfoVO> detailInfo(@PathVariable("id") String id) {
        McTaskMsgEntity entity = mcTaskMsgService.getInfo(id);
        McTaskMsgInfoVO vo = JsonUtil.getJsonToBean(entity, McTaskMsgInfoVO.class);

        //处理id字段转名称，若无需转或者为空可删除
        String toUserIds = vo.getToUserIds();
        if (StringUtils.isNotEmpty(toUserIds) && !MessageConfig.ALL_USERS.equals(toUserIds)) {
            // 指定用户
            List<String> toUserArr = Arrays.asList(toUserIds.split(Delimiters.COMMA));
            List<UserEntity> userList = userService.getUserByIds(toUserArr);
            if (CollectionUtil.isNotEmpty(userList)) {
                vo.setToUserIds(userList.stream().map(UserEntity::getRealName).collect(Collectors.joining(Delimiters.COMMA)));
            }
        }
        String toDeptIds = vo.getToDeptIds();
        if (StringUtils.isNotEmpty(toDeptIds)) {
            // 指定用户
            List<String> toDeptArr = Arrays.asList(toDeptIds.split(Delimiters.COMMA));
            List<OrganizeEntity> deptList = organizeService.getListAll(toDeptArr, null);
            if (CollectionUtil.isNotEmpty(deptList)) {
                vo.setToDeptIds(deptList.stream().map(OrganizeEntity::getFullName).collect(Collectors.joining(Delimiters.COMMA)));
            }
        }
        //添加到详情表单对象中
        vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
        vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));

        return ActionResult.success(vo);
    }


    /**
     * 更新
     *
     * @param id
     * @return
     */
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid McTaskMsgUpForm mcTaskMsgUpForm) throws DataException {
        McTaskMsgEntity entity = mcTaskMsgService.getInfo(id);
        if (entity != null) {
            mcTaskMsgService.update(id, mcTaskMsgUpForm);
            return ActionResult.success("更新成功");
        } else {
            return ActionResult.fail("更新失败，数据不存在");
        }
    }


    /**
     * 删除
     *
     * @param id
     * @return
     */
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        McTaskMsgEntity entity = mcTaskMsgService.getInfo(id);
        if (entity != null) {
            mcTaskMsgService.delete(entity);
        }
        return ActionResult.success("删除成功");
    }


    /**
     * 发送消息
     *
     * @param id
     * @return
     */
    @PutMapping("/send/{id}")
    public ActionResult send(@PathVariable("id") String id) throws DataException {
        McTaskMsgEntity entity = mcTaskMsgService.getInfo(id);
        if (entity != null) {
            try {
                mcTaskMsgService.sendTaskMsgReceive(entity);
                return ActionResult.success("发送成功");
            } catch (Exception e) {
                e.printStackTrace();
                final String message = "发送失败：" + e.getMessage();
                log.error(message);
                return ActionResult.fail(message);
            }
        } else {
            return ActionResult.fail("发送失败，数据不存在");
        }
    }
}
