package com.bringspring.system.msgcenter.service.strategy.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.model.OnlineUserModel;
import com.bringspring.common.model.OnlineUserProvider;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.enums.EnabledMarkEnum;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.service.context.SendMessageContext;
import com.bringspring.system.msgcenter.service.strategy.AbstractMessageSender;
import com.bringspring.system.msgcenter.util.BlacklistUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.stream.Collectors;

/**
 * 站内信策略（包含黑名单过滤和WebSocket通知）
 */
@Component
@Slf4j
public class MsgToSystemService extends AbstractMessageSender {

    @Override
    protected int getBatchSize() {
        return 1000;
    }

    @Override
    public ChannelTypeEnum getChannelType() {
        return ChannelTypeEnum.SYS_MSG;
    }


    @Override
    protected void validateParameters(SendMessageContext context) {
        // 无特殊校验
    }

    @Override
    protected void doSend(SendMessageContext context) {
        if (CollectionUtil.isNotEmpty(context.getValidList())) {
            List<McTaskMsgReceiveEntity> receiveList = context.getValidList();
            receiveList.stream().forEach(e -> {
                e.setEnabledMark(EnabledMarkEnum.SUCCESS.getCode());
            });
            context.setValidList(receiveList);

            // 发送WebSocket通知
            sendWebSocketNotification(receiveList, context.getTaskMsgContent(), context.getCurrentUser());
        }
    }

    @Override
    protected void sendBatchNonSystemUser(boolean isAgain, McMsgSendTemplateModel sendTemplate,
                                          McTaskMsgContentEntity taskMsgContent,
                                          List<McTaskMsgReceiveEntity> receiveListS, UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 站内信消息【非系统用户】下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }
        sendBatch(isAgain, sendTemplate, taskMsgContent, receiveListS, currentUserInfo);
    }

    /**
     * 发送webhook
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 业务消息内容
     */
    @Override
    protected void sendBatch(boolean isAgain,
                             McMsgSendTemplateModel sendTemplate,
                             McTaskMsgContentEntity taskMsgContent,
                             List<McTaskMsgReceiveEntity> receiveListS,
                             UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 站内信消息下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }

        // 1. 过滤黑名单
        List<McTaskMsgReceiveEntity> filtered = BlacklistUtil.receiveListFilter(receiveListS);
        if (!filtered.isEmpty()) {
            mcTaskMsgReceiveService.updateBlacklist(filtered, currentUserInfo);
        }

        // 2. 更新发送状态
        receiveListS.forEach(entity -> {
            entity.setEnabledMark(1); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
            entity.setSendTime(new Date());
            entity.setSendUserId(currentUserInfo.getUserId());
        });
        mcTaskMsgReceiveService.updateBatchById(receiveListS);

        // 3. 发送WebSocket通知
        sendWebSocketNotification(receiveListS, taskMsgContent, currentUserInfo);
    }

    private void sendWebSocketNotification(List<McTaskMsgReceiveEntity> receiveListS,
                                           McTaskMsgContentEntity taskMsgContent,
                                           UserInfo currentUserInfo) {

        // ====================== jsbos基础平台 站内信 WebSocket消息推送 START===============================
        // 消息推送 - PC端，这段实现依赖jsbos基础平台架构，若消息中心实现迁移其他基础平台，这段需要考量实现。
        // 遍历接收者列表，发送websocket消息
        List<String> toUserIds = receiveListS.stream().map(i -> i.getSysUserId()).collect(Collectors.toList());
        for (int i = 0; i < toUserIds.size(); i++) {
            CopyOnWriteArraySet<OnlineUserModel> onlineUserList =
                    OnlineUserProvider.getOnlineUserList();
            for (OnlineUserModel item : onlineUserList) {
                if (toUserIds.get(i).equals(item.getUserId()) && currentUserInfo.getTenantId().equals(item.getTenantId())) {
                    JSONObject map = new JSONObject();
                    map.put("method", "messagePush");
                    map.put("unreadNoticeCount", 1);
                    map.put("messageType", 2);
                    map.put("userId", currentUserInfo.getTenantId());
                    map.put("toUserId", toUserIds);
                    map.put("title", taskMsgContent.getTitle());
                    synchronized (map) {
                        item.getWebSocket().getAsyncRemote().sendText(map.toJSONString());
                    }
                }
            }
        }
        // ====================== jsbos基础平台 站内信 WebSocket消息推送 END===============================
    }
}