package com.bringspring.system.permission.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.util.DateUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.system.permission.entity.GroupEntity;
import com.bringspring.system.permission.mapper.GroupMapper;
import com.bringspring.system.permission.service.GroupService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 分组管理业务类实现类
 *
 * @author ：RKKJ开发平台组
 * @version: V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date ：2022/3/10 18:00
 */
@Service
public class GroupServiceImpl extends ServiceImpl<GroupMapper, GroupEntity> implements GroupService {

    @Autowired
    private UserProvider userProvider;

    @Override
    public List<GroupEntity> getList(Pagination pagination) {
        QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
        // 定义变量判断是否需要使用修改时间倒序
        boolean flag = false;
        // 判断关键字
        String keyword = pagination.getKeyword();
        if (StringUtils.isNotEmpty(keyword)) {
            flag = true;
            queryWrapper.lambda().and(
                    t -> t.like(GroupEntity::getFullName, keyword).or().like(GroupEntity::getEnCode, keyword)
            );
        }
        // 获取列表
        queryWrapper.lambda().orderByAsc(GroupEntity::getSortCode).orderByDesc(GroupEntity::getCreatorTime);
        if (flag) {
            queryWrapper.lambda().orderByDesc(GroupEntity::getLastModifyTime);
        }
        Page<GroupEntity> page = new Page<>(pagination.getCurrentPage(), pagination.getPageSize());
        IPage<GroupEntity> iPage = this.page(page, queryWrapper);
        return pagination.setData(iPage.getRecords(), iPage.getTotal());
    }

    @Override
    public List<GroupEntity> list() {
        QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().orderByAsc(GroupEntity::getSortCode).orderByDesc(GroupEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public List<GroupEntity> list(String keyword) {
        QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(keyword)) {
            //通过关键字查询
            queryWrapper.lambda().and(t -> t.like(GroupEntity::getFullName, keyword));
        }
        queryWrapper.lambda().orderByAsc(GroupEntity::getSortCode).orderByDesc(GroupEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public GroupEntity getInfo(String id) {
        QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(GroupEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    public void crete(GroupEntity entity) {
        entity.setId(RandomUtil.uuId());
        entity.setCreatorUserId(userProvider.get().getUserId());
        entity.setCreatorTime(DateUtil.getNowDate());
        entity.setLastModifyTime(DateUtil.getNowDate());
        this.save(entity);
    }

    @Override
    public Boolean update(String id, GroupEntity entity) {
        entity.setId(id);
        entity.setLastModifyUserId(userProvider.get().getUserId());
        entity.setLastModifyTime(DateUtil.getNowDate());
        return this.updateById(entity);
    }

    @Override
    public void delete(GroupEntity entity) {
        this.removeById(entity.getId());
    }

    @Override
    public Boolean isExistByFullName(String fullName, String id) {
        QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(GroupEntity::getFullName, fullName);
        if (!StringUtils.isEmpty(id)) {
            queryWrapper.lambda().ne(GroupEntity::getId, id);
        }
        return this.count(queryWrapper) > 0;
    }

    @Override
    public Boolean isExistByEnCode(String enCode, String id) {
        QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(GroupEntity::getEnCode, enCode);
        if (!StringUtils.isEmpty(id)) {
            queryWrapper.lambda().ne(GroupEntity::getId, id);
        }
        return this.count(queryWrapper) > 0;
    }

    @Override
    public List<GroupEntity> getGroupList(List<String> idList) {
        if (idList.size() > 0) {
            QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(GroupEntity::getId, idList).select(GroupEntity::getId, GroupEntity::getFullName);
            return this.list(queryWrapper);
        }
        return new ArrayList<>();
    }

    @Override
    public List<GroupEntity> getGroupList(Set<String> idList) {
        if (idList.size() > 0) {
            QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().orderByAsc(GroupEntity::getSortCode).orderByDesc(GroupEntity::getCreatorTime);
            queryWrapper.lambda().select(GroupEntity::getId, GroupEntity::getFullName).in(GroupEntity::getId, idList);
            return this.list(queryWrapper);
        }
        return new ArrayList<>();
    }

    @Override
    public List<GroupEntity> getListByIds(List<String> ids) {
        List<GroupEntity> list = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(ids)){
            QueryWrapper<GroupEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(GroupEntity::getId, ids);
            queryWrapper.lambda().orderByAsc(GroupEntity::getSortCode).orderByDesc(GroupEntity::getCreatorTime);
            list = this.list(queryWrapper);
        }
        return list;
    }

}
