package com.bringspring.system.permission.service.strategy;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.permission.model.authorize.ConditionModel;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;


/**
 * 具体策略类:
 * 定义：实现策略接口的具体算法类。
 * 角色：提供策略接口中定义的算法的具体实现。
 * 职责：根据条件模型ConditionModel中的条件信息，生成QueryWrapper的查询条件。
 * 适用性：String、varchar类型的条件。
 * 依赖：UserProvider、OrganizeService。
 * 关联：ConditionModel、QueryWrapper。
 * 协作：ConditionStrategy、StringConditionStrategy。
 * <p>
 * string、varchar类型的条件策略
 */
@Component
public class StringConditionStrategy implements ConditionStrategy {

    @Autowired
    private ConditionStrategyFactory conditionStrategyFactory;

    @Autowired
    private ContentStringSelect contentStringSelect;

    @Autowired
    private ContentStringInputText contentStringInputText;

    @Override
    public void generateCondition(QueryWrapper<T> tw, String logic, ConditionModel.ConditionItemModel fieldItem) {
        String fieldSymbol = fieldItem.getOp();
        String fieldContent = fieldItem.getValue();
        /**
         * 字段类型为text时，处理方式如下：
         * 字段内容：TextConditionStrategy策略中的字段内容处理方式、
         *          @userAraSubordinates当前用户及下属、@organizationAndSuborganization当前组织及子组织
         *          organizeSelect组织选择、depSelect部门选择、posSelect岗位选择、roleSelect角色选择、groupSelect分组选择、
         *          userSelect用户选择、areaSelect区划选择
         * 符号：以上9种字段内容符号为 ==等于、<>不等于、in包含任意一个、notIn不包含任意一个、null为空、notNull不为空
         *
         * 处理方式：
         * 1. 字段内容处理
         *    a. @userAraSubordinates当前用户及下属 -> userProvider.get().getUserId() + userProvider.get().getSubordinateIds()
         *    b. @organizationAndSuborganization当前组织及子组织 ->
         *          String orgId = userProvider.get().getDepartmentId() || userProvider.get().getOrganizeId()
         *          orgId + organizeService.getUnderOrganizations(orgId)
         *    c. organizeSelect组织选择、depSelect部门选择、posSelect岗位选择、roleSelect角色选择、groupSelect分组选择、
         *        userSelect用户选择、areaSelect区划选择 -> 字段内容单选或多选，根据具体情况处理
         *
         * 2. 根据符号设置tw的查询条件
         */
        Object value;
        if (StringUtils.isNotBlank(fieldContent) && fieldContent.startsWith("@")) {
            value = conditionStrategyFactory.getFieldContentText(fieldContent, fieldSymbol);
            if (ObjectUtils.isEmpty(value)) {
                value = conditionStrategyFactory.getFieldContentString(fieldContent, fieldSymbol);
            }
        } else if (StringUtils.isNotBlank(fieldContent) && fieldContent.startsWith("[")) {
            // 处理select多选
            value = contentStringSelect.convertFieldContent(fieldContent, fieldSymbol);
        } else {
            // 处理input任意文本
            value = contentStringInputText.convertFieldContent(fieldContent, fieldSymbol);
        }

        // 设置tw的查询条件
        SymbolCommon.compare(logic, fieldItem.getField(), fieldSymbol, value, tw);
    }

    @Override
    public Integer key() {
        return 4;
    }

}
