package com.bringspring.system.msgcenter.controller;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.external.config.mutil.DingTalkConfiguration;
import com.bringspring.system.external.config.mutil.WxCpConfiguration;
import com.bringspring.system.msgcenter.constant.Constants.EnableStatus;
import com.bringspring.system.msgcenter.constant.Constants.SslStatus;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.model.mcmsgaccount.*;
import com.bringspring.system.msgcenter.service.McMsgAccountService;
import com.bringspring.system.msgcenter.service.McMsgSendTemplateService;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.io.IOException;
import java.util.List;

import static com.bringspring.system.msgcenter.constant.DictType.*;
import static com.bringspring.system.msgcenter.enums.ChannelTypeEnum.DING_MSG;
import static com.bringspring.system.msgcenter.enums.ChannelTypeEnum.QY_MSG;

/**
 * 账号配置表
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2024-06-03
 */
@Slf4j
@RestController
@Tag(name = "账号配置表")
@RequestMapping("/api/msgCenter/mcMsgAccount")
public class McMsgAccountController {

    @Autowired
    private BaseDataUtil baseDataUtil;

    @Autowired
    private McMsgAccountService mcMsgAccountService;
    @Autowired
    private McMsgSendTemplateService mcMsgSendTemplateService;

    @Autowired
    private WxCpConfiguration wxCpConfiguration;
    @Autowired
    private DingTalkConfiguration dingTalkConfiguration;

    /**
     * 列表
     *
     * @param mcMsgAccountPagination
     * @return
     */
    @PostMapping("/getList")
    public ActionResult list(@RequestBody McMsgAccountPagination mcMsgAccountPagination) throws IOException {
        List<McMsgAccountEntity> list = mcMsgAccountService.getList(mcMsgAccountPagination);
        List<McMsgAccountListVO> listVO = JsonUtil.getJsonToList(list, McMsgAccountListVO.class);
        //处理id字段转名称，若无需转或者为空可删除
        for (McMsgAccountListVO vo : listVO) {
            vo.setCategory(baseDataUtil.getDictName(vo.getCategory(), MSG_TYPE.getCode()));
            vo.setSslLink(baseDataUtil.switchSelectValue(vo.getSslLink(), SslStatus.ON.getDisplay(),
                    SslStatus.OFF.getDisplay()));
            vo.setChannel(baseDataUtil.getDictName(vo.getChannel(), SMS_TYPE.getCode()));
            vo.setWebhookType(baseDataUtil.getDictName(vo.getWebhookType(), WEB_HOOK_TYPE.getCode()));
//            vo.setEnabledMark(baseDataUtil.switchSelectValue(vo.getEnabledMark(), EnableStatus.ENABLED.getDisplay(), EnableStatus.DISABLED.getDisplay()));
            vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
            vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));
        }


        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        PaginationVO page = JsonUtil.getJsonToBean(mcMsgAccountPagination, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }


    /**
     * 创建
     *
     * @param mcMsgAccountCrForm
     * @return
     */
    @PostMapping
    @DSTransactional
    public ActionResult create(@RequestBody @Valid McMsgAccountCrForm mcMsgAccountCrForm) throws DataException {
        // 校验编码是否存在重复
        if (!mcMsgAccountService.checkEnCodeUnique(mcMsgAccountCrForm.getEnCode(), null)) {
            return ActionResult.fail("编码已存在");
        }
        // 校验名称是否存在重复
        if (!mcMsgAccountService.checkFullNameUnique(mcMsgAccountCrForm.getFullName(), null)) {
            return ActionResult.fail("名称已存在");
        }

        mcMsgAccountService.create(mcMsgAccountCrForm);

        return ActionResult.success("创建成功");
    }


    /**
     * 信息
     *
     * @param id
     * @return
     */
    @GetMapping("/{id}")
    public ActionResult<McMsgAccountInfoVO> info(@PathVariable("id") String id) {
        McMsgAccountEntity entity = mcMsgAccountService.getInfo(id);
        McMsgAccountInfoVO vo = JsonUtil.getJsonToBean(entity, McMsgAccountInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 表单信息(详情页)
     *
     * @param id
     * @return
     */
    @GetMapping("/detail/{id}")
    public ActionResult<McMsgAccountInfoVO> detailInfo(@PathVariable("id") String id) {
        McMsgAccountEntity entity = mcMsgAccountService.getInfo(id);
        McMsgAccountInfoVO vo = JsonUtil.getJsonToBean(entity, McMsgAccountInfoVO.class);

        vo.setCategory(baseDataUtil.getDictName(vo.getCategory(), MSG_TYPE.getCode()));
        vo.setSslLink(baseDataUtil.switchSelectValue(vo.getSslLink(), SslStatus.ON.getDisplay(),
                SslStatus.OFF.getDisplay()));
        vo.setChannel(baseDataUtil.getDictName(vo.getChannel(), SMS_TYPE.getCode()));
        vo.setWebhookType(baseDataUtil.getDictName(vo.getWebhookType(), WEB_HOOK_TYPE.getCode()));
        vo.setEnabledMark(baseDataUtil.switchSelectValue(vo.getEnabledMark(), EnableStatus.ENABLED.getDisplay(), EnableStatus.DISABLED.getDisplay()));
        vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
        vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));

        return ActionResult.success(vo);
    }


    /**
     * 更新
     *
     * @param id
     * @return
     */
    @PutMapping("/{id}")
    @DSTransactional
    public ActionResult update(@PathVariable("id") String id,
                               @RequestBody @Valid McMsgAccountUpForm mcMsgAccountUpForm) throws DataException {
        // 校验编码是否存在重复
        if (!mcMsgAccountService.checkEnCodeUnique(mcMsgAccountUpForm.getEnCode(), id)) {
            return ActionResult.fail("编码已存在");
        }
        // 校验名称是否存在重复
        if (!mcMsgAccountService.checkFullNameUnique(mcMsgAccountUpForm.getFullName(), id)) {
            return ActionResult.fail("名称已存在");
        }

        McMsgAccountEntity entity = mcMsgAccountService.getInfo(id);
        if (ObjectUtil.isNotNull(entity)) {
            mcMsgAccountService.update(id, mcMsgAccountUpForm);
            if (QY_MSG.getCode().equals(entity.getCategory())) {
                // 企业微信多实例配置重新初始化
                wxCpConfiguration.initServices();
            }
            if (DING_MSG.getCode().equals(entity.getCategory())) {
                // 阿里钉钉多实例配置重新初始化
                dingTalkConfiguration.initModels();
            }
            return ActionResult.success("更新成功");
        } else {
            return ActionResult.fail("更新失败，数据不存在");
        }
    }


    /**
     * 删除
     *
     * @param id
     * @return
     */
    @DeleteMapping("/{id}")
    @DSTransactional
    public ActionResult delete(@PathVariable("id") String id) {
        McMsgAccountEntity entity = mcMsgAccountService.getInfo(id);
        if (ObjectUtil.isNotNull(entity)) {
            if (mcMsgSendTemplateService.getCountByAccountConfigId(entity.getId()) > 0) {
                return ActionResult.fail("发送配置已关联账号，不允许删除！");
            }
            mcMsgAccountService.delete(entity);

        }
        return ActionResult.success("删除成功");
    }

    /**
     * 消息账号配置下拉框
     *
     * @return
     */
    @Operation(summary="消息账号配置下拉框")
    @GetMapping("/Selector")
    public ActionResult<PageListVO<McMsgAccountSelector>> selector(McMsgAccountPagination pagination) {
        List<McMsgAccountEntity> list = mcMsgAccountService.getList(pagination);
        List<McMsgAccountSelector> listSelector = JsonUtil.getJsonToList(list, McMsgAccountSelector.class);
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(listSelector, paginationVO);
    }


}
