package com.bringspring.system.permission.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONArray;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.common.annotation.HandleLog;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.GlobalMarkEnum;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.util.treeutil.ListToTreeUtil;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.DictionaryDataEntity;
import com.bringspring.system.base.exception.BaseException;
import com.bringspring.system.base.service.DictionaryDataService;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.system.permission.constant.PermissionConst;
import com.bringspring.system.permission.entity.*;
import com.bringspring.system.permission.exception.PermissionException;
import com.bringspring.system.permission.model.role.*;
import com.bringspring.system.permission.service.*;
import com.bringspring.system.permission.util.PermissionUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 角色管理
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Tag(name = "角色管理", description = "Role")
@RestController
@RequestMapping("/api/permission/Role")
public class RoleController {

    @Autowired
    private RoleService roleService;
    @Autowired
    private UserService userService;
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private UserRelationService userRelationService;
    @Autowired
    private OrganizeRelationService organizeRelationService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    private OrganizeAdministratorService organizeAdministratorService;
    @Autowired
    private DictionaryDataService dictionaryDataService;
    @Autowired
    private PositionService positionService;
    @Autowired
    private BasePositionRelationService positionRelationService;

    /**
     * 获取角色列表
     *
     * @param pagination
     * @return
     */
    @Operation(summary = "获取角色列表")
    @GetMapping
    public ActionResult list(RolePagination pagination) {

        List<RoleEntity> list = roleService.getListByDataPermission(pagination);
        List<RoleListVO> listVO = new ArrayList<>();
        for (RoleEntity entity : list) {
            // 角色类型展示
            RoleListVO vo = JsonUtil.getJsonToBean(entity, RoleListVO.class);

            if (StringUtils.isNotEmpty(entity.getType())){
//                保存角色类型名称
                String type = "";
                String[] split = entity.getType().split(",");
                for (int i=0;i<split.length;i++){
                    DictionaryDataEntity swapInfo = dictionaryDataService.getSwapInfo(split[i], "4501f6f26a384757bce12d4c4b03342c");
                    if (ObjectUtil.isNotNull(swapInfo) && StringUtils.isNotEmpty(swapInfo.getFullName())) {
                        type+=swapInfo.getFullName()+"，";
                    }
                }
                // 去掉最后一个逗号
                vo.setRoleType(type.substring(0, type.length() - 1));
            }

            if (ObjectUtil.isNotEmpty(entity) && entity.getGlobalMark().equals(GlobalMarkEnum.NOT_ORGANIZE.getCode())) {
                vo.setType("全局");
            } else {
                vo.setType("组织");
                QueryWrapper<OrganizeRelationEntity> query = new QueryWrapper<>();
                query.lambda().eq(OrganizeRelationEntity::getObjectType, PermissionConst.ROLE);
                query.lambda().eq(OrganizeRelationEntity::getObjectId, entity.getId());
                List<String> ids = new ArrayList<>();
                for (OrganizeRelationEntity relation : organizeRelationService.list(query)) {
                    ids.add(relation.getOrganizeId());
                }
                String orgInfos = PermissionUtil.getLinkInfoByOrgId(ids, organizeService);
                if (orgInfos.length() > 0) {
                    vo.setOrganizeInfo(orgInfos.substring(0, orgInfos.length() - 1));
                } else {
                    vo.setOrganizeInfo("");
                }

            }
            listVO.add(vo);
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(listVO, paginationVO);
    }

    /**
     * 角色下拉框列表
     *
     * @return
     */
    @Operation(summary = "角色下拉框列表")
    @GetMapping("/Selector")
    public ActionResult<ListVO<RoleSelectorVO>> listAll() {
        List<RoleEntity> list1 = roleService.getList();
        List<OrganizeEntity> list2 = organizeService.getList();

        List<RoleModel> roleModels = new ArrayList<>();
        List<RoleSelectorVO> modelList = new ArrayList<>();
        // 如果是管理员需要展示全局角色
        if (userProvider.get().getIsAdministrator()) {
            // 全局展示
            RoleSelectorVO globalParent = new RoleSelectorVO();
            String globalParentId = "1";
            globalParent.setFullName("全局");
            globalParent.setId("1");
            globalParent.setId(globalParentId);
            globalParent.setOnlyId("organizeList_0");
            List<RoleSelectorVO> globalModelList = JsonUtil.getJsonToList(roleService.getGlobalList(), RoleSelectorVO.class);
            globalModelList.forEach(g -> {
                g.setType("role");
                g.setOnlyId(UUID.randomUUID().toString());
                g.setParentId(globalParentId);
            });
            globalParent.setHasChildren(globalModelList.size() > 0);
            globalParent.setChildren(globalModelList);
            globalParent.setIsLeaf(false);
            globalParent.setIcon("icon-ym icon-ym-global-role");
            modelList.add(globalParent);
        }

        for (RoleEntity roleEntity : list1) {
            List<OrganizeRelationEntity> relationListByObjectIdAndType = organizeRelationService.getRelationListByObjectIdAndType(PermissionConst.ROLE, roleEntity.getId());
            for (OrganizeRelationEntity entity : relationListByObjectIdAndType) {
                RoleModel roleVo = JsonUtil.getJsonToBean(roleEntity, RoleModel.class);
                // 必须加这个标识，不然前端会报错
                roleVo.setType("role");
                roleVo.setOnlyId(UUID.randomUUID().toString());
                roleVo.setParentId(entity.getOrganizeId());
                roleModels.add(roleVo);
            }
        }
        List<RoleModel> orgList = new ArrayList<>(16);
        list2.stream().forEach(org -> {
            RoleModel orgVo = JsonUtil.getJsonToBean(org, RoleModel.class);
            orgVo.setType(org.getCategory());
            orgVo.setIcon(StringUtils.isNotEmpty(org.getCategory()) ? "company".equals(org.getCategory()) ? "icon-ym icon-ym-tree-organization3" : "icon-ym icon-ym-tree-department1" : "");
            orgVo.setOnlyId(UUID.randomUUID().toString());
            orgList.add(orgVo);
        });

        JSONArray objects = ListToTreeUtil.treeWhere(roleModels, orgList);
        List<RoleModel> jsonToList = JsonUtil.getJsonToList(objects, RoleModel.class);

        List<RoleModel> list = new ArrayList<>(16);
        // 得到角色的值
        List<RoleModel> collect = jsonToList.stream().filter(t -> "role".equals(t.getType())).sorted(Comparator.comparing(RoleModel::getSortCode)).collect(Collectors.toList());
        list.addAll(collect);
        jsonToList.removeAll(collect);
        List<RoleModel> collect1 = jsonToList.stream().sorted(Comparator.comparing(RoleModel::getSortCode).thenComparing(RoleModel::getCreatorTime, Comparator.reverseOrder())).collect(Collectors.toList());
        list.addAll(collect1);

        List<SumTree<RoleModel>> trees = TreeDotUtils.convertListToTreeDot(list);
        modelList.addAll(JsonUtil.getJsonToList(trees, RoleSelectorVO.class));
        ListVO vo = new ListVO();
        vo.setList(modelList);
        return ActionResult.success(vo);
    }

    /**
     * 分级管理下角色下拉框列表
     *
     * @return
     */
    @Operation(summary = "分级管理下角色下拉框列表")
    @GetMapping("/SelectorByPermission")
    public ActionResult<ListVO<RoleSelectorVO>> roleListAll() {
        boolean isAdministrator = userProvider.get().getIsAdministrator();
        List<RoleEntity> list1 = new ArrayList<>(16);
        List<OrganizeEntity> list2 = organizeService.getList();

        List<RoleModel> roleModels = new ArrayList<>();
        List<RoleSelectorVO> modelList = new ArrayList<>();
        // 如果是管理员需要展示全局角色
        if (isAdministrator) {
            // 全局展示
            RoleSelectorVO globalParent = new RoleSelectorVO();
            String globalParentId = "global";
            globalParent.setFullName("全局");
            globalParent.setId(globalParentId);
            globalParent.setOnlyId("organizeList_0");
            List<RoleSelectorVO> globalModelList = JsonUtil.getJsonToList(roleService.getGlobalList(), RoleSelectorVO.class);
            globalModelList.forEach(g -> {
                g.setType("role");
                g.setOnlyId(UUID.randomUUID().toString());
                g.setParentId(globalParentId);
            });
            globalParent.setHasChildren(globalModelList.size() > 0);
            globalParent.setChildren(globalModelList);
            globalParent.setIsLeaf(false);
            globalParent.setIcon("icon-ym icon-ym-global-role");
            modelList.add(globalParent);
        }

        if (!isAdministrator) {
            Set<String> set = new HashSet<>(16);
            // 获取用户分级管理的权限
            List<OrganizeAdministratorEntity> organizeAdministratorEntity = organizeAdministratorService.getOrganizeAdministratorEntity(userProvider.get().getUserId());
            organizeAdministratorEntity.stream().forEach(t -> {
                // 如果有本层编辑权限
                if ("1".equals(String.valueOf(t.getThisLayerEdit()))) {
                    set.add(t.getOrganizeId());
                }
                if ("1".equals(String.valueOf(t.getSubLayerEdit()))) {
                    // 如果有子层编辑权限
                    List<String> underOrganizations = organizeService.getUnderOrganizations(t.getOrganizeId());
                    set.addAll(underOrganizations);
                }
            });
            List<OrganizeEntity> finalOrganizeEntities = new ArrayList<>(16);
            set.forEach(t -> {
                OrganizeEntity entity = organizeService.getInfo(t);
                if (entity != null) {
                    finalOrganizeEntities.add(entity);
                }
            });
            List<OrganizeRelationEntity> relationListByOrganizeId = organizeRelationService.getRelationListByOrganizeId(finalOrganizeEntities.stream().map(OrganizeEntity::getId).collect(Collectors.toList()));
            list1 = roleService.getListByIds(relationListByOrganizeId.stream().filter(t -> PermissionConst.ROLE.equals(t.getObjectType())).map(OrganizeRelationEntity::getObjectId).collect(Collectors.toList()));
        } else {
            list1 = roleService.getList();
        }

        for (RoleEntity roleEntity : list1) {
            List<OrganizeRelationEntity> relationListByObjectIdAndType = organizeRelationService.getRelationListByObjectIdAndType(PermissionConst.ROLE, roleEntity.getId());
            for (OrganizeRelationEntity entity : relationListByObjectIdAndType) {
                RoleModel roleVo = JsonUtil.getJsonToBean(roleEntity, RoleModel.class);
                // 必须加这个标识，不然前端会报错
                roleVo.setType("role");
                roleVo.setOnlyId(UUID.randomUUID().toString());
                roleVo.setParentId(entity.getOrganizeId());
                roleModels.add(roleVo);
            }
        }
        List<RoleModel> orgList = new ArrayList<>(16);
        list2.stream().forEach(org -> {
            RoleModel orgVo = JsonUtil.getJsonToBean(org, RoleModel.class);
            orgVo.setType(org.getCategory());
            orgVo.setIcon(StringUtils.isNotEmpty(org.getCategory()) ? "company".equals(org.getCategory()) ? "icon-ym icon-ym-tree-organization3" : "icon-ym icon-ym-tree-department1" : "");
            orgVo.setOnlyId(UUID.randomUUID().toString());
            orgList.add(orgVo);
        });

        JSONArray objects = ListToTreeUtil.treeWhere(roleModels, orgList);
        List<RoleModel> jsonToList = JsonUtil.getJsonToList(objects, RoleModel.class);

        List<RoleModel> list = new ArrayList<>(16);
        // 得到角色的值
        List<RoleModel> collect = jsonToList.stream().filter(t -> "role".equals(t.getType())).sorted(Comparator.comparing(RoleModel::getSortCode)).collect(Collectors.toList());
        list.addAll(collect);
        jsonToList.removeAll(collect);
        List<RoleModel> collect1 = jsonToList.stream().sorted(Comparator.comparing(RoleModel::getSortCode).thenComparing(RoleModel::getCreatorTime, Comparator.reverseOrder())).collect(Collectors.toList());
        list.addAll(collect1);

        List<SumTree<RoleModel>> trees = TreeDotUtils.convertListToTreeDot(list);
//        trees.removeAll(trees.stream().filter(t->!"-1".equals(t.getParentId())).collect(Collectors.toList()));
        modelList.addAll(JsonUtil.getJsonToList(trees, RoleSelectorVO.class));
        ListVO vo = new ListVO();
        vo.setList(modelList);
        return ActionResult.success(vo);
    }


    /**
     * 权限管理菜单，选择角色获取数据，有数据权限
     *
     * @return
     */
    @Operation(summary = "分级管理下角色下拉框列表")
    @GetMapping("/SelectorRoleByPermission/{menuId}")
    public ActionResult<ListVO<RoleSelectorVO>> roleListPermissionAll(@PathVariable("menuId") String menuId) {
        boolean isAdministrator = userProvider.get().getIsAdministrator();
        List<RoleEntity> list1 = new ArrayList<>(16);
        List<OrganizeEntity> list2 = organizeService.getList();

        List<RoleModel> roleModels = new ArrayList<>();
        List<RoleSelectorVO> modelList = new ArrayList<>();
        // 如果是管理员需要展示全局角色
        if (isAdministrator) {
            // 全局展示
            RoleSelectorVO globalParent = new RoleSelectorVO();
            String globalParentId = "global";
            globalParent.setFullName("全局");
            globalParent.setId(globalParentId);
            globalParent.setOnlyId("organizeList_0");
            List<RoleSelectorVO> globalModelList = JsonUtil.getJsonToList(roleService.getGlobalList(), RoleSelectorVO.class);
            globalModelList.forEach(g -> {
                g.setType("role");
                g.setOnlyId(UUID.randomUUID().toString());
                g.setParentId(globalParentId);
            });
            globalParent.setHasChildren(globalModelList.size() > 0);
            globalParent.setChildren(globalModelList);
            globalParent.setIsLeaf(false);
            globalParent.setIcon("icon-ym icon-ym-global-role");
            modelList.add(globalParent);
        }

        if (!isAdministrator) {
            //根据数据权限获取公司列表
            List<OrganizeEntity> companyList = organizeService.getCompanyList(menuId);
            //获取公司下部门的id
            List<OrganizeEntity> parentIdList = organizeService.getCompanyById(companyList.get(0).getId());
            List<OrganizeRelationEntity> relationListByOrganizeId = organizeRelationService.getRelationListByOrganizeId(parentIdList.stream().map(OrganizeEntity::getId).collect(Collectors.toList()));
            list1 = roleService.getListByIds(relationListByOrganizeId.stream().filter(t -> PermissionConst.ROLE.equals(t.getObjectType())).map(OrganizeRelationEntity::getObjectId).collect(Collectors.toList()));
        } else {
            list1 = roleService.getList();
        }

        for (RoleEntity roleEntity : list1) {
            List<OrganizeRelationEntity> relationListByObjectIdAndType = organizeRelationService.getRelationListByObjectIdAndType(PermissionConst.ROLE, roleEntity.getId());
            for (OrganizeRelationEntity entity : relationListByObjectIdAndType) {
                RoleModel roleVo = JsonUtil.getJsonToBean(roleEntity, RoleModel.class);
                // 必须加这个标识，不然前端会报错
                roleVo.setType("role");
                roleVo.setOnlyId(UUID.randomUUID().toString());
                roleVo.setParentId(entity.getOrganizeId());
                roleModels.add(roleVo);
            }
        }
        List<RoleModel> orgList = new ArrayList<>(16);
        list2.stream().forEach(org -> {
            RoleModel orgVo = JsonUtil.getJsonToBean(org, RoleModel.class);
            orgVo.setType(org.getCategory());
            orgVo.setIcon(StringUtils.isNotEmpty(org.getCategory()) ? "company".equals(org.getCategory()) ? "icon-ym icon-ym-tree-organization3" : "icon-ym icon-ym-tree-department1" : "");
            orgVo.setOnlyId(UUID.randomUUID().toString());
            orgList.add(orgVo);
        });

        JSONArray objects = ListToTreeUtil.treeWhere(roleModels, orgList);
        List<RoleModel> jsonToList = JsonUtil.getJsonToList(objects, RoleModel.class);

        List<RoleModel> list = new ArrayList<>(16);
        // 得到角色的值
        List<RoleModel> collect = jsonToList.stream().filter(t -> "role".equals(t.getType())).sorted(Comparator.comparing(RoleModel::getSortCode)).collect(Collectors.toList());
        list.addAll(collect);
        jsonToList.removeAll(collect);
        List<RoleModel> collect1 = jsonToList.stream().sorted(Comparator.comparing(RoleModel::getSortCode).thenComparing(RoleModel::getCreatorTime, Comparator.reverseOrder())).collect(Collectors.toList());
        list.addAll(collect1);

        List<SumTree<RoleModel>> trees = TreeDotUtils.convertListToTreeDot(list);
//        trees.removeAll(trees.stream().filter(t->!"-1".equals(t.getParentId())).collect(Collectors.toList()));
        modelList.addAll(JsonUtil.getJsonToList(trees, RoleSelectorVO.class));
        ListVO vo = new ListVO();
        vo.setList(modelList);
        return ActionResult.success(vo);
    }

    /**
     * 获取角色信息
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "获取角色信息")
    @GetMapping("/{id}")
    public ActionResult<RoleInfoVO> getInfo(@PathVariable("id") String id) throws DataException {
        RoleEntity entity = roleService.getInfo(id);
        RoleInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, RoleInfoVO.class);
        // 通过组织角色关联表获取组织
        QueryWrapper<OrganizeRelationEntity> query = new QueryWrapper<>();
        query.lambda().eq(OrganizeRelationEntity::getObjectType, PermissionConst.ROLE);
        query.lambda().eq(OrganizeRelationEntity::getObjectId, id);

        List<String> ids = new ArrayList<>();
        organizeRelationService.list(query).forEach(relation -> {
            ids.add(relation.getOrganizeId());
        });
        vo.setOrganizeIdsTree(PermissionUtil.getOrgIdsTree(ids, 1, organizeService));
        return ActionResult.success(vo);
    }


    /**
     * 新建角色
     *
     * @param roleCrForm
     * @return
     */
    @Operation(summary = "新建角色")
    @PostMapping
    @DSTransactional
    @HandleLog(moduleName = "角色管理",requestMethod = "新建角色")
    public ActionResult<String> create(@RequestBody @Valid RoleCrForm roleCrForm) {

        RoleEntity entity = JsonUtil.getJsonToBean(roleCrForm, RoleEntity.class);
        if (roleService.isExistByFullName(roleCrForm.getFullName(), entity.getId())) {
            return ActionResult.fail("角色名称不能重复");
        }
        if (roleService.isExistByEnCode(roleCrForm.getEnCode(), entity.getId())) {
            return ActionResult.fail("角色编码不能重复");
        }

        // 角色的类型是啥，放organizeId，全局角色设置
        String roleId = RandomUtil.uuId();
        entity.setId(roleId);
        roleService.createOrganizeRoleRelation(roleCrForm.getOrganizeIdsTree(), roleId, roleCrForm.getGlobalMark());
        roleService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新角色
     *
     * @param id         主键值
     * @param roleUpForm
     * @return
     */
    @Operation(summary = "更新角色")
    @PutMapping("/{id}")
    @HandleLog(moduleName = "角色管理",requestMethod = "更新角色")
    public ActionResult<String> update(@RequestBody @Valid RoleUpForm roleUpForm, @PathVariable("id") String id) throws DataException, PermissionException {

        boolean flag = roleService.update(id, roleUpForm);
        if (!flag) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }
    /**
     * 更新角色
     *
     * @param id         主键值
     * @param roleUpForm
     * @return
     */
    @Operation(summary = "更新角色验证")
    @PutMapping("/updateVerify/{id}")
    public ActionResult<Integer> updateVerify(@RequestBody @Valid RoleUpForm roleUpForm, @PathVariable("id") String id) throws DataException, PermissionException {
        Integer errorCount = roleService.updateVerify(id, roleUpForm);
        if (errorCount > 0) {
            return ActionResult.success(errorCount);
        }else{
            return ActionResult.success(MsgCode.SU004.get());
        }
    }
    /**
     * 复制角色
     * 复制角色基本数据以及菜单权限数据
     *
     * @param id         主键值
     * @param roleUpForm
     * @return
     */
    @Operation(summary = "复制角色")
    @PostMapping("/copy/{id}")
    @DSTransactional
    @HandleLog(moduleName = "角色管理",requestMethod = "复制角色")
    public ActionResult<String> copyRole(@RequestBody @Valid RoleUpForm roleUpForm, @PathVariable("id") String id) throws BaseException {
        roleService.copyRole(roleUpForm, id);
        return ActionResult.success(MsgCode.SU004.get());
    }


    /**
     * 删除角色
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "删除角色")
    @DeleteMapping("/{id}")
    @HandleLog(moduleName = "角色管理",requestMethod = "删除角色")
    public ActionResult<String> delete(@PathVariable("id") String id) {
        // 当角色绑定用户不让其删除
        if (userRelationService.existByObjectId(id, PermissionConst.ROLE)) {
            return ActionResult.fail(MsgCode.FA024.get());
        }
        RoleEntity entity = roleService.getInfo(id);
        if (entity != null) {
            List<UserRelationEntity> userRelList = userRelationService.getListByObjectId(id);
            if (userRelList.size() > 0) {
                return ActionResult.fail("该角色下有数据权限");
            }
            for (UserRelationEntity entity1 : userRelList) {
                UserEntity entity2 = userService.getById(entity1.getUserId());
                if (entity2 != null) {
                    String newRoleId = entity2.getRoleId().replace(id, "");
                    if (entity2.getRoleId().contains(id)) {
                        if (newRoleId.length() != 0 && newRoleId.substring(0, 1) == ",") {
                            entity2.setRoleId(newRoleId.substring(1));
                        } else if (newRoleId.length() != 0) {
                            entity2.setRoleId(newRoleId.replace(",,", ","));
                        }
                    }
                }
            }
            // 删除所有用户角色关联
            userRelationService.deleteAllByObjId(id);
            // 删除所有组织角色关联
            organizeRelationService.deleteAllByObjectId(id, PermissionConst.ROLE);
            roleService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    /**
     * 更新角色状态
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "更新角色状态")
    @PutMapping("/{id}/Actions/State")
    @HandleLog(moduleName = "角色管理",requestMethod = "更新角色状态")
    public ActionResult<String> disable(@PathVariable("id") String id) {
        RoleEntity entity = roleService.getInfo(id);
        if (entity != null) {
            if ("1".equals(String.valueOf(entity.getEnabledMark()))) {
                entity.setEnabledMark(0);
            } else {
                entity.setEnabledMark(1);
            }
            roleService.update(entity);
            return ActionResult.success(MsgCode.SU005.get());
        }
        return ActionResult.fail(MsgCode.FA007.get());
    }

    /**
     * 通过组织id获取岗位列表
     *
     * @param organizeIds 组织id数组
     * @return 角色列表
     */
    @Operation(summary = "获取角色列表通过组织id数组")
    @PostMapping("/getListByOrgIds")
    public ActionResult<ListVO<RoleModel>> getListByOrganizeIds(@RequestBody @Valid Map<String, List<String>> organizeIds) {
        List<RoleModel> modelAll = new ArrayList<>();

        // 获取全局角色
        List<SumTree<RoleModel>> models = new ArrayList<>();
        roleService.getGlobalList().forEach(r -> {
            models.add(JsonUtil.getJsonToBean(r, RoleModel.class));
        });
        RoleModel modelGlobal = new RoleModel();
        modelGlobal.setFullName("全局");
        modelGlobal.setHasChildren(true);
        modelGlobal.setId("0");
        modelGlobal.setChildren(models);
        modelAll.add(modelGlobal);
        List<String> organizeIdList = organizeIds.get("organizeIds");
        if (CollectionUtil.isNotEmpty(organizeIdList)) {
            for (String organizeId : organizeIds.get("organizeIds")) {
                OrganizeEntity organizeEntity = organizeService.getInfo(organizeId);
                if (ObjectUtil.isNotEmpty(organizeEntity)) {
                    RoleModel organizeModel = JsonUtil.getJsonToBean(organizeEntity, RoleModel.class);
                    List<OrganizeRelationEntity> roleRelations = organizeRelationService.getListByTypeAndOrgId(PermissionConst.ROLE, organizeId);
                    List<SumTree<RoleModel>> roleList = new ArrayList<>();
                    for (OrganizeRelationEntity roleRelation : roleRelations) {
                        RoleEntity roleEntity = roleService.getInfo(roleRelation.getObjectId());
                        // 非全局
                        if (StringUtils.isNotNull(roleEntity) && roleEntity.getGlobalMark() == GlobalMarkEnum.ORGANIZE.getCode()) {
                            RoleModel roleModel = JsonUtil.getJsonToBean(roleService.getInfo(roleRelation.getObjectId()), RoleModel.class);
                            roleModel.setHasChildren(false);
                            roleList.add(roleModel);
                        }
                    }
                    organizeModel.setHasChildren(true);
                    organizeModel.setChildren(roleList);
                    modelAll.add(organizeModel);
                }
            }
        }
        ListVO<RoleModel> vo = new ListVO<>();
        vo.setList(modelAll);
        return ActionResult.success(vo);
    }


    /**
     * 获取数据权限岗位管理信息列表
     *
     * @param paginationPosition
     * @return
     */
    @Operation(summary = "获取岗位列表（分页）")
    @GetMapping("/roleDataPermission")
    public ActionResult<PageListVO<RoleListVO>> listDataPermission(RolePagination paginationPosition) {
        List<OrganizeEntity> list = organizeService.getList().stream().filter(t -> "1".equals(String.valueOf(t.getEnabledMark()))).collect(Collectors.toList());
//        获取数据权限方法
        List<RoleEntity> data = roleService.getListPositionDataPermission(paginationPosition);

        //添加部门信息，部门映射到organizeId
        List<RoleListVO> voList = new ArrayList<>();

        for (RoleEntity entity : data) {
            // 角色类型展示
            RoleListVO vo = JsonUtil.getJsonToBean(entity, RoleListVO.class);

            DictionaryDataEntity swapInfo = dictionaryDataService.getSwapInfo(entity.getType(), "4501f6f26a384757bce12d4c4b03342c");
            if (ObjectUtil.isNotNull(swapInfo) && StringUtils.isNotEmpty(swapInfo.getFullName())) {
                vo.setRoleType(swapInfo.getFullName());
            }

            if (ObjectUtil.isNotEmpty(entity) && entity.getGlobalMark().equals(GlobalMarkEnum.NOT_ORGANIZE.getCode())) {
                vo.setType("全局");
            } else {
                vo.setType("组织");
                QueryWrapper<OrganizeRelationEntity> query = new QueryWrapper<>();
                query.lambda().eq(OrganizeRelationEntity::getObjectType, PermissionConst.ROLE);
                query.lambda().eq(OrganizeRelationEntity::getObjectId, entity.getId());
                List<String> ids = new ArrayList<>();
                for (OrganizeRelationEntity relation : organizeRelationService.list(query)) {
                    ids.add(relation.getOrganizeId());
                }
                String orgInfos = PermissionUtil.getLinkInfoByOrgId(ids, organizeService);
                if (orgInfos.length() > 0) {
                    vo.setOrganizeInfo(orgInfos.substring(0, orgInfos.length() - 1));
                } else {
                    vo.setOrganizeInfo("");
                }

            }
            voList.add(vo);
        }
        //添加部门信息
//        for (PositionListVO entity1 : voList) {
//            OrganizeEntity entity = list.stream().filter(t -> t.getId().equals(entity1.getOrganizeId())).findFirst().orElse(new OrganizeEntity());
//            if (entity1.getOrganizeId().equals(entity.getId())) {
//                entity1.setDepartment(entity.getFullName());
//            }
//            entity1.setDepartment(PermissionUtil.getLinkInfoByOrgId(entity1.getOrganizeId(), organizeService, false));
//        }
//        //将type成中文名
//        List<DictionaryDataEntity> dictionaryDataEntities = dictionaryDataService.getList("dae93f2fd7cd4df999d32f8750fa6a1e", false);
//        for (PositionListVO entity1 : voList) {
//            DictionaryDataEntity entity = dictionaryDataEntities.stream().filter(t -> t.getEnCode().equals(entity1.getType())).findFirst().orElse(null);
//            if (entity != null) {
//                entity1.setType(entity.getFullName());
//            }
//        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(paginationPosition, PaginationVO.class);
        return ActionResult.page(voList, paginationVO);
    }

    /**
     * 通过组织id获取岗位列表
     *
     * @param positionIds 组织id数组
     * @return 角色列表
     */
    @Operation(summary = "获取角色列表通过组织id数组")
    @PostMapping("/getListByPositionIds")
    public ActionResult<ListVO<RoleModel>> getListByPositionIds(@RequestBody @Valid Map<String, List<String>> positionIds) {
        List<RoleModel> modelAll = new ArrayList<>();

        // 获取全局角色
        List<SumTree<RoleModel>> models = new ArrayList<>();
        roleService.getGlobalList().forEach(r -> {
            models.add(JsonUtil.getJsonToBean(r, RoleModel.class));
        });
        RoleModel modelGlobal = new RoleModel();
        modelGlobal.setFullName("全局");
        modelGlobal.setHasChildren(true);
        modelGlobal.setId("0");
        modelGlobal.setChildren(models);
        modelAll.add(modelGlobal);
        List<String> organizeIdList = positionIds.get("positionIds");
        if (CollectionUtil.isNotEmpty(organizeIdList)) {
            for (String positionId : positionIds.get("positionIds")) {
                PositionEntity positionEntity = positionService.getInfo(positionId);
                if (ObjectUtil.isNotEmpty(positionEntity)) {
                    RoleModel organizeModel = JsonUtil.getJsonToBean(positionEntity, RoleModel.class);
                    List<BasePositionRelationEntity> posRelations = positionRelationService.getListByTypeAndPosId(PermissionConst.ROLE, positionId);
                    List<SumTree<RoleModel>> roleList = new ArrayList<>();
                    for (BasePositionRelationEntity roleRelation : posRelations) {
                        RoleEntity roleEntity = roleService.getInfo(roleRelation.getObjectId());
                        // 非全局
                        if (roleEntity.getGlobalMark() == GlobalMarkEnum.ORGANIZE.getCode()) {
                            RoleModel roleModel = JsonUtil.getJsonToBean(roleService.getInfo(roleRelation.getObjectId()), RoleModel.class);
                            roleModel.setHasChildren(false);
                            roleList.add(roleModel);
                        }
                    }
                    organizeModel.setHasChildren(true);
                    organizeModel.setChildren(roleList);
                    modelAll.add(organizeModel);
                }
            }
        }
        ListVO<RoleModel> vo = new ListVO<>();
        vo.setList(modelAll);
        return ActionResult.success(vo);
    }

    /**
     * 根据菜单id获取角色列表
     *
     * @param menuId 菜单主键值
     * @return
     */
    @Operation(summary = "根据菜单id获取角色列表信息")
    @GetMapping("/getListBymenuId/{menuId}")
    public ActionResult<List<RoleListVO>> getListBymenuId(@PathVariable("menuId") String menuId) throws DataException {
        List<AuthorizeEntity> authorizeEntityList = roleService.getListBymenuId(menuId);
        UserInfo userInfo = userProvider.get();
        List<RoleEntity> list = new ArrayList<>();
        for (AuthorizeEntity authorizeEntity : authorizeEntityList) {
            RoleEntity entity = roleService.getInfo(authorizeEntity.getObjectId());
            if (StringUtils.isNotNull(entity)) {
                list.add(entity);
            }

        }
        List<RoleListVO> listVO = new ArrayList<>();
        for (int i =0 ; i< list.size();i++) {
            // 角色类型展示
            RoleListVO vo = JsonUtil.getJsonToBean(list.get(i), RoleListVO.class);
            if (list.get(i).getGlobalMark() != null && list.get(i).getGlobalMark().equals(GlobalMarkEnum.NOT_ORGANIZE.getCode())) {
                vo.setType("全局");
                listVO.add(vo);
            } else {
                vo.setType("组织");
                QueryWrapper<OrganizeRelationEntity> query = new QueryWrapper<>();
                query.lambda().eq(OrganizeRelationEntity::getObjectType, PermissionConst.ROLE);
                query.lambda().eq(OrganizeRelationEntity::getObjectId, list.get(i).getId());
                List<String> ids = new ArrayList<>();
                for (OrganizeRelationEntity relation : organizeRelationService.list(query)) {
                    OrganizeEntity info = organizeService.getInfo(relation.getOrganizeId());
                    if(StringUtils.isNotNull(info)){
                        if((userInfo.getIsAdministrator()) || info.getId().equals(userInfo.getOrganizeId()) || info.getParentId().equals(userInfo.getOrganizeId()))
                        {
                            ids.add(relation.getOrganizeId());
                        }else{
                            continue;
                        }
                    }

                }
                String orgInfos = PermissionUtil.getLinkInfoByOrgId(ids, organizeService);
                if (orgInfos.length() > 0) {
                    vo.setOrganizeInfo(orgInfos.substring(0, orgInfos.length() - 1));
                    listVO.add(vo);
                }

            }

        }
        return ActionResult.success(listVO);
    }
}
