package com.bringspring.system.permission.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.*;
import com.bringspring.system.permission.constant.PermissionConst;
import com.bringspring.system.permission.entity.AuthorizeEntity;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.entity.PositionEntity;
import com.bringspring.system.permission.entity.UserRelationEntity;
import com.bringspring.system.permission.mapper.OrganizeMapper;
import com.bringspring.system.permission.mapper.PositionMapper;
import com.bringspring.system.permission.model.authorize.AuthorizeConditionModel;
import com.bringspring.system.permission.model.organize.OrganizePagination;
import com.bringspring.system.permission.model.position.PaginationPosition;
import com.bringspring.system.permission.service.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bringspring.common.constant.CacheConsts.FOREVER_CACHE;

/**
 * 岗位信息
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Service
public class PositionServiceImpl extends ServiceImpl<PositionMapper, PositionEntity> implements PositionService {

    @Autowired
    private AuthorizeService authorizeService;
    @Autowired
    private UserRelationService userRelationService;
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private CacheUtil cacheUtil;
    @Autowired
    private CacheKeyUtil cacheKeyUtil;
    @Autowired
    private OrganizeRelationService organizeRelationService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    private UserService userService;
    @Autowired
    private OrganizeMapper organizeMapper;
    @Autowired
    private PositionService positionService;

    @Override
    public List<PositionEntity> getList() {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public List<PositionEntity> getList(String keyword) {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(keyword)) {
            //通过关键字查询
            queryWrapper.lambda().and(t -> t.like(PositionEntity::getFullName, keyword));
        }
        queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public List<PositionEntity> getListByOrgIds(List<String> orgIds) {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().in(PositionEntity::getOrganizeId, orgIds);
        queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public List<PositionEntity> getListByIds(List<String> ids) {
        List<PositionEntity> list = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(ids)){
            QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(PositionEntity::getId, ids);
            queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
            list = this.list(queryWrapper);
        }
        return list;
    }

    @Override
    public List<PositionEntity> getPositionList(List<String> idList) {
        if (idList.size() > 0) {
            QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(PositionEntity::getId, idList).select(PositionEntity::getId,
                    PositionEntity::getFullName);
            return this.list(queryWrapper);
        }
        return new ArrayList<>();
    }

    @Override
    public List<PositionEntity> getPositionList(Set<String> idList) {
        if (idList.size() > 0) {
            QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
            queryWrapper.lambda().select(PositionEntity::getId, PositionEntity::getFullName).in(PositionEntity::getId
                    , idList);
            return this.list(queryWrapper);
        }
        return new ArrayList<>();
    }


    @Override
    public List<PositionEntity> getPositionRedisList() {
        if (cacheUtil.exists(FOREVER_CACHE, cacheKeyUtil.getPositionList())) {
            Object positionlist = cacheUtil.query(FOREVER_CACHE, cacheKeyUtil.getOrganizeList());
            if (positionlist instanceof String) {
                return JsonUtil.getJsonToList(positionlist.toString(), PositionEntity.class);
            } else {
                return (List<PositionEntity>) positionlist;
            }
        }
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(PositionEntity::getEnabledMark, 1);

        List<PositionEntity> list = this.list(queryWrapper);
        if (list.size() > 0) {
            cacheUtil.insert(FOREVER_CACHE, JsonUtil.getObjectToString(list), 300);
        }
        return list;
    }

    @Override
    public List<PositionEntity> getList(PaginationPosition paginationPosition) {
        // 定义变量判断是否需要使用修改时间倒序
        UserInfo userInfo = userProvider.get();
        boolean flag = false;
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        //组织机构筛选
        if (!StringUtils.isEmpty(paginationPosition.getOrganizeId())) {
            flag = true;
            List<String> childOrganizeIds = organizeService.getUnderOrganizations(paginationPosition.getOrganizeId());
            childOrganizeIds.add(paginationPosition.getOrganizeId());
            if (CollectionUtil.isNotEmpty(childOrganizeIds)) {
                queryWrapper.lambda().in(PositionEntity::getOrganizeId, childOrganizeIds);
            }
        }
        //非管理员只能查看组织角色；只有管理员管理员，显示全局角色
        if (!userInfo.getIsAdministrator()) {
            //查询数据权限组织角色
            List<OrganizeEntity> organizeByDataAuthorize =
                    organizeService.getOrganizeByDataPermission(new OrganizePagination(paginationPosition.getMenuId()));
            List<String> organizeIds =
                    organizeByDataAuthorize.stream().map(OrganizeEntity::getId).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(organizeIds)) {
                //过滤数据权限所属组织
                queryWrapper.lambda().in(PositionEntity::getOrganizeId, organizeIds);
            }
        }
        //关键字（名称、编码）
        if (!StringUtils.isEmpty(paginationPosition.getKeyword())) {
            flag = true;
            queryWrapper.lambda().and(
                    t -> t.like(PositionEntity::getFullName, paginationPosition.getKeyword())
                            .or().like(PositionEntity::getEnCode, paginationPosition.getKeyword())
            );
        }
        //排序
        queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
        if (flag) {
            queryWrapper.lambda().orderByDesc(PositionEntity::getLastModifyTime);
        }
        Page<PositionEntity> page = new Page<>(paginationPosition.getCurrentPage(), paginationPosition.getPageSize());
        IPage<PositionEntity> iPage = this.page(page, queryWrapper);
        return paginationPosition.setData(iPage.getRecords(), page.getTotal());
    }

    @Override
    public List<PositionEntity> getListByUserId(String userId) {
        QueryWrapper<PositionEntity> query = new QueryWrapper<>();
        List<String> ids = new ArrayList<>();
        userRelationService.getListByObjectType(userId, PermissionConst.POSITION).forEach(r -> {
            ids.add(r.getObjectId());
        });
        if (ids.size() > 0) {
            query.lambda().in(PositionEntity::getId, ids);
            return this.list(query);
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    public PositionEntity getInfo(String id) {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(PositionEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    public PositionEntity getByFullName(String fullName) {
        PositionEntity positionEntity = new PositionEntity();
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(PositionEntity::getFullName, fullName);
        queryWrapper.lambda().select(PositionEntity::getId);
        List<PositionEntity> list = this.list(queryWrapper);
        if (list.size() > 0) {
            positionEntity = list.get(0);
        }
        return positionEntity;
    }

    @Override
    public boolean isExistByFullName(PositionEntity entity, boolean isFilter) {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        if (entity != null) {
            queryWrapper.lambda().eq(PositionEntity::getFullName, entity.getFullName());
        }
        //是否需要过滤
        if (isFilter) {
            queryWrapper.lambda().ne(PositionEntity::getId, entity.getId());
        }
        List<PositionEntity> entityList = this.list(queryWrapper);
        for (PositionEntity positionEntity : entityList) {
            //如果组织id相同则代表已存在
            if (entity != null && entity.getOrganizeId().equals(positionEntity.getOrganizeId())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean isExistByEnCode(PositionEntity entity, boolean isFilter) {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        if (entity != null) {
            queryWrapper.lambda().eq(PositionEntity::getEnCode, entity.getEnCode());
            if (isFilter) {
                queryWrapper.lambda().ne(PositionEntity::getId, entity.getId());
            }
        }
        List<PositionEntity> entityList = this.list(queryWrapper);
        for (PositionEntity positionEntity : entityList) {
            //如果组织id相同则代表已存在
            if (entity != null && entity.getOrganizeId().equals(positionEntity.getOrganizeId())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void create(PositionEntity entity) {
        if (StringUtils.isEmpty(entity.getId())) {
            entity.setId(RandomUtil.uuId());
        }
        entity.setCreatorUserId(userProvider.get().getUserId());
        this.save(entity);
    }

    @Override
    public boolean update(String id, PositionEntity entity) {
        entity.setId(id);
        entity.setLastModifyTime(new Date());
        entity.setLastModifyUserId(userProvider.get().getUserId());
        return this.updateById(entity);
    }

    @Override
    @DSTransactional
    public void delete(PositionEntity entity) {
        this.removeById(entity.getId());
        QueryWrapper<UserRelationEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(UserRelationEntity::getObjectId, entity.getId());
        userRelationService.remove(queryWrapper);
        QueryWrapper<AuthorizeEntity> wrapper = new QueryWrapper<>();
        wrapper.lambda().eq(AuthorizeEntity::getObjectId, entity.getId());
        authorizeService.remove(wrapper);
    }

    @Override
    @DSTransactional
    public boolean first(String id) {
        boolean isOk = false;
        //获取要上移的那条数据的信息
        PositionEntity upEntity = this.getById(id);
        Long upSortCode = upEntity.getSortCode() == null ? 0 : upEntity.getSortCode();
        //查询上几条记录
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .lt(PositionEntity::getSortCode, upSortCode)
                .eq(PositionEntity::getOrganizeId, upEntity.getOrganizeId())
                .orderByDesc(PositionEntity::getSortCode);
        List<PositionEntity> downEntity = this.list(queryWrapper);
        if (downEntity.size() > 0) {
            //交换两条记录的sort值
            Long temp = upEntity.getSortCode();
            upEntity.setSortCode(downEntity.get(0).getSortCode());
            downEntity.get(0).setSortCode(temp);
            this.updateById(downEntity.get(0));
            this.updateById(upEntity);
            isOk = true;
        }
        return isOk;
    }

    @Override
    @DSTransactional
    public boolean next(String id) {
        boolean isOk = false;
        //获取要下移的那条数据的信息
        PositionEntity downEntity = this.getById(id);
        Long upSortCode = downEntity.getSortCode() == null ? 0 : downEntity.getSortCode();
        //查询下几条记录
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .gt(PositionEntity::getSortCode, upSortCode)
                .eq(PositionEntity::getOrganizeId, downEntity.getOrganizeId())
                .orderByAsc(PositionEntity::getSortCode);
        List<PositionEntity> upEntity = this.list(queryWrapper);
        if (upEntity.size() > 0) {
            //交换两条记录的sort值
            Long temp = downEntity.getSortCode();
            downEntity.setSortCode(upEntity.get(0).getSortCode());
            upEntity.get(0).setSortCode(temp);
            this.updateById(upEntity.get(0));
            this.updateById(downEntity);
            isOk = true;
        }
        return isOk;
    }

    @Override
    public List<PositionEntity> getListPositionDataPermission(PaginationPosition paginationPosition) {
        UserInfo userInfo = userProvider.get();
        boolean flag = false;
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        List<String> orgDataPermissionList = new ArrayList<>();
        List<String> childOrganizeIds = new ArrayList<>();

        //数据权限
        if (!userInfo.getIsAdministrator()) {
            QueryWrapper<OrganizeEntity> queryWrapperOrganizeEntity = new QueryWrapper<>();
            Object companyAuth = authorizeService.getCondition(new AuthorizeConditionModel(queryWrapperOrganizeEntity
                    , paginationPosition.getMenuId(), "base_organize"));
            if (ObjectUtil.isEmpty(companyAuth)) {
                return new ArrayList<>();
            } else {
                queryWrapperOrganizeEntity = (QueryWrapper<OrganizeEntity>) companyAuth;
            }
            //数据权限的公司部门id
            List<OrganizeEntity> organizeEntityList = organizeMapper.selectList(queryWrapperOrganizeEntity);
            orgDataPermissionList =
                    organizeEntityList.stream().map(organizeEntity -> organizeEntity.getId()).collect(Collectors.toList());
        }
        //组织机构
        if (StringUtils.isNotEmpty(paginationPosition.getOrganizeId())) {
            //查询组织及当前组织
            List<String> orgIdList = organizeService.getUnderOrganizations(paginationPosition.getOrganizeId());
            if (!userInfo.getIsAdministrator()) {
                orgIdList.retainAll(orgDataPermissionList);
            }
            List<PositionEntity> list = positionService.getListByOrgIds(orgIdList);
            childOrganizeIds =
                    list.stream().map(PositionEntity -> PositionEntity.getOrganizeId()).collect(Collectors.toList());
            flag = true;
//            childOrganizeIds.add(paginationPosition.getOrganizeId());

            //        查询全部的部门信息
            List<OrganizeEntity> organizeListAll = organizeService.getList();
            //查询部门状态---为禁用的id（要下级部门）
            List<OrganizeEntity> biddenIds =
                    organizeListAll.stream().filter(s -> s.getEnabledMark() == 0).collect(Collectors.toList());
            if (!biddenIds.isEmpty()) { //有值
//                用来保存下级部门数据
                List<OrganizeEntity> parentIdAllList = new ArrayList<>();
                for (OrganizeEntity entity : biddenIds) {
//                查询下级部门
                    List<OrganizeEntity> parentIdList = organizeService.getParentIdList(entity.getId());
                    if (!parentIdList.isEmpty()) {
                        parentIdAllList.addAll(parentIdList);
                    }
                }
                biddenIds.addAll(parentIdAllList);//添加下级部门
                // 部门状态为禁用的数据
                List<String> organizeIds = biddenIds.stream().map(OrganizeEntity::getId).collect(Collectors.toList());
//                删除--部门状态为禁用数据
                childOrganizeIds.removeAll(organizeIds);
            }
            if (CollectionUtil.isNotEmpty(childOrganizeIds)) {
                queryWrapper.lambda().in(PositionEntity::getOrganizeId, childOrganizeIds);
            } else {
//                没有岗位，返回为空
                queryWrapper.lambda().in(PositionEntity::getOrganizeId, "");
            }
        }

        //关键字（名称、编码）
        if (!StringUtils.isEmpty(paginationPosition.getKeyword())) {
            flag = true;
            queryWrapper.lambda().and(
                    t -> t.like(PositionEntity::getFullName, paginationPosition.getKeyword())
                            .or().like(PositionEntity::getEnCode, paginationPosition.getKeyword())
            );
        }
        //排序
        queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
        if (flag) {
            queryWrapper.lambda().orderByDesc(PositionEntity::getLastModifyTime);
        }
        Page<PositionEntity> page = new Page<>(paginationPosition.getCurrentPage(), paginationPosition.getPageSize());
        IPage<PositionEntity> iPage = this.page(page, queryWrapper);
        return paginationPosition.setData(iPage.getRecords(), page.getTotal());
    }

    @Override
    public List<PositionEntity> getPositionName(List<String> id) {
        List<PositionEntity> roleList = new ArrayList<>();
        if (id.size() > 0) {
            QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(PositionEntity::getId, id);
            roleList = this.list(queryWrapper);
        }
        return roleList;
    }

    @Override
    public List<PositionEntity> getListByOrganizeId(String organizeId) {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(PositionEntity::getOrganizeId, organizeId);
        queryWrapper.lambda().orderByAsc(PositionEntity::getSortCode).orderByDesc(PositionEntity::getCreatorTime);
        queryWrapper.lambda().select(PositionEntity::getId, PositionEntity::getFullName);
        return this.list(queryWrapper);
    }

    @Override
    public List<PositionEntity> getListByOrgIdAndUserId(String organizeId, String userId) {
        // 用户绑定的所有岗位
        List<String> positionIds =
                userRelationService.getListByUserIdAndObjType(userId, PermissionConst.POSITION).stream()
                .map(UserRelationEntity::getObjectId).collect(Collectors.toList());
        if (positionIds.size() > 0) {
            List<PositionEntity> positionEntities = this.listByIds(positionIds);
            return positionEntities.stream().filter(p -> p.getOrganizeId().equals(organizeId)).collect(Collectors.toList());
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    public List<PositionEntity> getListByFullName(String fullName, String enCode) {
        QueryWrapper<PositionEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(PositionEntity::getFullName, fullName).eq(PositionEntity::getEnCode, enCode);
        return this.list(queryWrapper);
    }

}
