package com.bringspring.system.permission.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.base.GlobalMarkEnum;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.system.base.exception.BaseException;
import com.bringspring.system.permission.constant.PermissionConst;
import com.bringspring.system.permission.entity.*;
import com.bringspring.system.permission.exception.PermissionException;
import com.bringspring.system.permission.mapper.RoleMapper;
import com.bringspring.system.permission.model.authorize.AuthorizeConditionModel;
import com.bringspring.system.permission.model.organize.OrganizePagination;
import com.bringspring.system.permission.model.role.RolePagination;
import com.bringspring.system.permission.model.role.RoleUpForm;
import com.bringspring.system.permission.service.*;
import com.bringspring.system.permission.util.PermissionUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static com.bringspring.common.base.GlobalMarkEnum.NOT_ORGANIZE;

/**
 * 系统角色
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Service
@DSTransactional
public class RoleServiceImpl extends ServiceImpl<RoleMapper, RoleEntity> implements RoleService {

    @Autowired
    private UserProvider userProvider;
    @Autowired
    private UserService userService;
    @Autowired
    private RoleService roleService;
    @Autowired
    private UserRelationService userRelationService;
    @Autowired
    private AuthorizeService authorizeService;
    @Autowired
    private OrganizeRelationService organizeRelationService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    private BasePositionRelationService positionRelationService;

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getList() {
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().orderByAsc(RoleEntity::getSortCode).orderByDesc(RoleEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getList(String keyword) {
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(keyword)) {
            //通过关键字查询
            queryWrapper.lambda().and(t -> t.like(RoleEntity::getFullName, keyword));
        }
        queryWrapper.lambda().orderByAsc(RoleEntity::getSortCode).orderByDesc(RoleEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getList(RolePagination pagination) {

        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
        // null:全部显示 0：全局
        if (!userProvider.get().getIsAdministrator()) {
            Object obj = queryWrapper;
            queryWrapper =
                    (QueryWrapper<RoleEntity>) authorizeService.getCondition(new AuthorizeConditionModel(queryWrapper
                            , pagination.getMenuId(), "BASE_ROLE"));
        }
        int globalMark;
        if ("0".equals(pagination.getOrganizeId())) {
            // 显示全局
            globalMark = 1;
        } else if (StringUtils.isEmpty(pagination.getOrganizeId())) {
            // 全部显示
            globalMark = -1;
        } else {
            // 显示组织内
            globalMark = 0;
        }
        // 定义变量判断是否需要使用修改时间倒序
        boolean flag = false;
        if (StringUtils.isNotEmpty(pagination.getKeyword())) {
            flag = true;
            queryWrapper.lambda().and(t -> t.like(RoleEntity::getFullName, pagination.getKeyword()).or().like(RoleEntity::getEnCode, pagination.getKeyword()));
        }
        // 组织id查询相关的角色
        if (globalMark == 0) {
            List<OrganizeEntity> listLikeOrgIdTree = organizeService.getListLikeOrgIdTree(pagination.getOrganizeId());

            QueryWrapper<OrganizeRelationEntity> queryRelation = new QueryWrapper<>();
            queryRelation.lambda().eq(OrganizeRelationEntity::getOrganizeId, pagination.getOrganizeId());
            List<String> ids = new ArrayList<>();
            organizeRelationService.list(queryRelation).forEach(o -> {
                ids.add(o.getObjectId());
            });
            if (ids.size() > 0) {
                queryWrapper.lambda().in(RoleEntity::getId, ids);
            } else {
                queryWrapper.lambda().in(RoleEntity::getId, "0");
            }
        }

        if (globalMark != -1) {
            queryWrapper.lambda().eq(RoleEntity::getGlobalMark, globalMark);
        }
        queryWrapper.lambda().orderByAsc(RoleEntity::getSortCode).orderByDesc(RoleEntity::getCreatorTime);
        if (flag) {
            queryWrapper.lambda().orderByDesc(RoleEntity::getLastModifyTime);
        }
        Page<RoleEntity> page = new Page<>(pagination.getCurrentPage(), pagination.getPageSize());
        IPage<RoleEntity> userPage = this.page(page, queryWrapper);
        return pagination.setData(userPage.getRecords(), page.getTotal());
    }

    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getListByDataPermission(RolePagination pagination) {
        UserInfo userInfo = userProvider.get();
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();


        QueryWrapper<OrganizeRelationEntity> queryRelation = new QueryWrapper<>();
        //条件过滤
        if (StringUtils.isNotEmpty(pagination.getOrganizeId())) {
            List<String> childOrganizeIds = organizeService.getUnderOrganizations(pagination.getOrganizeId());
            childOrganizeIds.add(pagination.getOrganizeId());
            queryRelation.lambda().in(CollectionUtil.isNotEmpty(childOrganizeIds),
                    OrganizeRelationEntity::getOrganizeId, childOrganizeIds);
        }
        //全局
        if (StringUtils.isNotEmpty(pagination.getOrganizeId()) && "0".equals(pagination.getOrganizeId())) {
            queryWrapper.lambda().eq(RoleEntity::getGlobalMark, NOT_ORGANIZE.getCode());
        }
        //非管理员只能查看组织角色；只有管理员管理员，显示全局角色
        if (!userInfo.getIsAdministrator()) {
            queryWrapper.lambda().eq(RoleEntity::getGlobalMark, GlobalMarkEnum.ORGANIZE.getCode());
            //查询数据权限组织角色
            List<OrganizeEntity> organizeByDataAuthorize =
                    organizeService.getOrganizeByDataPermission(new OrganizePagination(pagination.getMenuId()));
            List<String> organizeIds =
                    organizeByDataAuthorize.stream().map(OrganizeEntity::getId).collect(Collectors.toList());
            //过滤数据权限所属组织
            if (!CollectionUtil.isEmpty(organizeIds)) { //数据权限不为空
                queryRelation.lambda().in(CollectionUtil.isNotEmpty(organizeIds),
                        OrganizeRelationEntity::getOrganizeId, organizeIds);
            } else {
//                为空 则返回空数据 防止没有权限---但查询出全部的数据
                queryRelation.lambda().in(OrganizeRelationEntity::getId, "");
            }
        }
        //        查询全部的部门信息
        List<OrganizeEntity> organizeListAll = organizeService.getList();
        //查询部门状态---为禁用的id（要下级部门）
        List<OrganizeEntity> biddenIds =
                organizeListAll.stream().filter(s -> s.getEnabledMark() == 0).collect(Collectors.toList());
        if (!biddenIds.isEmpty()) { //有值
//                用来保存下级部门数据
            List<OrganizeEntity> parentIdAllList = new ArrayList<>();
            for (OrganizeEntity entity : biddenIds) {
//                查询下级部门
                List<OrganizeEntity> parentIdList = organizeService.getParentIdList(entity.getId());
                if (!parentIdList.isEmpty()) {
                    parentIdAllList.addAll(parentIdList);
                }
            }
            biddenIds.addAll(parentIdAllList);//添加下级部门
            //        部门状态为禁用的数据
            List<String> organizeIds = biddenIds.stream().map(OrganizeEntity::getId).collect(Collectors.toList());
            queryRelation.lambda().notIn(OrganizeRelationEntity::getOrganizeId, organizeIds);
        }

        List<String> ids = new ArrayList<>();
        organizeRelationService.list(queryRelation).forEach(o -> {
            ids.add(o.getObjectId());
        });
        //如果数据权限和条件过滤完以后部门和角色管理为空，则返回空数据  ; 判断是否为全局 如果是全局 前端返回值是--0
        if (!"0".equals(pagination.getOrganizeId())) {
            if (CollectionUtil.isEmpty(ids)) {
                queryWrapper.lambda().in(RoleEntity::getId, "");
            } else {
//            不为空，当前组织下有角色，查询有值
                queryWrapper.lambda().in(RoleEntity::getId, ids);
            }
        }

        if (StringUtils.isNotEmpty(pagination.getKeyword())) {
            queryWrapper.lambda().and(t -> t.like(RoleEntity::getFullName, pagination.getKeyword()).or().like(RoleEntity::getEnCode, pagination.getKeyword()));
        }
        //过滤角色分类
        if (CollectionUtil.isNotEmpty(pagination.getTypeList())) {
            queryWrapper.lambda().in(RoleEntity::getType, pagination.getTypeList());
        }

        queryWrapper.lambda().orderByAsc(RoleEntity::getSortCode).orderByDesc(RoleEntity::getLastModifyTime);
        Page<RoleEntity> page = new Page<>(pagination.getCurrentPage(), pagination.getPageSize());
        IPage<RoleEntity> userPage = this.page(page, queryWrapper);
        return pagination.setData(userPage.getRecords(), page.getTotal());
    }

    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getListPositionDataPermission(RolePagination rolePagination) {
        UserInfo userInfo = userProvider.get();
        boolean flag = false;
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();

        List<String> childOrganizeIds = new ArrayList<>();
        //组织机构筛选
        if (!StringUtils.isEmpty(rolePagination.getPositionId())) {
//            if ("1".equals(rolePagination.getOrganizeId())){
//                //查询数据权限数据  直属组织
//                if (!userInfo.getIsAdministrator()){
//                    List<OrganizeEntity> list = organizeService.getOrganizeListDataPermission(rolePagination
//                    .getMenuId());
//                    if (!list.isEmpty()){
//                        for (OrganizeEntity organizeEntity:list){
//                            if (organizeEntity.getCategory().equals("company")){
//                                childOrganizeIds.add(organizeEntity.getOrganizeId());
//                            }
//                        }
//                        if(CollectionUtil.isNotEmpty(childOrganizeIds)){
//                            queryWrapper.lambda().in(RoleEntity::getOrganizeId,childOrganizeIds);
//                        }
//                    }
//                }
//            }
            flag = true;
//            childOrganizeIds = positionRelationService.getListByTypeAndPosId(PermissionConst.ROLE, rolePagination
//            .getOrganizeId());
            childOrganizeIds = positionRelationService.getListByTypeAndPosId(PermissionConst.ROLE,
                    rolePagination.getPositionId()).stream().map(v -> v.getObjectId()).collect(Collectors.toList());
            childOrganizeIds.add(rolePagination.getOrganizeId());
            if (CollectionUtil.isNotEmpty(childOrganizeIds)) {
                queryWrapper.lambda().in(RoleEntity::getId, childOrganizeIds);
            }
        }
        //关键字（名称、编码）
        if (!StringUtils.isEmpty(rolePagination.getKeyword())) {
            flag = true;
            queryWrapper.lambda().and(t -> t.like(RoleEntity::getFullName, rolePagination.getKeyword()).or().like(RoleEntity::getEnCode, rolePagination.getKeyword()));
        }
        //排序
        queryWrapper.lambda().orderByAsc(RoleEntity::getSortCode).orderByDesc(RoleEntity::getCreatorTime);
        if (flag) {
            queryWrapper.lambda().orderByDesc(RoleEntity::getLastModifyTime);
        }
        Page<RoleEntity> page = new Page<>(rolePagination.getCurrentPage(), rolePagination.getPageSize());
        IPage<RoleEntity> iPage = this.page(page, queryWrapper);
        return rolePagination.setData(iPage.getRecords(), page.getTotal());
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getListByEnCode(String enCode) {
        QueryWrapper<RoleEntity> query = new QueryWrapper<>();
        query.lambda().in(RoleEntity::getEnCode, enCode);
        return this.list(query);

    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getListByUserId(String userId) {
        QueryWrapper<RoleEntity> query = new QueryWrapper<>();
        List<String> roleRelations =
                userRelationService.getListByObjectType(userId, PermissionConst.ROLE).stream().map(UserRelationEntity::getObjectId).collect(Collectors.toList());
        if (roleRelations.size() > 0) {
            query.lambda().in(RoleEntity::getId, roleRelations);
            return this.list(query);
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getListByUserIdAndOrgId(String userId, String orgId) {
        return getListByUserId(userId).stream().filter(role -> organizeRelationService.existByObejctIdAndOrgIdAndObjectType(orgId, role.getId(), PermissionConst.ROLE)).collect(Collectors.toList());
    }

    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<String> getRoleIdsByCurrentUser() {
        UserEntity userEntity = userService.getInfo(userProvider.get().getUserId());
        return getAllRoleIdsByUserIdAndOrgId(userEntity.getId(), userEntity.getOrganizeId());
    }

    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<String> getRoleIdsByCurrentUser(String orgId) {
        UserEntity userEntity = userService.getInfo(userProvider.get().getUserId());
        return getAllRoleIdsByUserIdAndOrgId(userEntity.getId(), orgId);
    }

    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<String> getAllRoleIdsByUserIdAndOrgId(String userId, String orgId) {
        // 用户当前组织下的角色
        List<String> roleIds =
                getListByUserIdAndOrgId(userId, orgId).stream().map(RoleEntity::getId).collect(Collectors.toList());
        // 用户绑定的全局角色
        List<String> globalRoleIds =
                userRelationService.getListByUserIdAndObjType(userId, PermissionConst.ROLE).stream().map(UserRelationEntity::getObjectId).collect(Collectors.toList());
        globalRoleIds =
                roleService.getListByIds(globalRoleIds).stream().filter(r -> r.getGlobalMark() != null && r.getGlobalMark().intValue() == NOT_ORGANIZE.getCode().intValue()).map(RoleEntity::getId).collect(Collectors.toList());
        roleIds.addAll(globalRoleIds);
        return roleIds.stream().distinct().collect(Collectors.toList());
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result==null")
    public RoleEntity getInfo(String id) {
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(RoleEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    public Boolean isExistByFullName(String fullName, String id) {
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(RoleEntity::getFullName, fullName);
        if (!StringUtils.isEmpty(id)) {
            queryWrapper.lambda().ne(RoleEntity::getId, id);
        }
        return this.count(queryWrapper) > 0 ? true : false;
    }

    @Override
    public Boolean isExistByEnCode(String enCode, String id) {
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(RoleEntity::getEnCode, enCode);
        if (!StringUtils.isEmpty(id)) {
            queryWrapper.lambda().ne(RoleEntity::getId, id);
        }
        return this.count(queryWrapper) > 0 ? true : false;
    }

    @DSTransactional
    @Override
    @CacheEvict(value = "RoleCache", allEntries = true)
    public Boolean update(String id, RoleUpForm roleUpForm) throws PermissionException, DataException {
        // 如果角色下面有用户则不允许修改角色类型和所属组织
        RoleEntity roleEntity = roleService.getInfo(id);
        if (roleEntity == null) {
            throw new PermissionException("未找到角色");
        }
        //如果从【组织】角色变为【全局】角色
        if (roleUpForm.getGlobalMark() == 1 && roleEntity.getGlobalMark()==0) {
            //删除关联的组织
            organizeRelationService.deleteAllByObjectId(id, PermissionConst.ROLE);

            //如果是【组织】角色
        }else if (roleUpForm.getGlobalMark() == 0 ) {

            // 修改前所属组织角色
            List<OrganizeRelationEntity> relationListByRoleId = organizeRelationService.getRelationListByRoleId(id);
            List<String> beforeOrgIds = relationListByRoleId.stream().map(OrganizeRelationEntity::getOrganizeId).collect(Collectors.toList());

            //修改后提交的组织角色
            List<String> updateOrgIds = PermissionUtil.getOrgIdsByFormTree(organizeService, roleUpForm.getOrganizeIdsTree());

            // 并集：所有未修改的ID
            List<String> unUpdateOrgIds = beforeOrgIds.stream().filter(updateOrgIds::contains).collect(Collectors.toList());
            // 【删除】的的角色差集ID：去掉没有变动的ID
            beforeOrgIds.removeAll(unUpdateOrgIds);
            ///待删除的部门用户列表。需要和角色求交集
            List<UserRelationEntity> deleteUserList = userRelationService.getListByOrgId(beforeOrgIds);
            List<String> deleteOrgUserIdList = deleteUserList.stream().map(UserRelationEntity::getUserId).collect(Collectors.toList());
            ///待删除的角色用户列表。需要和部门求交集
            List<UserRelationEntity> bingUserByRoleList = userRelationService.getListByObjectId(id, PermissionConst.ROLE);
            List<String> deleteRoleUserIdList = bingUserByRoleList.stream().map(UserRelationEntity::getUserId).collect(Collectors.toList());
            //删除用户与角色的关联
            List<String> deletedUserIds = deleteRoleUserIdList.stream().filter(deleteOrgUserIdList::contains).collect(Collectors.toList());
            List<String> deletedUserRelationIds = bingUserByRoleList.stream().filter(userRelation -> deletedUserIds.contains(userRelation.getUserId())).map(UserRelationEntity::getId).collect(Collectors.toList());
            userRelationService.delete(deletedUserRelationIds.toArray(new String[deletedUserRelationIds.size()]));
        }
        // 组织关系
        RoleEntity entity = JsonUtil.getJsonToBean(roleUpForm, RoleEntity.class);
        if (roleService.isExistByFullName(roleUpForm.getFullName(), id)) {
            throw new PermissionException("角色名称不能重复");
        }
        if (roleService.isExistByEnCode(roleUpForm.getEnCode(), id)) {
            throw new PermissionException("角色编码不能重复");
        }

        entity.setId(id);
        this.updateById(entity);

        roleService.createOrganizeRoleRelation(roleUpForm.getOrganizeIdsTree(), id, roleUpForm.getGlobalMark());
        return true;
    }

    @Override
    @DSTransactional
    @CacheEvict(value = "RoleCache", allEntries = true)
    public Boolean update(RoleEntity entity) {
        return this.updateById(entity);
    }

    @Override
    @CacheEvict(value = "RoleCache", allEntries = true)
    public int updateVerify(String id, RoleUpForm roleUpForm) throws PermissionException, DataException {
        // 如果角色下面有用户则不允许修改角色类型和所属组织
        RoleEntity roleEntity = roleService.getInfo(id);
        if (roleEntity == null) {
            throw new PermissionException("未找到角色");
        }
        //如果从【组织】角色变为【全局】角色
        if (roleUpForm.getGlobalMark() == 0 ) {

            // 修改前所属组织角色
            List<OrganizeRelationEntity> relationListByRoleId = organizeRelationService.getRelationListByRoleId(id);
            List<String> beforeOrgIds = relationListByRoleId.stream().map(OrganizeRelationEntity::getOrganizeId).collect(Collectors.toList());

            //修改后提交的组织角色
            List<String> updateOrgIds = PermissionUtil.getOrgIdsByFormTree(organizeService, roleUpForm.getOrganizeIdsTree());

            // 并集：所有未修改的ID
            List<String> unUpdateOrgIds = beforeOrgIds.stream().filter(updateOrgIds::contains).collect(Collectors.toList());
            // 【删除】的的角色差集ID：去掉没有变动的ID
            beforeOrgIds.removeAll(unUpdateOrgIds);
            ///待删除的部门用户列表。需要和角色求交集
            List<UserRelationEntity> deleteUserList = userRelationService.getListByOrgId(beforeOrgIds);
            List<String> deleteOrgUserIdList = deleteUserList.stream().map(UserRelationEntity::getUserId).collect(Collectors.toList());
            ///待删除的角色用户列表。需要和部门求交集
            List<UserRelationEntity> bingUserByRoleList = userRelationService.getListByObjectId(id, PermissionConst.ROLE);
            List<String> deleteRoleUserIdList = bingUserByRoleList.stream().map(UserRelationEntity::getUserId).collect(Collectors.toList());
            //删除用户与角色的关联
            List<String> deletedUserIds = deleteRoleUserIdList.stream().filter(deleteOrgUserIdList::contains).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(deletedUserIds)) {
                return deletedUserIds.size();
            }
        }
        return 0;
    }

    @Override
    @CacheEvict(value = "RoleCache", allEntries = true)
    public void create(RoleEntity entity) {

        this.save(entity);
    }

    @Override
    @DSTransactional
    @CacheEvict(value = "RoleCache", allEntries = true)
    public void copyRole(RoleUpForm roleUpForm, String fromId) throws BaseException {

        RoleEntity entity = JsonUtil.getJsonToBean(roleUpForm, RoleEntity.class);
        if (roleService.isExistByFullName(roleUpForm.getFullName(), null)) {
            throw new BaseException("角色名称不能重复");
        }
        if (roleService.isExistByEnCode(roleUpForm.getEnCode(), null)) {
            throw new BaseException("角色编码不能重复");
        }
        entity.setId(RandomUtil.uuId());
        //创建角色
        this.save(entity);

        this.createOrganizeRoleRelation(roleUpForm.getOrganizeIdsTree(), entity.getId(), entity.getGlobalMark());
        //复制数据
        List<AuthorizeEntity> authorizeEntities = authorizeService.getListByObjectId(fromId);
        for (AuthorizeEntity authorizeEntity : authorizeEntities) {
            authorizeEntity.setObjectId(entity.getId());
            authorizeEntity.setId(RandomUtil.uuId());
            authorizeService.save(authorizeEntity);
        }

    }

    /**
     * 添加组织角色关联关系
     */
    @Override
    @DSTransactional
    @CacheEvict(value = "RoleCache", allEntries = true)
    public Boolean createOrganizeRoleRelation(List<List<String>> organizeIdsTree, String roleId, Integer globalMark) {
        // 清除之前的关联关系
        QueryWrapper<OrganizeRelationEntity> query = new QueryWrapper<>();
        query.lambda().eq(OrganizeRelationEntity::getObjectType, PermissionConst.ROLE);
        query.lambda().eq(OrganizeRelationEntity::getObjectId, roleId);
        organizeRelationService.remove(query);
        // globalMark等于0时，为组织角色
        if (globalMark.equals(0)) {
            List<OrganizeRelationEntity> relationList = new ArrayList<>();
            for (List<String> organizeIds : organizeIdsTree) {
                // 组织id数组树最后一个数组最后一个id，是需要储存的id
                String organizeId = organizeIds.get(organizeIds.size() - 1);
                // 添加与组织的关联关系
                OrganizeRelationEntity organizeRelationEntity = new OrganizeRelationEntity();
                organizeRelationEntity.setId(RandomUtil.uuId());
                organizeRelationEntity.setOrganizeId(organizeId);
                organizeRelationEntity.setObjectType(PermissionConst.ROLE);
                organizeRelationEntity.setObjectId(roleId);
                relationList.add(organizeRelationEntity);
            }
            organizeRelationService.saveBatch(relationList);
        }
        return true;
    }

    @Override
    @DSTransactional
    @CacheEvict(value = "RoleCache", allEntries = true)
    public void delete(RoleEntity entity) {
        if (entity != null) {
            this.removeById(entity.getId());
            QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().eq(AuthorizeEntity::getObjectId, entity.getId());
            authorizeService.remove(queryWrapper);
            QueryWrapper<UserRelationEntity> wrapper = new QueryWrapper<>();
            wrapper.lambda().eq(UserRelationEntity::getObjectId, entity.getId());
            userRelationService.remove(wrapper);
        }
    }

    /**
     * 根据菜单id获取角色列表信息
     *
     * @param menuId
     * @return
     */
    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<AuthorizeEntity> getListBymenuId(String menuId) {
        QueryWrapper<AuthorizeEntity> query = new QueryWrapper<>();
//        query.lambda().eq(AuthorizeEntity::getItemType, "module");
        query.lambda().eq(AuthorizeEntity::getItemId, menuId);
        query.lambda().eq(AuthorizeEntity::getObjectType, "Role");
        List<AuthorizeEntity> ls = authorizeService.list(query);
        List<AuthorizeEntity> distinctPersons = ls.stream().collect(Collectors.toMap(AuthorizeEntity::getObjectId,
                p -> p, (p1, p2) -> p1)).values().stream().collect(Collectors.toList());
        return distinctPersons;
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getListByIds(List<String> id) {
        List<RoleEntity> roleList = new ArrayList<>();
        if (id.size() > 0) {
            QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(RoleEntity::getId, id);
            roleList = this.list(queryWrapper);
        }
        return roleList;
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result==null")
    public RoleEntity getInfoByFullName(String fullName) {
        QueryWrapper<RoleEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(RoleEntity::getFullName, fullName);
        return this.getOne(queryWrapper);
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getGlobalList() {
        QueryWrapper<RoleEntity> query = new QueryWrapper<>();
        query.lambda().eq(RoleEntity::getGlobalMark, NOT_ORGANIZE.getCode());
        return this.list(query);
    }


    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getCurRolesByOrgId(String orgId) {
        String userId = userProvider.get().getUserId();
        List<UserRelationEntity> userRelations = userRelationService.getListByObjectType(userId, PermissionConst.ROLE);
        List<RoleEntity> roles = new ArrayList<>();
        userRelations.forEach(ur -> {
            // 获取全局角色
            RoleEntity roleEntity = this.getInfo(ur.getObjectId());
            if (ObjectUtil.isNotEmpty(roleEntity) && ObjectUtil.isNotEmpty(roleEntity.getGlobalMark()) && NOT_ORGANIZE.getCode().intValue() == roleEntity.getGlobalMark().intValue()) {
                roles.add(roleEntity);
            } else {
                organizeRelationService.getRelationListByRoleId(ur.getObjectId()).forEach(or -> {
                    if (or.getOrganizeId().equals(orgId)) {
                        roles.add(roleEntity);
                    }
                });
            }
        });
        return roles;
    }

    @Override
    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public List<RoleEntity> getRolesByOrgId(String orgId) {
        List<String> ids = new ArrayList<>();
        organizeRelationService.getListByTypeAndOrgId(PermissionConst.ROLE, orgId).forEach(o -> {
            ids.add(o.getObjectId());
        });
        QueryWrapper<RoleEntity> query = new QueryWrapper<>();
        if (ids.size() > 0) {
            query.lambda().in(RoleEntity::getId, ids);
            return this.list(query);
        } else {
            return new ArrayList<>();
        }
    }

    @Override
//    @Cacheable(value = "RoleCache", keyGenerator = "cacheKeyGenerator", unless = "#result==null")
    public String getBindInfo(String roleId, List<String> reduceOrgIds) {
        if (reduceOrgIds.size() > 0) {
            StringBuilder info = new StringBuilder();
            RoleEntity roleEntity = this.getInfo(roleId);
            List<UserRelationEntity> bingUserByRoleList = userRelationService.getListByObjectId(roleId,
                    PermissionConst.ROLE);
            if (bingUserByRoleList.size() < 1) {
                return null;
            }
            info.append("已绑定用户：");
            boolean bindFlag = false;
            for (UserRelationEntity bingUser : bingUserByRoleList) {
                String userId = bingUser.getUserId();
                //角色是全局，
                if (ObjectUtil.isNotEmpty(roleEntity) && ObjectUtil.isNotEmpty(roleEntity.getGlobalMark()) && NOT_ORGANIZE.getCode().intValue() == roleEntity.getGlobalMark().intValue()) {
                    UserEntity user = userService.getInfo(userId);
                    info.append("[ ").append(user.getRealName()).append(" ] ");
                    bindFlag = true;
                } else {
                    // 这个用户所绑定的组织
                    List<UserRelationEntity> bingUserByOrg = userRelationService.getListByObjectType(userId,
                            PermissionConst.ORGANIZE);
                    for (UserRelationEntity bingOrg : bingUserByOrg) {
                        String orgId = bingOrg.getObjectId();
                        for (String reduceOrgId : reduceOrgIds) {
//                            判断角色所属组织和角色下的用户所属组织是否有相同的
                            if (reduceOrgId.equals(orgId)) {
                                OrganizeEntity org = organizeService.getInfo(orgId);
                                UserEntity user = userService.getInfo(bingOrg.getUserId());
                                info.append("[").append(org.getFullName()).append("：用户（").append(user.getRealName()).append("）]; ");
                                bindFlag = false;
                                break;
                            } else {
                                bindFlag = true;
                            }
                        }
                    }
                }
            }
            if (bindFlag) {
                return info.toString();
            } else {
                return null;
            }
        } else {
            return null;
        }
    }
}
