package com.bringspring.system.scheduletask.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.model.task.LocalTaskModel;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.system.scheduletask.entity.TimeTaskEntity;
import com.bringspring.system.scheduletask.entity.TimeTaskLogEntity;
import com.bringspring.system.scheduletask.model.*;
import com.bringspring.system.scheduletask.service.TimetaskService;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 任务调度
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Tag(name = "任务调度", description = "TimeTask")
@RestController
@RequestMapping("/api/scheduletask")
public class TimeTaskController {

    @Autowired
    private TimetaskService timetaskService;
    @Autowired
    private ConcurrentHashMap<String, LocalTaskModel> concurrentHashMap;

    /**
     * 获取任务调度列表
     *
     * @param pagination
     * @return
     */
    @Operation(summary = "获取任务调度列表")
    @GetMapping
    public ActionResult<PageListVO<TaskVO>> list(Pagination pagination) {
        List<TimeTaskEntity> data = timetaskService.getList(pagination);
        List<TaskVO> list = JsonUtil.getJsonToList(data, TaskVO.class);
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(list, paginationVO);
    }

    /**
     * 获取本地任务列表
     *
     * @return
     */
    @Operation(summary = "获取任务调度列表")
    @GetMapping("/TaskMethods")
    public ActionResult<List<TaskMethodsVO>> taskMethods() {
        List<TaskMethodsVO> list = new ArrayList<>(16);
        for (String id : concurrentHashMap.keySet()) {
            LocalTaskModel model = concurrentHashMap.get(id);
            TaskMethodsVO taskMethodsVO = new TaskMethodsVO();
            taskMethodsVO.setId(model.getId());
            taskMethodsVO.setFullName(model.getFullName());
            taskMethodsVO.setDescription(model.getDescription());
            list.add(taskMethodsVO);
        }
        return ActionResult.success(list);
    }

    /**
     * 获取任务调度日志列表
     *
     * @param pagination
     * @param taskId     任务Id
     * @return
     */
    @Operation(summary = "获取任务调度日志列表")
    @GetMapping("/{id}/TaskLog")
    public ActionResult<PageListVO<TaskLogVO>> list(@PathVariable("id") String taskId, TaskPage pagination) {
        List<TimeTaskLogEntity> data = timetaskService.getTaskLogList(taskId, pagination);
        List<TaskLogVO> list = JsonUtil.getJsonToList(data, TaskLogVO.class);
        PaginationVO pageModel = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(list, pageModel);
    }

    /**
     * 获取任务调度信息
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "获取任务调度信息")
    @GetMapping("/Info/{id}")
    public ActionResult<TaskInfoVO> info(@PathVariable("id") String id) throws DataException {
        TimeTaskEntity entity = timetaskService.getInfo(id);
        TaskInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, TaskInfoVO.class);
        return ActionResult.success(vo);
    }


    /**
     * 根据encode获取任务调度信息
     *
     * @param encode 主键值
     * @return
     */
    @Operation(summary = "获取任务调度信息")
    @GetMapping("/getListByEncode/{encode}")
    public ActionResult<List<TaskVO>> getListByEncode(@PathVariable("encode") String encode) throws DataException {
        List<TimeTaskEntity> listByEncode = timetaskService.getListByEncode(encode);
        List<TaskVO> list = JsonUtil.getJsonToList(listByEncode, TaskVO.class);
        return ActionResult.success(list);
    }

    /**
     * 新建任务调度
     *
     * @param taskCrForm
     * @return
     */
    @Operation(summary = "新建任务调度")
    @PostMapping
    public ActionResult create(@RequestBody @Valid TaskCrForm taskCrForm) {
        TimeTaskEntity entity = JsonUtil.getJsonToBean(taskCrForm, TimeTaskEntity.class);
        if (timetaskService.isExistByFullName(entity.getFullName(), entity.getId())) {
            return ActionResult.fail("任务名称不能重复");
        }
        timetaskService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 修改任务调度
     *
     * @param id         主键值
     * @param taskUpForm
     * @return
     */
    @Operation(summary = "修改任务调度")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid TaskUpForm taskUpForm) {
        TimeTaskEntity entity = JsonUtil.getJsonToBean(taskUpForm, TimeTaskEntity.class);
        if (timetaskService.isExistByFullName(entity.getFullName(), id)) {
            return ActionResult.fail("任务名称不能重复");
        }
        TimeTaskEntity taskEntity = timetaskService.getInfo(id);
        if (taskEntity == null) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        entity.setEnabledMark(taskEntity.getEnabledMark());
        entity.setRunCount(taskEntity.getRunCount());
        boolean flag = timetaskService.update(id, entity);
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除任务
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "删除任务")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        TimeTaskEntity entity = timetaskService.getInfo(id);
        if (entity != null) {
            timetaskService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    /**
     * 任务调度立即执行一次
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "立即执行任务")
    @PutMapping("/{id}/Actions/startNow")
    public ActionResult jobStartNow(@PathVariable("id") String id) {
        TimeTaskEntity entity = timetaskService.getInfo(id);
        if (entity != null) {
            timetaskService.jobStartNow(id, entity);
            return ActionResult.success(MsgCode.SU005.get());
        }
        return ActionResult.fail("操作失败，任务不存在");
    }

    /**
     * 停止任务调度
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "停止任务调度")
    @PutMapping("/{id}/Actions/Stop")
    public ActionResult stop(@PathVariable("id") String id) {
        TimeTaskEntity entity = timetaskService.getInfo(id);
        if (entity != null) {
            entity.setEnabledMark(0);
            timetaskService.update(id, entity);
            return ActionResult.success(MsgCode.SU005.get());
        }
        return ActionResult.fail("操作失败，任务不存在");
    }

    /**
     * 启动任务调度
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "启动任务调度")
    @PutMapping("/{id}/Actions/Enable")
    public ActionResult enable(@PathVariable("id") String id) {
        TimeTaskEntity entity = timetaskService.getInfo(id);
        if (entity != null) {
            entity.setEnabledMark(1);
            timetaskService.update(id, entity);
            return ActionResult.success(MsgCode.SU005.get());
        }
        return ActionResult.fail("操作失败，任务不存在");
    }

}
