package com.bringspring.system.base.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.DictionaryTypeEntity;
import com.bringspring.system.base.model.dictionarytype.*;
import com.bringspring.system.base.service.DictionaryTypeService;
import com.bringspring.system.base.util.JsonUtilEx;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.LinkedList;
import java.util.List;

/**
 * 字典分类
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Tag(name = "数据字典分类")
@RestController
@RequestMapping("/api/system/DictionaryType")
public class DictionaryTypeController {

    @Autowired
    private DictionaryTypeService dictionaryTypeService;

    /**
     * 获取字典分类
     *
     * @return ignore
     */
    @Operation(summary="获取字典分类")
    @GetMapping
    public ActionResult<ListVO<DictionaryTypeListVO>> treeView() {
        List<DictionaryTypeEntity> data = dictionaryTypeService.getList();

        List<DictionaryTypeModel> voListVO = JsonUtil.getJsonToList(data, DictionaryTypeModel.class);
        List<SumTree<DictionaryTypeModel>> sumTrees = TreeDotUtils.convertListToTreeDot(voListVO);
        List<DictionaryTypeListVO> list = JsonUtil.getJsonToList(sumTrees, DictionaryTypeListVO.class);
        List<DictionaryTypeListVO> listVo = new LinkedList<>();
        for (DictionaryTypeListVO dictionaryTypeListVO : list) {
            if ("-1".equals(dictionaryTypeListVO.getParentId())) {
                listVo.add(dictionaryTypeListVO);
            }
        }
        ListVO<DictionaryTypeListVO> vo = new ListVO<>();
        vo.setList(listVo);
        return ActionResult.success(vo);
    }

    /**
     * 获取字典分类
     *
     * @return ignore
     */
    @Operation(summary="获取字典分类")
    @PostMapping("/getListByIdList")
    public ActionResult<ListVO<DictionaryTypeListVO>> getListByIdList(@RequestBody List<String> idList) {
        List<DictionaryTypeEntity> data = dictionaryTypeService.getListByIdList(idList);

        List<DictionaryTypeModel> voListVO = JsonUtil.getJsonToList(data, DictionaryTypeModel.class);
        List<SumTree<DictionaryTypeModel>> sumTrees = TreeDotUtils.convertListToTreeDot(voListVO);
        List<DictionaryTypeListVO> list = JsonUtil.getJsonToList(sumTrees, DictionaryTypeListVO.class);
        ListVO<DictionaryTypeListVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(vo);
    }

    /**
     * 获取字典分类
     *
     * @param id 主键
     * @return ignore
     */
    @Operation(summary="获取所有字典分类下拉框列表")
    @GetMapping("/Selector/{id}")
    public ActionResult<ListVO<DictionaryTypeSelectVO>> selectorTreeView(@PathVariable("id") String id) {
        List<DictionaryTypeEntity> data = dictionaryTypeService.getList();
        if (!"0".equals(id)) {
            data.remove(dictionaryTypeService.getInfo(id));
        }
        List<DictionaryTypeSelectModel> voListVO = JsonUtil.getJsonToList(data, DictionaryTypeSelectModel.class);
        List<SumTree<DictionaryTypeSelectModel>> sumTrees = TreeDotUtils.convertListToTreeDotFilter(voListVO);
        List<DictionaryTypeSelectVO> list = JsonUtil.getJsonToList(sumTrees, DictionaryTypeSelectVO.class);
        List<DictionaryTypeSelectVO> listVo = new LinkedList<>();
        for (DictionaryTypeSelectVO dictionaryTypeListVO : list) {
            if ("-1".equals(dictionaryTypeListVO.getParentId())) {
                listVo.add(dictionaryTypeListVO);
            }
        }
        ListVO<DictionaryTypeSelectVO> vo = new ListVO<>();
        vo.setList(listVo);
        return ActionResult.success(vo);
    }


    /**
     * 获取字典分类信息
     *
     * @param id 主键
     * @return ignore
     * @throws DataException ignore
     */
    @Operation(summary="获取字典分类信息")
    @GetMapping("/{id}")
    public ActionResult<DictionaryTypeInfoVO> info(@PathVariable("id") String id) throws DataException {
        DictionaryTypeEntity entity = dictionaryTypeService.getInfo(id);
        DictionaryTypeInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, DictionaryTypeInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * @描述：根据父级Id获取数据字典分类
     * @参数：
     * @版权：荣科科技股份有限公司
     * @版本：V1.0.0
     * @作者：yuefy
     * @时间：2024/8/1 15:31
     */
    @Operation(summary="根据父级Id获取字典分类信息")
    @GetMapping("listByParentId/{parentId}")
    public ActionResult getDictByParentId(@PathVariable("parentId") String parentId){
        QueryWrapper<DictionaryTypeEntity> wrapper = new QueryWrapper<>();
        wrapper.lambda().eq(DictionaryTypeEntity :: getParentId,parentId);
        List<DictionaryTypeEntity> list = dictionaryTypeService.list(wrapper);
        return ActionResult.success(list);
    }

    /**
     * 添加字典分类
     *
     * @param dictionaryTypeCrForm 实体对象
     * @return ignore
     */
    @Operation(summary="添加字典分类")
    @PostMapping
    public ActionResult create(@RequestBody @Valid DictionaryTypeCrForm dictionaryTypeCrForm) {
        DictionaryTypeEntity entity = JsonUtil.getJsonToBean(dictionaryTypeCrForm, DictionaryTypeEntity.class);
        entity.setParentId(entity.getParentId());
        if (dictionaryTypeService.isExistByFullName(entity.getFullName(), entity.getId())) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        if (dictionaryTypeService.isExistByEnCode(entity.getEnCode(), entity.getId())) {
            return ActionResult.fail(MsgCode.EXIST002.get());
        }
        dictionaryTypeService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 修改字典分类
     *
     * @param dictionaryTypeUpForm 实体对象
     * @param id                   主键值
     * @return ignore
     */
    @Operation(summary="修改字典分类")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid DictionaryTypeUpForm dictionaryTypeUpForm) {
        DictionaryTypeEntity entity = JsonUtil.getJsonToBean(dictionaryTypeUpForm, DictionaryTypeEntity.class);
        if (dictionaryTypeService.isExistByFullName(entity.getFullName(), id)) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        if (dictionaryTypeService.isExistByEnCode(entity.getEnCode(), id)) {
            return ActionResult.fail(MsgCode.EXIST002.get());
        }
        boolean flag = dictionaryTypeService.update(id, entity);
        if (!flag) {
            return ActionResult.success(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除字典分类
     *
     * @param id 主键值
     * @return ignore
     */
    @Operation(summary="删除字典分类")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        DictionaryTypeEntity entity = dictionaryTypeService.getInfo(id);
        if (entity != null) {
            boolean isOk = dictionaryTypeService.delete(entity);
            if (isOk) {
                return ActionResult.success(MsgCode.SU003.get());
            } else {
                return ActionResult.fail("字典类型下面有字典值禁止删除");
            }
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

}
