package com.bringspring.system.scheduletask.service.impl;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.model.task.LocalTaskModel;
import com.bringspring.common.util.*;
import com.bringspring.system.base.util.jobutil.TimeUtil;
import com.bringspring.system.scheduletask.entity.TimeTaskEntity;
import com.bringspring.system.scheduletask.entity.TimeTaskLogEntity;
import com.bringspring.system.scheduletask.mapper.TimeTaskMapper;
import com.bringspring.system.scheduletask.model.ContentModel;
import com.bringspring.system.scheduletask.model.TaskPage;
import com.bringspring.system.scheduletask.service.TimeTaskLogService;
import com.bringspring.system.scheduletask.service.TimetaskService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 定时任务
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Service
public class TimetaskServiceImpl extends ServiceImpl<TimeTaskMapper, TimeTaskEntity> implements TimetaskService {

    @Autowired
    private UserProvider userProvider;
    @Autowired
    private TimeTaskLogService timeTaskLogService;
    @Autowired
    private ConcurrentHashMap<String, LocalTaskModel> concurrentHashMap;

    @Override
    public List<TimeTaskEntity> getList() {
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(TimeTaskEntity::getEnabledMark, 1);
        queryWrapper.lambda().orderByAsc(TimeTaskEntity::getSortCode).orderByDesc(TimeTaskEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public List<TimeTaskEntity> getList(Pagination pagination) {
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        if (pagination.getKeyword() != null) {
            queryWrapper.lambda().and(t -> t.like(TimeTaskEntity::getEnCode, pagination.getKeyword()).or().like(TimeTaskEntity::getFullName, pagination.getKeyword()));
        }
        //排序
        queryWrapper.lambda().orderByAsc(TimeTaskEntity::getSortCode).orderByDesc(TimeTaskEntity::getCreatorTime);
        Page page = new Page(pagination.getCurrentPage(), pagination.getPageSize());
        IPage<TimeTaskEntity> iPage = this.page(page, queryWrapper);
        return pagination.setData(iPage.getRecords(), page.getTotal());
    }

    @Override
    public List<TimeTaskLogEntity> getTaskLogList(String taskId, TaskPage pagination) {
        QueryWrapper<TimeTaskLogEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(TimeTaskLogEntity::getTaskId, taskId);
        //日期范围（近7天、近1月、近3月、自定义）
        String startTime = pagination.getStartTime() != null ? pagination.getStartTime() : null;
        String endTime = pagination.getEndTime() != null ? pagination.getEndTime() : null;
        if (!org.apache.commons.lang3.StringUtils.isEmpty(startTime) && !org.apache.commons.lang3.StringUtils.isEmpty(endTime)) {
            Date startTimes = DateUtil.stringToDate(DateUtil.daFormatYmd(Long.parseLong(startTime)) + " 00:00:00");
            Date endTimes = DateUtil.stringToDate(DateUtil.daFormatYmd(Long.parseLong(endTime)) + " 23:59:59");
            queryWrapper.lambda().ge(TimeTaskLogEntity::getRunTime, startTimes).le(TimeTaskLogEntity::getRunTime, endTimes);
        }
        //排序
        if (StringUtils.isEmpty(pagination.getSidx())) {
            queryWrapper.lambda().orderByDesc(TimeTaskLogEntity::getRunTime);
        } else {
            queryWrapper = "asc".equals(pagination.getSort().toLowerCase()) ? queryWrapper.orderByAsc(pagination.getSidx()) : queryWrapper.orderByDesc(pagination.getSidx());
        }
        if (pagination.getRunResult() != null) {
            queryWrapper.lambda().eq(TimeTaskLogEntity::getRunResult, pagination.getRunResult());
        }
        Page page = new Page(pagination.getCurrentPage(), pagination.getPageSize());
        IPage<TimeTaskLogEntity> iPage = timeTaskLogService.page(page, queryWrapper);
        return pagination.setData(iPage.getRecords(), page.getTotal());
    }

    @Override
    public TimeTaskEntity getInfo(String id) {
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(TimeTaskEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    public boolean isExistByFullName(String fullName, String id) {
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(TimeTaskEntity::getFullName, fullName);
        if (!StringUtils.isEmpty(id)) {
            queryWrapper.lambda().ne(TimeTaskEntity::getId, id);
        }
        return this.count(queryWrapper) > 0 ? true : false;
    }

    @Override
    public boolean isExistByEnCode(String enCode, String id) {
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(TimeTaskEntity::getEnCode, enCode);
        if (!StringUtils.isEmpty(id)) {
            queryWrapper.lambda().ne(TimeTaskEntity::getId, id);
        }
        return this.count(queryWrapper) > 0 ? true : false;
    }

    @Override
    public void create(TimeTaskEntity entity) {
        UserInfo userInfo = userProvider.get();
        String token = UserProvider.getToken();
        entity.setId(RandomUtil.uuId());
        entity.setEnabledMark(1);
        entity.setCreatorUserId(userInfo.getUserId());
        ContentModel model = JsonUtil.getJsonToBean(entity.getExecuteContent(), ContentModel.class);
        if ("3".equals(entity.getExecuteType()) && StringUtils.isEmpty(entity.getExecuteCronExpression())) {
            // executeType：3-本地方法/任务 && 执行周期(Cron表达式) 是空
            if (concurrentHashMap.containsKey(model.getLocalHostTaskId())) {
                // 本地方法/任务存在，获取并赋值 执行周期(Cron表达式)
                LocalTaskModel taskModel = concurrentHashMap.get(model.getLocalHostTaskId());
                entity.setExecuteCronExpression(taskModel.getCron());
                model.setCron(taskModel.getCron());
            }
        }
        if (this.save(entity)) { //定时任务新建成功
            start(model, entity, userInfo, token); //启动任务调度
        }
    }

    @Override
    public boolean update(String id, TimeTaskEntity entity) {
        UserInfo userInfo = userProvider.get();
        String token = UserProvider.getToken();
        entity.setId(id);
        entity.setLastModifyTime(DateUtil.getNowDate());
        entity.setLastModifyUserId(userInfo.getUserId());

        ContentModel model = JsonUtil.getJsonToBean(entity.getExecuteContent(), ContentModel.class);
        if (isjson(entity.getExecuteContent())) {
            //如果Cron表达式是空，则使用默认表达式执行定时任务
            if ("3".equals(entity.getExecuteType()) && StringUtils.isEmpty(entity.getExecuteCronExpression())) {
                if (concurrentHashMap.containsKey(model.getLocalHostTaskId())) {
                    LocalTaskModel taskModel = concurrentHashMap.get(model.getLocalHostTaskId());
                    entity.setExecuteCronExpression(taskModel.getCron());
                    model.setCron(taskModel.getCron());
                }
            }

            if (this.updateById(entity)) { //定时任务变更成功
                del(entity, userInfo); // 删除任务调度
                if (entity.getEnabledMark() == 1) {
                    start(model, entity, userInfo, token); //启动任务调度
                }
                return true;
            }
        }
        return false;
    }


    @Override
    public void delete(TimeTaskEntity entity) {
        UserInfo userInfo = userProvider.get();
        if (this.removeById(entity.getId())) { //定时任务删除成功
            del(entity, userInfo); // 删除任务调度
            timeTaskLogService.delete(entity.getId()); // 删除任务日志
        }
    }

    @Override
    @DSTransactional
    public boolean first(String id) {
        boolean isOk = false;
        //获取要上移的那条数据的信息
        TimeTaskEntity upEntity = this.getById(id);
        Long upSortCode = upEntity.getSortCode() == null ? 0 : upEntity.getSortCode();
        //查询上几条记录
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().lt(TimeTaskEntity::getSortCode, upSortCode).orderByDesc(TimeTaskEntity::getSortCode);
        List<TimeTaskEntity> downEntity = this.list(queryWrapper);
        if (downEntity.size() > 0) {
            //交换两条记录的sort值
            Long temp = upEntity.getSortCode();
            upEntity.setSortCode(downEntity.get(0).getSortCode());
            downEntity.get(0).setSortCode(temp);
            this.updateById(downEntity.get(0));
            this.updateById(upEntity);
            isOk = true;
        }
        return isOk;
    }

    @Override
    @DSTransactional
    public boolean next(String id) {
        boolean isOk = false;
        //获取要下移的那条数据的信息
        TimeTaskEntity downEntity = this.getById(id);
        Long upSortCode = downEntity.getSortCode() == null ? 0 : downEntity.getSortCode();
        //查询下几条记录
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().gt(TimeTaskEntity::getSortCode, upSortCode).orderByAsc(TimeTaskEntity::getSortCode);
        List<TimeTaskEntity> upEntity = this.list(queryWrapper);
        if (upEntity.size() > 0) {
            //交换两条记录的sort值
            Long temp = downEntity.getSortCode();
            downEntity.setSortCode(upEntity.get(0).getSortCode());
            upEntity.get(0).setSortCode(temp);
            this.updateById(upEntity.get(0));
            this.updateById(downEntity);
            isOk = true;
        }
        return isOk;
    }

    @Override
    public void createTaskLog(TimeTaskLogEntity entity) {
        entity.setId(RandomUtil.uuId());
        entity.setRunTime(DateUtil.getNowDate());
        timeTaskLogService.save(entity);
    }

    /**
     * 判断是否是json格式
     *
     * @param string
     * @return
     */
    private boolean isjson(String string) {
        try {
            JSONObject jsonStr = JSONObject.parseObject(string);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 启动任务调度
     *
     * @param model
     * @param entity
     * @param userInfo
     */
    private void start(ContentModel model, TimeTaskEntity entity, UserInfo userInfo, String token) {
        TimeUtil.startJob(model, userInfo, entity, token);
    }

    /**
     * 立即执行一次
     *
     * @param id
     * @param entity
     */
    @Override
    public boolean jobStartNow(String id, TimeTaskEntity entity) {
        UserInfo userInfo = userProvider.get();
        String token = UserProvider.getToken();
        ContentModel model = JsonUtil.getJsonToBean(entity.getExecuteContent(), ContentModel.class);
        if (isjson(entity.getExecuteContent())) {
            TimeUtil.startJobNow(model, userInfo, entity, token);
        }
        return true;
    }

    @Override
    public List<TimeTaskEntity> getListByContent(String content) {
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().like(TimeTaskEntity::getExecuteContent, content);
        return this.list(queryWrapper);
    }

    @Override
    public List<TimeTaskEntity> getListByEncode(String content) {
        QueryWrapper<TimeTaskEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(TimeTaskEntity::getEnCode, content);
        return this.list(queryWrapper);
    }

    /**
     * 删除任务调度
     *
     * @param entity
     * @param userInfo
     */
    private void del(TimeTaskEntity entity, UserInfo userInfo) {
        TimeUtil.removeJob(entity, userInfo);
    }

}
