package com.bringspring.system.base.controller;

import cn.hutool.core.collection.CollectionUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Page;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.util.treeutil.ListToTreeUtil;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.ProvinceEntity;
import com.bringspring.system.base.model.province.*;
import com.bringspring.system.base.service.ProvinceService;
import com.bringspring.system.base.util.JsonUtilEx;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import static com.bringspring.common.base.EnabledMarkEnum.ENABLE;

/**
 * 行政区划
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Tag(name = "行政区划")
@RestController
@RequestMapping("/api/system/Area")
public class AreaController {

    @Autowired
    private ProvinceService provinceService;

    /**
     * 列表（异步加载）
     *
     * @param nodeId 节点Id
     * @param page   分页对象
     * @return
     */
    @Operation(summary="列表（异步加载）")
    @GetMapping("/{nodeId}")
    public ActionResult<ListVO<ProvinceListVO>> list(@PathVariable("nodeId") String nodeId, Page page) {
        List<ProvinceEntity> data = provinceService.getListByNodeId(nodeId, page);
        List<ProvinceEntity> dataAll = data;
        List<ProvinceEntity> result = JsonUtil.getJsonToList(ListToTreeUtil.treeWhere(data, dataAll),
                ProvinceEntity.class);
        List<ProvinceListVO> treeList = JsonUtil.getJsonToList(result, ProvinceListVO.class);
        int i = 0;
        for (ProvinceListVO entity : treeList) {
            boolean childNode = provinceService.getListByParentId(entity.getId()).size() <= 0;
            ProvinceListVO provinceListVO = JsonUtil.getJsonToBean(entity, ProvinceListVO.class);
            provinceListVO.setIsLeaf(childNode);
            provinceListVO.setHasChildren(!childNode);
            treeList.set(i, provinceListVO);
            i++;
        }
        ListVO<ProvinceListVO> vo = new ListVO<>();
        vo.setList(treeList);
        return ActionResult.success(vo);
    }

    /**
     * 获取区划下拉框列表
     * 启用状态数据转换为树形结构
     *
     * @return
     */
    @Operation(summary="获取区划下拉框列表")
    @GetMapping("/Selector")
    public ActionResult<ListVO<ProvinceSelector>> getSelector(ProvinceListQuery query) {
        List<ProvinceEntity> allList = provinceService.getListAll(query);
        List<ProvinceEntity> data =
                allList.stream().filter(t -> ENABLE.getCode() == t.getEnabledMark()).collect(Collectors.toList());
        List<ProvinceModel> models = JsonUtil.getJsonToList(data, ProvinceModel.class);
        List<SumTree<ProvinceModel>> trees = TreeDotUtils.convertListToTreeDot(models);
        List<ProvinceSelector> listVO = JsonUtil.getJsonToList(trees, ProvinceSelector.class);
        ListVO vo = new ListVO();
        vo.setList(listVO);
        return ActionResult.success(vo);
    }

    /**
     * 根据上级区划id获取行政区划下拉框数据
     *
     * @param parentId  上级区划id
     * @param ignoreIds ignore
     * @return ignore
     */
    @Operation(summary="获取行政区划下拉框数据")
    @GetMapping("/{parentId}/Selector/{ignoreIds}")
    public ActionResult<ListVO<ProvinceSelectListVO>> selectList(@PathVariable("parentId") String parentId,
                                                                 @PathVariable("ignoreIds") String ignoreIds) {
        List<ProvinceEntity> data = provinceService.getListByParentId(parentId);
        if (!"0".equals(ignoreIds)) {
            //排除子集
            filterData(data, new ArrayList<>(Arrays.asList(new String[]{ignoreIds})));
        }
        List<ProvinceSelectListVO> treeList = JsonUtil.getJsonToList(data, ProvinceSelectListVO.class);
        int i = 0;
        for (ProvinceSelectListVO entity : treeList) {
//            boolean childNode = provinceService.getList(entity.getId()).size() <= 0;
            ProvinceSelectListVO provinceListVO = JsonUtil.getJsonToBean(entity, ProvinceSelectListVO.class);
            provinceListVO.setIsLeaf(false);
            treeList.set(i, provinceListVO);
            i++;
        }
        ListVO<ProvinceSelectListVO> vo = new ListVO<>();
        vo.setList(treeList);
        return ActionResult.success(vo);
    }

    /**
     * 递归排除子集
     *
     * @param data 普通列表
     * @param id   id列表
     */
    private void filterData(List<ProvinceEntity> data, List<String> id) {
        List<ProvinceEntity> collect = null;
        //获取子集信息
        for (String ids : id) {
            collect = data.stream().filter(t -> ids.equals(t.getParentId())).collect(Collectors.toList());
            data.removeAll(collect);
        }
        //递归移除子集的子集
        if (collect != null) {
            if (collect.size() > 0) {
                filterData(data, collect.stream().map(t -> t.getId()).collect(Collectors.toList()));
            }
        }
    }

    /**
     * 信息
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="获取行政区划信息")
    @GetMapping("/{id}/Info")
    public ActionResult<ProvinceInfoVO> info(@PathVariable("id") String id) throws DataException {
        ProvinceEntity entity = provinceService.getInfo(id);
        ProvinceInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, ProvinceInfoVO.class);
        if (!"-1".equals(entity.getParentId())) {
            ProvinceEntity parent = provinceService.getInfo(entity.getParentId());
            vo.setParentName(parent.getFullName());
        }
        return ActionResult.success(vo);
    }

    /**
     * 新建
     *
     * @param provinceCrForm 实体对象
     * @return
     */
    @Operation(summary="添加行政区划")
    @PostMapping
    public ActionResult create(@RequestBody @Valid ProvinceCrForm provinceCrForm) {
        ProvinceEntity entity = JsonUtil.getJsonToBean(provinceCrForm, ProvinceEntity.class);
        if (provinceService.isExistByEnCode(provinceCrForm.getEnCode(), entity.getId())) {
            return ActionResult.fail("区域编码不能重复");
        }
        if (StringUtils.isEmpty(provinceCrForm.getParentId())) {
            entity.setParentId("-1");
        }
        provinceService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新
     *
     * @param id             主键值
     * @param provinceUpForm 实体对象
     * @return
     */
    @Operation(summary="修改行政区划")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid ProvinceUpForm provinceUpForm) {
        ProvinceEntity entity = JsonUtil.getJsonToBean(provinceUpForm, ProvinceEntity.class);
        if (provinceService.isExistByEnCode(provinceUpForm.getEnCode(), id)) {
            return ActionResult.fail("区域编码不能重复");
        }
        boolean flag = provinceService.update(id, entity);
        if (!flag) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="删除")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        // 判断是否有子节点
        if (provinceService.getListByParentId(id).size() == 0) {
            ProvinceEntity entity = provinceService.getInfo(id);
            if (entity != null) {
                provinceService.delete(entity);
                return ActionResult.success(MsgCode.SU003.get());
            }
            return ActionResult.fail(MsgCode.FA003.get());
        } else {
            return ActionResult.fail("删除失败，当前有子节点数据");
        }
    }

    /**
     * 更新行政区划状态
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="更新行政区划状态")
    @PutMapping("/{id}/Actions/State")
    public ActionResult upState(@PathVariable("id") String id) {
        ProvinceEntity entity = provinceService.getInfo(id);
        if (entity.getEnabledMark() == null || "1".equals(String.valueOf(entity.getEnabledMark()))) {
            entity.setEnabledMark(0);
        } else {
            entity.setEnabledMark(1);
        }
        boolean flag = provinceService.update(id, entity);
        if (!flag) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 行政区划id转名称
     *
     * @param model 二维数组
     * @return
     */
    @Operation(summary="行政区划id转名称")
    @PostMapping("/GetAreaByIds")
    public ActionResult getAreaByIds(@RequestBody AreaModel model) {
        // 返回给前端的list
        List<List<String>> list = new LinkedList<>();
        for (List<String> idList : model.getIdsList()) {
            //List<String> collList=Arrays.asList(idList.toString().replace("[","").replace("]","").split(","));
            List<ProvinceEntity> proList = provinceService.getProList(idList);
            List<String> collect = proList.stream().map(ProvinceEntity::getFullName).collect(Collectors.toList());
            list.add(collect);
        }
        return ActionResult.success(list);
    }

    /**
     * 行政区划id转名称
     *
     * @param idList 一维数组
     * @return
     */
    @Operation(summary="行政区划id转名称")
    @PostMapping("/getAreaFullNameByIds")
    public ActionResult getAreaFullNameByIds(@RequestBody List<String> idList) {
        if (CollectionUtil.isEmpty(idList)) {
            return ActionResult.success(new ArrayList<>());
        }

        // 返回给前端的list
        List<List<String>> list = new LinkedList<>();
        List<ProvinceEntity> provinceEntities = provinceService.listByIds(idList);
        provinceEntities.stream().forEach(t -> {
            if (StringUtils.isNotEmpty(t.getParentIdTree())) {
                List<String> ids = Arrays.asList(t.getParentIdTree().split(","));
                List<ProvinceEntity> proList = provinceService.getProList(ids);
                List<String> collect = proList.stream().map(ProvinceEntity::getFullName).collect(Collectors.toList());
                list.add(collect);
            }
        });
        return ActionResult.success(list);
    }
}
