package com.bringspring.system.msgcenter.service.strategy.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.msgcenter.constant.Constants.Delimiters;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.entity.McMsgTemplateEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.enums.EnabledMarkEnum;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.model.message.MyEmailModel;
import com.bringspring.system.msgcenter.service.context.SendMessageContext;
import com.bringspring.system.msgcenter.service.strategy.AbstractMessageSender;
import com.bringspring.system.msgcenter.util.BlacklistUtil;
import com.bringspring.system.msgcenter.util.EmailUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

/**
 * 消息中心-电子邮件下发工具类
 */
@Component
@Slf4j
public class MsgToEmailService extends AbstractMessageSender {

    @Override
    protected int getBatchSize() {
        return 1000;
    }

    @Override
    public ChannelTypeEnum getChannelType() {
        return ChannelTypeEnum.MAIL_MSG;
    }


    @Override
    protected void validateParameters(SendMessageContext context) {
        validateTemplate(context);
        validateAccount(context);
    }

    @Override
    protected void doSend(SendMessageContext context) {
        if (CollectionUtil.isNotEmpty(context.getValidList())) {
            List<McTaskMsgReceiveEntity> receiveList = context.getValidList();
            // 预处理：应用账号配置
            McMsgAccountEntity account = context.getMsgAccount();
            Map<String, String> objModel = new HashMap<>();
            objModel.put("emailSmtpHost", account.getSmtpServer());
            objModel.put("emailSmtpPort", account.getSmtpPort().toString());
            objModel.put("emailSenderName", account.getAddressorName());
            objModel.put("emailAccount", account.getSmtpUser());
            objModel.put("emailPassword", account.getSmtpPassword());
            objModel.put("emailSsl", account.getSslLink().equals("1") ? "true" : "false");

            // 预处理：消息内容
            McTaskMsgContentEntity taskMsgContent = context.getTaskMsgContent();

            // 预处理：接收人记录
            final List<String> emailList = receiveList.stream().map(r -> r.getReceiveUserId()).collect(toList());
            final String toUserMail = StringUtils.join(emailList, Delimiters.COMMA);

            MyEmailModel emailModel = JsonUtil.getJsonToBean(objModel, MyEmailModel.class);
            // 设置邮件标题
            emailModel.setEmailTitle(taskMsgContent.getTitle());
            // 设置邮件内容
            emailModel.setEmailContent(taskMsgContent.getContent());
            // 设置接收人员
            emailModel.setEmailToUsers(toUserMail);

            List<McTaskMsgReceiveEntity> validList = new ArrayList<>();
            List<McTaskMsgReceiveEntity> invalidList = new ArrayList<>();
            try {

                /**
                 * 调用API 发送消息
                 */
                // 发送邮件
                JSONObject retJson = EmailUtil.sendMail(emailModel);

                /**
                 * 处理发送结果
                 */
                if (!retJson.getBoolean("code")) {
                    // 失败
                    invalidList.addAll(receiveList);
                    invalidList.stream().forEach(e -> {
                        e.setEnabledMark(EnabledMarkEnum.FAIL.getCode());
                        e.setDescription(retJson.getString("error"));
                    });
                } else {
                    // 成功
                    validList.addAll(receiveList);
                    receiveList.stream().forEach(e -> {
                        e.setEnabledMark(EnabledMarkEnum.SUCCESS.getCode());
                        e.setDescription( retJson.getString("error"));
                    });
                }
            } catch (Exception e) {
                e.printStackTrace();
                invalidList.addAll(receiveList);
                invalidList.stream().forEach(r -> {
                    r.setEnabledMark(EnabledMarkEnum.FAIL.getCode());
                    r.setDescription(e.getMessage());
                });
            }

            context.getInvalidList().addAll(invalidList);
            context.setValidList(validList);
        }
    }



    @Override
    protected void sendBatchNonSystemUser(boolean isAgain, McMsgSendTemplateModel sendTemplate,
                                          McTaskMsgContentEntity taskMsgContent,
                                          List<McTaskMsgReceiveEntity> receiveListS, UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 电子邮件消息【非系统用户】下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }
        sendBatch(isAgain, sendTemplate, taskMsgContent, receiveListS, currentUserInfo);
    }

    /**
     * 发送电子邮件
     *
     * @param isAgain        是否重发
     * @param sendTemplate   发送配置模板
     * @param taskMsgContent 业务消息内容
     * @param receiveListS   接收人source
     */
    @Override
    protected void sendBatch(boolean isAgain, McMsgSendTemplateModel sendTemplate,
                             McTaskMsgContentEntity taskMsgContent,
                             List<McTaskMsgReceiveEntity> receiveListS, UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 电子邮件消息下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }
        List<McTaskMsgReceiveEntity> receiveList = JsonUtil.getJsonToList(receiveListS, McTaskMsgReceiveEntity.class);

        // 消息模板
        McMsgTemplateEntity templateInfo = mcMsgTemplateService.getInfo(sendTemplate.getTemplateId());

        // 账号
        McMsgAccountEntity msgAccount = mcMsgAccountService.getInfo(sendTemplate.getAccountConfigId());

        if (ObjectUtil.isNotNull(templateInfo) && ObjectUtil.isNotNull(msgAccount)) {

            // 黑名单成员
            List<McTaskMsgReceiveEntity> receiveBlacklist = BlacklistUtil.receiveListFilter(receiveList);

            if (CollectionUtil.isNotEmpty(receiveList)) {
                // 无电子邮箱
                List<McTaskMsgReceiveEntity> emailNullList =
                        receiveList.stream().filter(r -> StringUtils.isEmpty(r.getReceiveUserId())).collect(Collectors.toList());
                // 有电子邮箱
                List<McTaskMsgReceiveEntity> emailNotNullList =
                        receiveList.stream().filter(r -> StringUtils.isNotEmpty(r.getReceiveUserId())).collect(Collectors.toList());
                List<String> emailList =
                        emailNotNullList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(emailList)) {
                    //账号配置——电子邮箱
                    Map<String, String> objModel = new HashMap<>();
                    objModel.put("emailSmtpHost", msgAccount.getSmtpServer());
                    objModel.put("emailSmtpPort", msgAccount.getSmtpPort().toString());
                    objModel.put("emailSenderName", msgAccount.getAddressorName());
                    objModel.put("emailAccount", msgAccount.getSmtpUser());
                    objModel.put("emailPassword", msgAccount.getSmtpPassword());
                    objModel.put("emailSsl", msgAccount.getSslLink().equals("1") ? "true" : "false");
                    MyEmailModel emailModel = JsonUtil.getJsonToBean(objModel, MyEmailModel.class);
                    // 设置邮件标题
                    emailModel.setEmailTitle(taskMsgContent.getTitle());
                    // 设置邮件内容
                    emailModel.setEmailContent(taskMsgContent.getContent());
                    // 获取收件人的邮箱地址
                    String toUserMail = StringUtils.join(emailList, Delimiters.COMMA);
                    // 设置接收人员
                    emailModel.setEmailToUsers(toUserMail);

                    /**
                     * 调用API 发送消息
                     */
                    // 发送邮件
                    JSONObject retJson = EmailUtil.sendMail(emailModel);

                    /**
                     * 处理发送结果
                     */
                    if (!retJson.getBoolean("code")) {
                        // 失败
                        String errMsg = (isAgain ? "重发失败：" : "下发失败：") + retJson.getString("error");
                        mcTaskMsgReceiveService.updateByList(isAgain, emailNotNullList, 2, errMsg, currentUserInfo);
                    } else {
                        // 成功
                        String errMsg = (isAgain ? "重发成功：" : "下发成功：") + retJson.getString("error");
                        mcTaskMsgReceiveService.updateByList(isAgain, emailNotNullList, 1, errMsg, currentUserInfo);
                    }
                }

                if (CollectionUtil.isNotEmpty(emailNullList)) {
                    String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "无有效电子邮箱";
                    mcTaskMsgReceiveService.updateByList(isAgain, emailNullList, 2, errMsg, currentUserInfo);
                }
            }
            if (CollectionUtil.isNotEmpty(receiveBlacklist)) {
                // 黑名单
                mcTaskMsgReceiveService.updateBlacklist(receiveBlacklist, currentUserInfo);
            }
        }
    }

}