package com.bringspring.system.msgcenter.service.strategy.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.msgcenter.constant.Constants.Delimiters;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.entity.McMsgTemplateEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.enums.EnabledMarkEnum;
import com.bringspring.system.msgcenter.model.MySmsModel;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.service.context.SendMessageContext;
import com.bringspring.system.msgcenter.service.strategy.AbstractMessageSender;
import com.bringspring.system.msgcenter.util.BlacklistUtil;
import com.bringspring.system.msgcenter.util.SmsSentUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.bringspring.system.msgcenter.constant.Constants.MessageConfig;
import static java.util.stream.Collectors.toList;

/**
 * 消息中心-短信下发工具类
 */
@Component
@Slf4j
public class MsgToSmsService extends AbstractMessageSender {

    @Override
    protected int getBatchSize() {
        return 1000;
    }

    @Override
    public ChannelTypeEnum getChannelType() {
        return ChannelTypeEnum.SMS_MSG;
    }

    @Override
    protected void validateParameters(SendMessageContext context) {
        validateTemplate(context);
        validateAccount(context);
    }

    @Override
    protected void doSend(SendMessageContext context) {
        if (CollectionUtil.isNotEmpty(context.getValidList())) {
            List<McTaskMsgReceiveEntity> receiveList = context.getValidList();
            // 预处理：应用账号配置
            McMsgAccountEntity account = context.getMsgAccount();
            String zoneParam = account.getZoneParam();
            String smsSignature = account.getSmsSignature();
            Map<String, String> objModel = new HashMap<>(16);
            objModel.put("aliAccessKey", account.getAppId());
            objModel.put("aliSecret", account.getAppSecret());
            objModel.put("tencentSecretId", account.getAppId());
            objModel.put("tencentSecretKey", account.getAppSecret());
            objModel.put("tencentAppId", account.getSdkAppId());
            objModel.put("tencentAppKey", account.getAppKey());
            MySmsModel smsConfig = JsonUtil.getJsonToBean(objModel, MySmsModel.class);
            int company = Integer.valueOf(account.getChannel());
            String endPoint = "";
            String channelStr = String.valueOf(company);
            if ("1".equals(channelStr)) {
                endPoint = account.getEndPoint();
            } else if ("2".equals(channelStr)) {
                endPoint = account.getZoneName();
            }

            // 预处理：消息模板
            McMsgTemplateEntity templateInfo = context.getTemplateInfo();
            String templateCode = templateInfo.getTemplateCode();

            // 预处理：消息内容
            McTaskMsgContentEntity taskMsgContent = context.getTaskMsgContent();
            String content = taskMsgContent.getContent();
            Map<String, Object> msgMap = JsonUtil.stringToMap(content);

            // 预处理：接收人记录
            final List<String> phoneList = receiveList.stream().map(r -> r.getReceiveUserId()).collect(toList());
            final String phoneNumbers = StringUtils.join(phoneList, Delimiters.COMMA);

            List<McTaskMsgReceiveEntity> validList = new ArrayList<>();
            List<McTaskMsgReceiveEntity> invalidList = new ArrayList<>();
            try {

                /**
                 * 调用API 发送消息
                 */
                String res = SmsSentUtil.sentSms(company, smsConfig, endPoint, zoneParam,
                        phoneNumbers, smsSignature, templateCode, msgMap);

                /**
                 * 处理发送结果
                 */
                if (MessageConfig.SMS_OK.equals(res)) {
                    // 成功
                    validList.addAll(receiveList);
                    receiveList.stream().forEach(e -> {
                        e.setEnabledMark(EnabledMarkEnum.SUCCESS.getCode());
                        e.setDescription(res);
                    });
                } else {
                    // 失败
                    invalidList.addAll(receiveList);
                    invalidList.stream().forEach(e -> {
                        e.setEnabledMark(EnabledMarkEnum.FAIL.getCode());
                        e.setDescription(res);
                    });
                }

            } catch (Exception e) {
                e.printStackTrace();
                invalidList.addAll(receiveList);
                invalidList.stream().forEach(r -> {
                    r.setEnabledMark(EnabledMarkEnum.FAIL.getCode());
                    r.setDescription(e.getMessage());
                });
            }

            context.getInvalidList().addAll(invalidList);
            context.setValidList(validList);
        }
    }


    @Override
    protected void sendBatchNonSystemUser(boolean isAgain, McMsgSendTemplateModel sendTemplate,
                                          McTaskMsgContentEntity taskMsgContent,
                                          List<McTaskMsgReceiveEntity> receiveListS, UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 短信消息【非系统用户】下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }
        sendBatch(isAgain, sendTemplate, taskMsgContent, receiveListS, currentUserInfo);
    }

    /**
     * 发送短信
     *
     * @param isAgain        是否重发
     * @param sendTemplate   发送配置模板
     * @param taskMsgContent 业务消息内容
     * @param receiveListS   接收人source
     */
    @Override
    protected void sendBatch(boolean isAgain, McMsgSendTemplateModel sendTemplate,
                             McTaskMsgContentEntity taskMsgContent,
                             List<McTaskMsgReceiveEntity> receiveListS, UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 短信消息下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }
        List<McTaskMsgReceiveEntity> receiveList = JsonUtil.getJsonToList(receiveListS, McTaskMsgReceiveEntity.class);

        // 消息模板
        McMsgTemplateEntity templateInfo = mcMsgTemplateService.getInfo(sendTemplate.getTemplateId());

        // 账号
        McMsgAccountEntity msgAccount = mcMsgAccountService.getInfo(sendTemplate.getAccountConfigId());

        if (ObjectUtil.isNotNull(templateInfo) && ObjectUtil.isNotNull(msgAccount)) {
            // 黑名单成员
            List<McTaskMsgReceiveEntity> receiveBlacklist = BlacklistUtil.receiveListFilter(receiveList);

            if (CollectionUtil.isNotEmpty(receiveList)) {
                // 无手机号
                List<McTaskMsgReceiveEntity> phoneNullList =
                        receiveList.stream().filter(r -> StringUtils.isEmpty(r.getReceiveUserId())).collect(Collectors.toList());
                // 有手机号
                List<McTaskMsgReceiveEntity> phoneNotNullList =
                        receiveList.stream().filter(r -> StringUtils.isNotEmpty(r.getReceiveUserId())).collect(Collectors.toList());
                List<String> phoneList =
                        phoneNotNullList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(phoneList)) {
                    // 组装接受用户
                    String phoneNumbers = StringUtils.join(phoneList, Delimiters.COMMA);

                    //账号配置——短信
                    Map<String, String> objModel = new HashMap<>(16);
                    objModel.put("aliAccessKey", msgAccount.getAppId());
                    objModel.put("aliSecret", msgAccount.getAppSecret());
                    objModel.put("tencentSecretId", msgAccount.getAppId());
                    objModel.put("tencentSecretKey", msgAccount.getAppSecret());
                    objModel.put("tencentAppId", msgAccount.getSdkAppId());
                    objModel.put("tencentAppKey", msgAccount.getAppKey());
                    MySmsModel smsConfig = JsonUtil.getJsonToBean(objModel, MySmsModel.class);
                    int company = Integer.valueOf(msgAccount.getChannel());

                    //获取消息模板参数
                    String content = taskMsgContent.getContent();
                    Map<String, Object> msgMap = JsonUtil.stringToMap(content);
                    // 短信模板
                    String templateCode = templateInfo.getTemplateCode();
                    String endPoint = "";
                    String channelStr = String.valueOf(msgAccount.getChannel());
                    if ("1".equals(channelStr)) {
                        endPoint = msgAccount.getEndPoint();
                    } else if ("2".equals(channelStr)) {
                        endPoint = msgAccount.getZoneName();
                    }

                    /**
                     * 调用API 发送消息
                     */
                    String res = SmsSentUtil.sentSms(company, smsConfig, endPoint, msgAccount.getZoneParam(),
                            phoneNumbers, msgAccount.getSmsSignature(), templateCode, msgMap);

                    /**
                     * 处理发送结果
                     */
                    if (MessageConfig.SMS_OK.equals(res)) {
                        // 成功
                        String errMsg = (isAgain ? "重发成功：" : "下发成功：") + res;
                        mcTaskMsgReceiveService.updateByList(isAgain, phoneNotNullList, 1, errMsg, currentUserInfo);
                    } else {
                        // 失败
                        String errMsg = (isAgain ? "重发失败：" : "下发失败：") + res;
                        mcTaskMsgReceiveService.updateByList(isAgain, phoneNotNullList, 2, errMsg, currentUserInfo);
                    }
                }

                if (CollectionUtil.isNotEmpty(phoneNullList)) {
                    String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "无有效手机号";
                    mcTaskMsgReceiveService.updateByList(isAgain, phoneNullList, 2, errMsg, currentUserInfo);
                }

            }
            if (CollectionUtil.isNotEmpty(receiveBlacklist)) {
                // 黑名单
                mcTaskMsgReceiveService.updateBlacklist(receiveBlacklist, currentUserInfo);
            }
        }
    }

}