package com.bringspring.system.external.controller;

import com.bringspring.system.base.service.SysConfigService;
import com.bringspring.system.external.config.mutil.WxCpConfiguration;
import com.bringspring.system.external.util.JsonUtils;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import me.chanjar.weixin.common.error.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.bean.message.WxCpXmlMessage;
import me.chanjar.weixin.cp.bean.message.WxCpXmlOutMessage;
import me.chanjar.weixin.cp.util.crypto.WxCpCryptUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;


/**
 * 企业微信回调
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2024年03月05日 上午9:18
 */
@Slf4j
@Tag(name = "企业微信回调")
@RestController
@RequestMapping("/api/white/callback/wxcp")
public class WxCpCallBackController {

    @Autowired
    private SysConfigService sysConfigService;

    @Autowired
    WxCpConfiguration wxCpConfiguration;


    @PostMapping("/test")
    public String test(@PathVariable String corpId,
                       @PathVariable Integer agentId,
                       @RequestParam(name = "userId", required = true) String userId) throws WxErrorException {

        final val wxCpService = wxCpConfiguration.getCpService(corpId, agentId);
        List<String> list = wxCpService.getExternalContactService().listExternalContacts(userId);

        return list.toString();
    }

    @GetMapping(value="/{corpId}/{agentId}", produces = "text/plain;charset=utf-8")
    public String authGet(@PathVariable String corpId,
                          @PathVariable Integer agentId,
                          @RequestParam(name = "msg_signature", required = false) String signature,
                          @RequestParam(name = "timestamp", required = false) String timestamp,
                          @RequestParam(name = "nonce", required = false) String nonce,
                          @RequestParam(name = "echostr", required = false) String echostr) {
        log.info("\n接收到来自微信服务器的认证消息：signature = [{}], timestamp = [{}], nonce = [{}], echostr = [{}]",
                signature, timestamp, nonce, echostr);

        if (org.apache.commons.lang3.StringUtils.isAnyBlank(signature, timestamp, nonce, echostr)) {
            throw new IllegalArgumentException("请求参数非法，请核实!");
        }

        final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, agentId);
//        final WxCpService wxCpService = wxCpConfiguration.getCorpService(corpId);
        if (wxCpService == null) {
            throw new IllegalArgumentException(String.format("未找到对应agentId=[%d]的配置，请核实！", agentId));
        }

        if (wxCpService.checkSignature(signature, timestamp, nonce, echostr)) {
            return new WxCpCryptUtil(wxCpService.getWxCpConfigStorage()).decrypt(echostr);
        }

        return "非法请求";
    }

    @PostMapping(value="/{corpId}/{agentId}", produces = "application/xml; charset=UTF-8")
    public String post(@PathVariable String corpId,
                       @PathVariable Integer agentId,
                       @RequestBody String requestBody,
                       @RequestParam("msg_signature") String signature,
                       @RequestParam("timestamp") String timestamp,
                       @RequestParam("nonce") String nonce) {
        log.info("\n接收微信请求：[signature=[{}], timestamp=[{}], nonce=[{}], requestBody=[\n{}\n] ",
                signature, timestamp, nonce, requestBody);

        final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, agentId);
        if (wxCpService == null) {
            throw new IllegalArgumentException(String.format("未找到对应agentId=[%d]的配置，请核实！", agentId));
        }

        WxCpXmlMessage inMessage = WxCpXmlMessage.fromEncryptedXml(requestBody, wxCpService.getWxCpConfigStorage(),
                timestamp, nonce, signature);
        log.info("\n消息解密后内容为：\n{} ", JsonUtils.toJson(inMessage));

        WxCpXmlOutMessage outMessage = this.route(corpId, agentId, inMessage);
        if (outMessage == null) {
            return "";
        }

        String out = outMessage.toEncryptedXml(wxCpService.getWxCpConfigStorage());
        log.info("\n组装回复信息：{}", out);
        return out;
    }

    private WxCpXmlOutMessage route(String corpId, Integer agentId, WxCpXmlMessage message) {
        try {
            return wxCpConfiguration.getRouters().get(corpId + agentId).route(message);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

}
