package com.bringspring.system.base.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.base.entity.ComFieldsEntity;
import com.bringspring.system.base.model.comfields.ComFieldsCrForm;
import com.bringspring.system.base.model.comfields.ComFieldsInfoVO;
import com.bringspring.system.base.model.comfields.ComFieldsListVO;
import com.bringspring.system.base.model.comfields.ComFieldsUpForm;
import com.bringspring.system.base.service.ComFieldsService;
import com.bringspring.system.base.util.JsonUtilEx;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 常用字段
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/12 15:31
 */
@Api(tags = "常用字段", value = "CommonFields")
@RestController
@RequestMapping("/api/system/CommonFields")
public class ComFieldsController {

    @Autowired
    private ComFieldsService comFieldsService;

    /**
     * 获取常用字段列表
     *
     * @return ignore
     */
    @ApiOperation("获取常用字段列表")
    @GetMapping
    public ActionResult<ListVO<ComFieldsListVO>> list() {
        List<ComFieldsEntity> data = comFieldsService.getList();
        List<ComFieldsListVO> list = JsonUtil.getJsonToList(data, ComFieldsListVO.class);
        ListVO<ComFieldsListVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(vo);
    }

    /**
     * 获取常用字段
     *
     * @param id 主键
     * @return ignore
     */
    @ApiOperation("获取常用字段")
    @GetMapping("/{id}")
    public ActionResult<ComFieldsInfoVO> info(@PathVariable("id") String id) throws DataException {
        ComFieldsEntity entity = comFieldsService.getInfo(id);
        ComFieldsInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, ComFieldsInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 新增常用字段
     *
     * @param comFieldsCrForm 新增常用字段模型
     * @return ignore
     */
    @ApiOperation("添加常用字段")
    @PostMapping
    public ActionResult create(@RequestBody @Valid ComFieldsCrForm comFieldsCrForm) {
        ComFieldsEntity entity = JsonUtil.getJsonToBean(comFieldsCrForm, ComFieldsEntity.class);
        if (comFieldsService.isExistByFullName(entity.getField(), entity.getId())) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        comFieldsService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 修改常用字段
     *
     * @param id              主键
     * @param comFieldsUpForm 修改常用字段模型
     * @return ignore
     */
    @ApiOperation("修改常用字段")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid ComFieldsUpForm comFieldsUpForm) {
        ComFieldsEntity entity = JsonUtil.getJsonToBean(comFieldsUpForm, ComFieldsEntity.class);
        if (comFieldsService.isExistByFullName(entity.getField(), id)) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        boolean flag = comFieldsService.update(id, entity);
        if (!flag) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除常用字段
     *
     * @param id 主键
     * @return ignore
     */
    @ApiOperation("删除常用字段")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        ComFieldsEntity entity = comFieldsService.getInfo(id);
        if (entity != null) {
            comFieldsService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }
}

