package com.bringspring.system.base.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.database.model.entity.DbLinkEntity;
import com.bringspring.common.database.model.DataSourceModel;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.util.UserProvider;
import com.bringspring.common.util.XSSEscape;
import com.bringspring.common.util.enums.DictionaryDataEnum;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.DictionaryDataEntity;
import com.bringspring.system.base.model.dblink.*;
import com.bringspring.system.base.service.DblinkService;
import com.bringspring.system.base.service.DictionaryDataService;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.UserService;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 数据连接
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Api(tags = "数据连接", value = "DataSource")
@RestController
@RequestMapping("/api/system/DataSource")
public class DbLinkController {

    @Autowired
    private DblinkService dblinkService;
    @Autowired
    private UserService userService;
    @Autowired
    private DictionaryDataService dictionaryDataService;
    @Autowired
    private DataSourceModel dataSourceModel;
    @Autowired
    private BaseDataUtil baseDataUtil;
    @Autowired
    private UserProvider userProvider;

    /**
     * 列表
     *
     * @return ignore
     */
    @GetMapping("/Selector")
    @ApiOperation("获取数据连接下拉框列表")
    public ActionResult<ListVO<DbLinkSelectorListVO>> selectorList(String type) {
        List<DbLinkModel> modelAll = new LinkedList<>();
        //将主节点添加到容器当中=======
        List<DbLinkEntity> data = dblinkService.getList();
        //连接中类型所对应的字典集合
        List<DictionaryDataEntity> dataEntityList = dictionaryDataService.getList(DictionaryDataEnum.SYSTEM_DBLINK.getDictionaryTypeId());
        for (DictionaryDataEntity entity : dataEntityList) {
            DbLinkModel model = new DbLinkModel();
            model.setFullName(entity.getFullName());
            model.setParentId("-1");
            model.setId(entity.getId());
            Long num = data.stream().filter(t -> t.getDbType().equals(entity.getEnCode())).count();
            model.setNum(num);
            if (num > 0) {
                modelAll.add(model);
            }
        }
        //字典里存在则添加到容器当中=======
        for (DbLinkEntity entity : data) {
            DbLinkModel model = JsonUtil.getJsonToBean(entity, DbLinkModel.class);
            DictionaryDataEntity dataEntity = dataEntityList.stream().filter(t -> t.getEnCode().equals(entity.getDbType())).findFirst().orElse(null);
            if (dataEntity != null) {
                model.setParentId(dataEntity.getId());
                modelAll.add(model);
            }
        }
        List<SumTree<DbLinkModel>> trees = TreeDotUtils.convertListToTreeDot(modelAll);
        List<DbLinkSelectorListVO> list = new ArrayList<>();
        // type不为空时不返回默认库
        if(type == null){
            DbLinkSelectorListVO defaultDb = new DbLinkSelectorListVO();
            defaultDb.setFullName("");
            DbLinkListVO dbLink = new DbLinkListVO();
            dbLink.setFullName("默认数据库");
            dbLink.setId("0");
            dbLink.setDbType(dataSourceModel.getDbType());
            defaultDb.setChildren(Collections.singletonList(dbLink));
            list.add(defaultDb);
        }
        list.addAll(JsonUtil.getJsonToList(trees, DbLinkSelectorListVO.class));
        ListVO<DbLinkSelectorListVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(vo);
    }



    /**
     * 列表
     *
     * @return ignore
     */
    @PostMapping("/ByPermissionSelector")
    @ApiOperation("根据数据权限获取数据连接下拉框列表")
    public ActionResult<ListVO<DbLinkSelectorListVO>> byPermissionSelectorList(PaginationDbLink page) {
        UserInfo userInfo = userProvider.get();
        List<DbLinkModel> modelAll = new LinkedList<>();
        //将主节点添加到容器当中=======
        List<DbLinkEntity> data = dblinkService.getListByPermission(page);
        //连接中类型所对应的字典集合
        List<DictionaryDataEntity> dataEntityList = dictionaryDataService.getList(DictionaryDataEnum.SYSTEM_DBLINK.getDictionaryTypeId());
        for (DictionaryDataEntity entity : dataEntityList) {
            DbLinkModel model = new DbLinkModel();
            model.setFullName(entity.getFullName());
            model.setParentId("-1");
            model.setId(entity.getId());
            Long num = data.stream().filter(t -> t.getDbType().equals(entity.getEnCode())).count();
            model.setNum(num);
            if (num > 0) {
                modelAll.add(model);
            }
        }
        //字典里存在则添加到容器当中=======
        for (DbLinkEntity entity : data) {
            DbLinkModel model = JsonUtil.getJsonToBean(entity, DbLinkModel.class);
            DictionaryDataEntity dataEntity = dataEntityList.stream().filter(t -> t.getEnCode().equals(entity.getDbType())).findFirst().orElse(null);
            if (dataEntity != null) {
                model.setParentId(dataEntity.getId());
                modelAll.add(model);
            }
        }
        List<SumTree<DbLinkModel>> trees = TreeDotUtils.convertListToTreeDot(modelAll);
        List<DbLinkSelectorListVO> list = new ArrayList<>();
        // 是超级管理员
        if(userInfo.getIsAdministrator()){
            DbLinkSelectorListVO defaultDb = new DbLinkSelectorListVO();
            defaultDb.setFullName("");
            DbLinkListVO dbLink = new DbLinkListVO();
            dbLink.setFullName("默认数据库");
            dbLink.setId("0");
            dbLink.setDbType(dataSourceModel.getDbType());
            defaultDb.setChildren(Collections.singletonList(dbLink));
            list.add(defaultDb);
        }
        list.addAll(JsonUtil.getJsonToList(trees, DbLinkSelectorListVO.class));
        ListVO<DbLinkSelectorListVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(vo);
    }

    /**
     * 列表
     *
     * @param page 关键字
     * @return ignore
     */
    @GetMapping
    @ApiOperation("获取数据连接列表")
    public ActionResult getList(PaginationDbLink page) {
        List<DbLinkEntity> data = dblinkService.getList(page);
        List<String> userId = data.stream().map(t -> t.getCreatorUserId()).collect(Collectors.toList());
        List<String> lastUserId = data.stream().map(t -> t.getLastModifyUserId()).collect(Collectors.toList());
        List<UserEntity> userEntities = userService.getUserName(userId);
        List<UserEntity> lastUserIdEntities = userService.getUserName(lastUserId);
        List<DictionaryDataEntity> typeList = dictionaryDataService.getList(DictionaryDataEnum.SYSTEM_DBLINK.getDictionaryTypeId());
        List<DbLinkListVO> jsonToList = JsonUtil.getJsonToList(data, DbLinkListVO.class);
        for (DbLinkListVO vo : jsonToList) {
            //存在类型的字典对象
            DictionaryDataEntity dataEntity = typeList.stream().filter(t -> t.getEnCode().equals(vo.getDbType())).findFirst().orElse(null);
            if (dataEntity != null) {
                vo.setDbType(dataEntity.getFullName());
            } else {
                vo.setDbType("");
            }
            //创建者
            UserEntity creatorUser = userEntities.stream().filter(t -> t.getId().equals(vo.getCreatorUserId())).findFirst().orElse(null);
            vo.setCreatorUserId(creatorUser != null ? creatorUser.getRealName() : vo.getCreatorUserId());
            //修改人
            UserEntity lastModifyUser = lastUserIdEntities.stream().filter(t -> t.getId().equals(vo.getLastModifyUserId())).findFirst().orElse(null);
            vo.setLastModifyUserId(lastModifyUser != null ? lastModifyUser.getRealName() : vo.getLastModifyUserId());
            //所属公司
            if (StringUtils.isNotEmpty(vo.getCompanyId())){
                vo.setCompanyId(baseDataUtil.comSelectValues(vo.getCompanyId()));
            }
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(page, PaginationVO.class);
        return ActionResult.page(jsonToList , paginationVO);
    }

    /**
     * 单条数据连接
     *
     * @param id 主键
     * @return ignore
     * @throws DataException ignore
     */
    @GetMapping("/{id}")
    @ApiOperation("获取数据连接")
    public ActionResult<DbLinkInfoVO> get(@PathVariable("id") String id) throws DataException {
        return ActionResult.success(new DbLinkInfoVO().getDbLinkInfoVO(dblinkService.getInfo(XSSEscape.escape(id))));
    }

    /**
     * 新建数据连接
     *
     * @param dbLinkCreUpForm 新建数据连接表单对象
     * @return ignore
     */
    @PostMapping
    @ApiOperation("添加数据连接")
    public ActionResult<String> create(@RequestBody @Valid DbLinkCreUpForm dbLinkCreUpForm) {
        DbLinkEntity entity = dbLinkCreUpForm.getDbLinkEntity(dbLinkCreUpForm);
        if (dblinkService.isExistByFullName(entity.getFullName(), entity.getId())) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        dblinkService.create(entity);
        return ActionResult.success("创建成功");
    }

    /**
     * 更新数据连接
     *
     * @param id              主键
     * @param dbLinkCreUpForm dto实体
     * @return ignore
     */
    @PutMapping("/{id}")
    @ApiOperation("修改数据连接")
    public ActionResult<String> update(@PathVariable("id") String id, @RequestBody @Valid DbLinkCreUpForm dbLinkCreUpForm) {
        id = XSSEscape.escape(id);
        DbLinkEntity entity = dbLinkCreUpForm.getDbLinkEntity(dbLinkCreUpForm);
        if (dblinkService.isExistByFullName(entity.getFullName(), id)) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        if (!dblinkService.update(id, entity)) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除
     *
     * @param id 主键
     * @return ignore
     */
    @DeleteMapping("/{id}")
    @ApiOperation("删除数据连接")
    public ActionResult<String> delete(@PathVariable("id") String id) {
        DbLinkEntity entity = dblinkService.getInfo(id);
        if (entity != null) {
            dblinkService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    /**
     * 测试连接
     *
     * @param dbLinkBaseForm 数据连接参数
     * @return ignore
     * @throws DataException ignore
     */
    @PostMapping("/Actions/Test")
    @ApiOperation("测试连接")
    public ActionResult<String> test(@RequestBody DbLinkBaseForm dbLinkBaseForm) throws DataException {
        boolean data = dblinkService.testDbConnection(dbLinkBaseForm.getDbLinkEntity(dbLinkBaseForm));
        if (data) {
            return ActionResult.success("连接成功");
        } else {
            return ActionResult.fail("连接失败");
        }
    }
}
