package com.bringspring.system.external.util;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.external.bean.linkedcorp.MyWxCpLinkedCorpUser;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.permission.entity.*;
import com.bringspring.system.permission.service.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.bringspring.system.external.constant.WxCpSysConfigConsts.REGEX_CHAR;
import static com.bringspring.system.external.constant.WxCpSysConfigConsts.TARGET_CHAR;
import static com.bringspring.system.message.util.SynThirdConsts.DATA_TYPE_USER;
import static com.bringspring.system.message.util.SynThirdConsts.THIRD_TYPE_QY;
import static com.bringspring.system.permission.constant.PermissionConst.*;

/**
 * 公共校验
 */
@Component
public class ValidateCommonUtil {

    /**
     * 第一步: 接口返回组织部门成员.
     * ## 通过[互联企业]功能,获取应用的可见范围,遍历可见范围获取部门列表,遍历部门获取部门成员.
     * <p>
     * 第二步: 取系统账号account.
     * ## account取不为空的mobile(手机号)>email(邮箱)>userid.
     * <p>
     * 第三步: 判断成员是否新增.
     * ## 标准1:表base_sync_third_info中不存在third_object_id即为新增,否则为更新;
     * ##   新增new一个SynThirdInfoEntity,更新则获取系统SynThirdInfoEntity.
     * ## 注意:third_object_id的规则是corpid/userid.
     * ## 标准2:表base_user中不存在account即为新增,否则为更新;
     * ##   新增new一个UserEntity,更新则回去系统UserEntity.
     * <p>
     * 第四步: 用户所属组织和部门信息处理.
     * ## 用户在一个企业组织内可归属多个部门
     * 注意:验证系统是否存在组织部门信息,不存在 不建立用户关系表.
     * <p>
     * 第五步: 用户岗位信息处理.
     * ## 不同组织的用户信息都有一个岗位信息,系统岗位配置和部门关联
     * ## 所以 需将企业微信返回岗位名称转化为 position/deptId1,position/deptId2,.
     * <p>
     * 第六步: 工号信息处理
     * ## 企业微信返回的用户信息,扩展属性extattr或有工号信息,需解析出来.
     * <p>
     * 第七步: 新增UserEntity设置默认信息,更新则忽略.
     * ## 角色\头像\性别\有效标志\密钥\是否管理员\账户密码等
     * <p>
     * 第八步: 判断成员账户account是否重复.
     * ## 判断账户account在已经遍历过的成员信息中是否存在,存在账号合并,不存在则put.
     * ## 企业微信中 一个用户可归属多个公司组织,
     * ## 所以third_object_id与system_object_id是多对一的关系(一个用户可归属多个组织).
     * 注意:账号合并即合并组织,合并岗位等信息.设置表base_sync_third_info与表base_user多对一关系.
     * <p>
     * 第九步: 岗位信息持久化转换
     * ## 经过前边八步处理,[互联企业]接口返回的成员信息已全部处理完成,接下来将数据持久化到本地.
     * ## 新增new一个PositionEntity岗位存入数据库.
     * ## 取出positionId,用positionId与user建立关联关系.在base_user_relation用户关系表内.
     * 注意:系统存在要更新的用户,组织和岗位以企业微信为准,系统配置的会丢失.
     * <p>
     * 第十步: 数据批量持久化
     * ## 用户关系表;岗位表;用户表;第三方记录表.
     */

    @Autowired
    private SynThirdInfoService synThirdInfoService;
    @Autowired
    private UserService userService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    PositionService positionService;
    @Autowired
    private RoleService roleService;
    @Autowired
    private UserRelationService userRelationService;
    Map<String, SynThirdInfoEntity> thirdMap = new HashMap<>();
    Map<String, UserEntity> userIdMap = new HashMap<>();
    Map<String, UserEntity> accountMap = new HashMap<>();
    Map<String, UserEntity> phoneMap = new HashMap<>();
    Map<String, UserEntity> emailMap = new HashMap<>();
    Map<String, OrganizeEntity> organizeMap = new HashMap<>();
    Map<String, PositionEntity> positionIdMap = new HashMap<>();
    /**
     * 默认普通员工角色
     */
    String defaultRoleId = "393332333610065861";

    /**
     * 初始化系统数据
     * 用户\组织\岗位\第三方同步记录
     */
    public void initData() {

        // 当前指定数据类型的数据列表 1:企业微信;2:用户
        List<SynThirdInfoEntity> thirdList = synThirdInfoService.getList(THIRD_TYPE_QY, DATA_TYPE_USER);
        if (StringUtils.isNotEmpty(thirdList)) {
            for (SynThirdInfoEntity entity : thirdList) {
                thirdMap.put(entity.getThirdObjectId(), entity);
            }
        }

        // 系统当前用户列表
        List<UserEntity> userList = userService.getList();
        if (StringUtils.isNotEmpty(userList)) {
            for (UserEntity entity : userList) {
                userIdMap.put(entity.getId(), entity);
                accountMap.put(entity.getAccount(), entity);
                phoneMap.put(entity.getMobilePhone(), entity);
                emailMap.put(entity.getEmail(), entity);
            }
        }

        // 系统当前组织列表
        List<OrganizeEntity> organizeList = organizeService.getList();
        if (StringUtils.isNotEmpty(organizeList)) {
            for (OrganizeEntity entity : organizeList) {
                organizeMap.put(entity.getId(), entity);
            }
        }

        // 系统当前岗位列表
        List<PositionEntity> positionList = positionService.getList();
        if (StringUtils.isNotEmpty(positionList)) {
            for (PositionEntity entity : positionList) {
                positionIdMap.put(entity.getFullName() + TARGET_CHAR + entity.getOrganizeId(), entity);
            }
        }

        // 默认角色
        List<RoleEntity> roleList = roleService.getListByEnCode(SIMPLE_USER);
        if (CollectionUtil.isNotEmpty(roleList)) {
            defaultRoleId = roleList.get(0).getId();
        }
    }

    /**
     * 用户默认角色
     *
     * @return
     */
    public String getDefaultRoleId() {
        return defaultRoleId;
    }

    /**
     * 第二步: 取系统账号account.
     * ## account取不为空的mobile(手机号)>email(邮箱)>userid.
     *
     * @param qyuser
     * @return
     */
    public String getAccount(MyWxCpLinkedCorpUser qyuser) {
        if (StringUtils.isNotEmpty(qyuser.getMobile())) {
            return qyuser.getMobile();
        } else if (StringUtils.isNotEmpty(qyuser.getEmail())) {
            return qyuser.getEmail();
        }
        return qyuser.getUserId();
    }

    /**
     * 第三步: 判断成员是否新增.
     * ## 标准1:表base_sync_third_info中不存在third_object_id即为新增,否则为更新;
     * ##   新增new一个SynThirdInfoEntity,更新则回去系统SynThirdInfoEntity.
     * ## 注意:third_object_id的规则是corpid/userid.
     *
     * @param thirdObjId
     * @return
     */
    public SynThirdInfoEntity getThirdEntityByThirdId(String thirdObjId) {
        if (thirdMap.containsKey(thirdObjId)) {
            return thirdMap.get(thirdObjId);
        }
        return null;
    }

    /**
     * 第三步: 判断成员是否新增.
     * ## 标准1:表base_sync_third_info中不存在third_object_id即为新增,否则为更新;
     * ##   新增new一个SynThirdInfoEntity,更新则获取系统SynThirdInfoEntity.
     *
     * @param thirdObjid
     * @return
     */
    public SynThirdInfoEntity newSynThirdInfoForWxUser(String thirdObjid) {
        SynThirdInfoEntity synThirdInfoEntity = new SynThirdInfoEntity();
        synThirdInfoEntity.setId(RandomUtil.uuId());
        synThirdInfoEntity.setThirdType(Integer.valueOf(THIRD_TYPE_QY));
        synThirdInfoEntity.setDataType(Integer.valueOf(DATA_TYPE_USER));
        synThirdInfoEntity.setThirdObjectId(thirdObjid);
        return synThirdInfoEntity;
    }

    /**
     * 获取UserEntity
     *
     * @param account
     * @param phone
     * @param email
     * @return
     */
    public UserEntity getUserEntity(String account, String phone, String email) {
        if (StringUtils.isNotEmpty(account) && accountMap.containsKey(account)) {
            return accountMap.get(account);
        } else if (StringUtils.isNotEmpty(phone) && phoneMap.containsKey(phone)) {
            return phoneMap.get(phone);
        } else if (StringUtils.isNotEmpty(email) && emailMap.containsKey(email)) {
            return emailMap.get(email);
        }
        return null;
    }

    /**
     * 根据系统system_object_id获取UserEntity
     *
     * @param userid
     * @return
     */
    public UserEntity getUserEntityByUserid(String userid) {
        if (userIdMap.containsKey(userid)) {
            return userIdMap.get(userid);
        }
        return null;
    }

    /**
     * 第三步: 判断成员是否新增.
     * ## 标准2:表base_user中不存在account即为新增,否则为更新;
     * ##   新增new一个UserEntity,更新则获取系统UserEntity.
     *
     * @param account
     * @return
     */
    public UserEntity getUserEntityByAccount(String account) {
        if (accountMap.containsKey(account)) {
            return accountMap.get(account);
        }
        return null;
    }

    /**
     * 获取系统deptId
     * 判断是否存在
     *
     * @param deptId
     * @return
     */
    public OrganizeEntity getOrganizeEntityByDeptid(String deptId) {
        if (organizeMap.containsKey(deptId)) {
            return organizeMap.get(deptId);
        }
        return null;
    }

    /**
     * 系统poitionKey
     * 判断是否存在
     *
     * @param poitionKey
     * @return
     */
    public PositionEntity getPositionByUserid(String poitionKey) {
        if (positionIdMap.containsKey(poitionKey)) {
            return positionIdMap.get(poitionKey);
        }
        return null;
    }


    /**
     * 第四步: 用户所属组织和部门信息处理.
     * ## 用户在一个企业组织内可归属多个部门
     * 注意:验证系统是否存在组织部门信息,不存在 不建立用户关系表.
     * <p>
     * 第五步: 用户岗位信息处理.
     * ## 不同组织的用户信息都有一个岗位信息,系统岗位配置和部门关联
     * ## 所以 需将企业微信返回岗位名称转化为 position/deptId1,position/deptId2,.
     *
     * @param qyuser
     * @return
     */

    public void setDeptAndPosition(MyWxCpLinkedCorpUser qyuser, UserEntity userEntity) {
        // 企业微信返回的position是一个岗位名称
        String position = qyuser.getPosition();
        // 企业微信返回的department是一个数组
        if (StringUtils.isNotEmpty(qyuser.getDepartment())) {
            String[] deptList = qyuser.getDepartment();
            String deptIds = "";
            String poitionIds = "";
            for (String cropDept : deptList) {
                String[] cropDeptArr = cropDept.split(TARGET_CHAR);
                String deptId = cropDeptArr[cropDeptArr.length - 1];
                OrganizeEntity organizeEntity = getOrganizeEntityByDeptid(deptId);
                if (ObjectUtil.isNotEmpty(organizeEntity)) {
                    // 系统存在组织信息
                    deptIds += deptId + REGEX_CHAR;
                    if (StringUtils.isNotEmpty(position)) {
                        // 企业微信 岗位信息不为空
                        poitionIds += position + TARGET_CHAR + deptId + REGEX_CHAR;
                    }
                }

            }
            if (StringUtils.isNotEmpty(deptIds)) {
                deptIds = deptIds.substring(0, deptIds.length() - 1);
            }
            if (StringUtils.isNotEmpty(poitionIds)) {
                poitionIds = poitionIds.substring(0, poitionIds.length() - 1);
            }
            userEntity.setOrganizeId(deptIds);
            userEntity.setPositionId(poitionIds);
        } else {
            userEntity.setOrganizeId("");
            userEntity.setOrganizeId("");
        }
    }

    /**
     * 第六步: 工号信息处理
     * ## 企业微信返回的用户信息,扩展属性extattr或有工号信息,需解析出来.
     *
     * @param qyuser
     * @return
     */
    public String getCode(MyWxCpLinkedCorpUser qyuser) {
        String value = null;
        // 企业微信返回的extattr扩展属性中寻找工号进行设置
        if (StringUtils.isNotEmpty(qyuser.getExtattr())) {
            String extattr = qyuser.getExtattr();
            JSONObject jsonObject = JSONObject.parseObject(extattr);
            JSONArray attrs = jsonObject.getJSONArray("attrs");
            if (attrs.size() != 0) {
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    String name = attr.getString("name");
                    if (StringUtils.isNotEmpty(name) && name.contains("工号")) {
                        value = attr.getString("value");
                    }
                }
            }
        }
        return value;
    }

    /**
     * 系统已存在做更新的用户
     * 保留用户关联信息
     *
     * @param userEntity
     */
    public void setUserRelation(UserEntity userEntity) {
        String organizeIds = ""; // 组织
        String positionIds = ""; // 岗位
        String roleIds = ""; // 角色
        String groupIds = ""; // 分组
        // 根据用户主键ID 获取用户关系表数据
        List<UserRelationEntity> relationList = userRelationService.getListByUserId(userEntity.getId());
        if (StringUtils.isNotEmpty(relationList)) {
            for (UserRelationEntity item : relationList) {
                switch (item.getObjectType()) {
                    case ORGANIZE:
                        organizeIds += item.getObjectId() + REGEX_CHAR;
                        break;
                    case POSITION:
                        positionIds += item.getObjectId() + REGEX_CHAR;
                        break;
                    case ROLE:
                        roleIds += item.getObjectId() + REGEX_CHAR;
                        break;
                    case GROUP:
                        groupIds += item.getObjectId() + REGEX_CHAR;
                        break;
                    default:
                        break;
                }
            }
        }
        userEntity.setOrganizeId(organizeIds); // 保留系统配置
        userEntity.setPositionId(positionIds); // 保留系统配置
        userEntity.setRoleId(roleIds); // 保留系统配置
        userEntity.setGroupId(groupIds); // 保留系统配置
    }

    /**
     * 合并去重组织信息
     * 将bEntity的organizeId合并到aEntity的organizeId
     *
     * @param aEntity
     * @param bEntity
     */
    public void mergeDistinctOrganize(UserEntity aEntity, UserEntity bEntity) {
        // 合并组织
        String orgId1 = aEntity.getOrganizeId();
        String[] arr1 = StringUtils.isNotEmpty(orgId1) ? orgId1.split(REGEX_CHAR) : new String[0];
        // userEntity
        String orgId2 = bEntity.getOrganizeId();
        String[] arr2 = StringUtils.isNotEmpty(orgId2) ? orgId2.split(REGEX_CHAR) : new String[0];
        // 合并,去重,set组织
        String[] arrAll = ArrayUtil.addAll(arr1, arr2);
        String[] distinct = ArrayUtil.distinct(arrAll);
        String orgId3 = ArrayUtil.join(distinct, REGEX_CHAR);
        aEntity.setOrganizeId(orgId3);
    }

    /**
     * 合并去重岗位信息
     * 将bEntity的positionId合并到aEntity的positionId
     *
     * @param aEntity
     * @param bEntity
     */
    public void mergeDistinctPosition(UserEntity aEntity, UserEntity bEntity) {
        // 合并岗位
        String posId1 = aEntity.getPositionId();
        String[] posArr1 = StringUtils.isNotEmpty(posId1) ? posId1.split(REGEX_CHAR) : new String[0];
        String posId2 = bEntity.getPositionId();
        // userEntity
        String[] posArr2 = StringUtils.isNotEmpty(posId2) ? posId2.split(REGEX_CHAR) : new String[0];
        // 合并,去重,set岗位
        String[] posArrAll = ArrayUtil.addAll(posArr1, posArr2);
        String[] distinctPos = ArrayUtil.distinct(posArrAll);
        String posId3 = ArrayUtil.join(distinctPos, REGEX_CHAR);
        aEntity.setPositionId(posId3);
    }


    /**
     * 岗位名称转换positionId
     * 第九步: 岗位信息持久化转换
     * ## 经过前边八步处理,[互联企业]接口返回的成员信息已全部处理完成,接下来将数据持久化到本地.
     * ## 新增new一个PositionEntity岗位存入数据库.
     * ## 取出positionId,用positionId与user建立关联关系.在base_user_relation用户关系表内.
     * 注意:系统存在要更新的用户,组织和岗位以企业微信为准,系统配置的会丢失.
     *
     * @param savePositionMap
     * @param userEntity
     */
    public void convertPosition(Map<String, PositionEntity> savePositionMap, UserEntity userEntity) {

        // 岗位名称转换positionId
        String positionId = userEntity.getPositionId();
        String[] posArr = StringUtils.isNotEmpty(positionId) ? positionId.split(REGEX_CHAR) : new String[0];
        int posLength = posArr.length;
        String[] positionIdArr = new String[posLength];
        for (int i = 0; i < posLength; i++) {
            String poitionKey = posArr[i];
            PositionEntity positionEntity = getPositionByUserid(poitionKey);
            if (ObjectUtil.isNotEmpty(positionEntity)) {
                // 存在
                positionIdArr[i] = positionEntity.getId();
            } else {
                if (savePositionMap.containsKey(poitionKey)) {
                    // 存在
                    positionIdArr[i] = savePositionMap.get(poitionKey).getId();
                } else {
                    // 不存在
                    if (poitionKey.contains(TARGET_CHAR)) {
                        String position = poitionKey.substring(0, poitionKey.lastIndexOf(TARGET_CHAR));
                        String deptId = poitionKey.substring(poitionKey.lastIndexOf(TARGET_CHAR) + 1);
                        if (StringUtils.isNotEmpty(position)) {
                            // 企业微信 岗位信息不为空
                            PositionEntity posEntity = new PositionEntity();
                            posEntity.setId(RandomUtil.uuId());
                            posEntity.setFullName(position);
                            posEntity.setOrganizeId(deptId);
                            posEntity.setSortCode(0L);
                            // 新增岗位信息加入待保存列表
                            savePositionMap.put(poitionKey, posEntity);
                            positionIdArr[i] = posEntity.getId();
                        }
                    }
                }
            }
        }
        String posId3 = ArrayUtil.join(positionIdArr, REGEX_CHAR);
        userEntity.setPositionId(posId3);
    }

}
