package com.bringspring.system.external.util;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.permission.entity.*;
import com.bringspring.system.permission.service.*;
import lombok.Data;
import me.chanjar.weixin.cp.bean.WxCpUser;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.bringspring.system.external.constant.WxCpSysConfigConsts.REGEX_CHAR;
import static com.bringspring.system.external.constant.WxCpSysConfigConsts.TARGET_CHAR;
import static com.bringspring.system.message.util.SynThirdConsts.*;
import static com.bringspring.system.permission.constant.PermissionConst.*;

/**
 * 公共校验
 */
@Data
@Component
public class WxCpCorpValidateUtil {

    @Autowired
    private SynThirdInfoService synThirdInfoService;
    @Autowired
    private UserService userService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    PositionService positionService;
    @Autowired
    private RoleService roleService;
    @Autowired
    private UserRelationService userRelationService;
    Map<String, SynThirdInfoEntity> thirdCorpOrgMap = new HashMap<>();
    Map<String, OrganizeEntity> organizeMap = new HashMap<>();
    Map<String, SynThirdInfoEntity> thirdCorpUserMap = new HashMap<>();
    Map<String, UserEntity> userIdMap = new HashMap<>();
    Map<String, UserEntity> accountMap = new HashMap<>();
    Map<String, UserEntity> phoneMap = new HashMap<>();
    Map<String, UserEntity> emailMap = new HashMap<>();
    Map<String, PositionEntity> positionIdMap = new HashMap<>();
    /**
     * 默认普通员工角色
     */
    String defaultRoleId = "393332333610065861";

    /**
     * 初始化系统数据
     * 第三方组织同步记录
     */
    public void initData(String corpId) {

        // 当前指定数据类型的数据列表 1:企业微信;1:部门组织
        List<SynThirdInfoEntity> thirdList = synThirdInfoService.getListByCrop(THIRD_TYPE_QY, DATA_TYPE_ORG, corpId);
        thirdCorpOrgMap.clear();
        if (StringUtils.isNotEmpty(thirdList)) {
            for (SynThirdInfoEntity entity : thirdList) {
                thirdCorpOrgMap.put(entity.getThirdObjectId(), entity);
            }
        }

    }

    /**
     * 初始化系统数据
     * 第三方用户同步记录、系统用户、岗位、默认角色
     */
    public void initDataByUser(String corpId) {

        // 当前指定数据类型的数据列表 1:企业微信;1:部门组织
        List<SynThirdInfoEntity> orgThirdList = synThirdInfoService.getListByCrop(THIRD_TYPE_QY, DATA_TYPE_ORG, corpId);
        thirdCorpOrgMap.clear();
        List<String> orgIds = new ArrayList<>();
        if (StringUtils.isNotEmpty(orgThirdList)) {
            for (SynThirdInfoEntity entity : orgThirdList) {
                thirdCorpOrgMap.put(entity.getThirdObjectId(), entity);
                orgIds.add(entity.getSystemObjectId());
            }
        }

        // 系统当前公司岗位列表
        List<PositionEntity> positionList = positionService.getListByOrgIds(orgIds);
        positionIdMap.clear();
        if (StringUtils.isNotEmpty(positionList)) {
            for (PositionEntity entity : positionList) {
                positionIdMap.put(entity.getFullName() + TARGET_CHAR + entity.getOrganizeId(), entity);
            }
        }

        // 当前指定数据类型的数据列表 1:企业微信;2:用户
        List<SynThirdInfoEntity> thirdList = synThirdInfoService.getListByCrop(THIRD_TYPE_QY, DATA_TYPE_USER, corpId);
        thirdCorpUserMap.clear();
        List<String> userIds = new ArrayList<>();
        if (StringUtils.isNotEmpty(thirdList)) {
            for (SynThirdInfoEntity entity : thirdList) {
                if (StringUtils.isNotEmpty(entity.getThirdObjectId()) && entity.getThirdObjectId().contains(TARGET_CHAR)) {
                    String thirdObjectId = entity.getThirdObjectId().substring(entity.getThirdObjectId().lastIndexOf(TARGET_CHAR) + 1);
                    thirdCorpUserMap.put(thirdObjectId, entity);
                } else {
                    thirdCorpUserMap.put(entity.getThirdObjectId(), entity);
                }
                userIds.add(entity.getSystemObjectId());
            }
        }

        // 系统当前公司用户列表
        List<UserEntity> userList = userService.getUserByIds(userIds);
        userIdMap.clear();
        accountMap.clear();
        phoneMap.clear();
        emailMap.clear();
        if (StringUtils.isNotEmpty(userList)) {
            for (UserEntity entity : userList) {
                userIdMap.put(entity.getId(), entity);
                accountMap.put(entity.getAccount(), entity);
                phoneMap.put(entity.getMobilePhone(), entity);
                emailMap.put(entity.getEmail(), entity);
            }
        }

        // 默认角色
        List<RoleEntity> roleList = roleService.getListByEnCode(SIMPLE_USER);
        if (CollectionUtil.isNotEmpty(roleList)) {
            defaultRoleId = roleList.get(0).getId();
        }
    }

    /**
     * ## 表base_sync_third_info中不存在third_object_id即为新增,否则为更新;
     * ##   新增new一个SynThirdInfoEntity,更新则回去系统SynThirdInfoEntity.
     *
     * @param thirdObjid
     * @return
     */
    public SynThirdInfoEntity getThirdCorpOrgByThirdId(String thirdObjid) {
        if (thirdCorpOrgMap.containsKey(thirdObjid)) {
            return thirdCorpOrgMap.get(thirdObjid);
        }
        return null;
    }

    /**
     * 判断部门是否新增.
     * ## 表base_sync_third_info中不存在third_object_id即为新增,否则为更新;
     * ##   新增new一个SynThirdInfoEntity,更新则获取系统SynThirdInfoEntity.
     *
     * @param thirdObjid
     * @return
     */
    public SynThirdInfoEntity newSynThirdInfoForWxDept(String corpId, String thirdObjid) {
        SynThirdInfoEntity synThirdInfoEntity = new SynThirdInfoEntity();
        synThirdInfoEntity.setId(RandomUtil.uuId());
        synThirdInfoEntity.setCropId(corpId); // 企业微信公司id
        synThirdInfoEntity.setThirdType(Integer.valueOf(THIRD_TYPE_QY)); // 第三方类型(1:企业微信;2:钉钉)
        synThirdInfoEntity.setDataType(Integer.valueOf(DATA_TYPE_ORG)); // 数据类型(1:组织(公司与部门);2:用户)
        synThirdInfoEntity.setThirdObjectId(thirdObjid); // 第三对象ID
        return synThirdInfoEntity;
    }

    /**
     * 取系统账号account.
     * ## account取不为空的mobile(手机号)>email(邮箱)>userid.
     *
     * @param wxCpUser
     * @return
     */
    public String getAccount(WxCpUser wxCpUser) {
        if (StringUtils.isNotEmpty(wxCpUser.getMobile())) {
            return wxCpUser.getMobile();
        } else if (StringUtils.isNotEmpty(wxCpUser.getEmail())) {
            return wxCpUser.getEmail();
        }
        return wxCpUser.getUserId();
    }

    /**
     * 判断成员是否新增.
     * ## 标准1:表base_sync_third_info中不存在third_object_id即为新增,否则为更新;
     * ##   新增new一个SynThirdInfoEntity,更新则回去系统SynThirdInfoEntity.
     * ## 注意:third_object_id的规则是corpid/userid.
     *
     * @param thirdObjid
     * @return
     */
    public SynThirdInfoEntity getThirdCorpUserByThirdId(String thirdObjid) {
        if (thirdCorpUserMap.containsKey(thirdObjid)) {
            return thirdCorpUserMap.get(thirdObjid);
        }
        return null;
    }

    /**
     * 判断成员是否新增.
     * ## 标准1:表base_sync_third_info中不存在third_object_id即为新增,否则为更新;
     * ##   新增new一个SynThirdInfoEntity,更新则获取系统SynThirdInfoEntity.
     *
     * @param thirdObjid
     * @return
     */
    public SynThirdInfoEntity newSynThirdInfoForWxUser(String thirdObjid) {
        SynThirdInfoEntity synThirdInfoEntity = new SynThirdInfoEntity();
        synThirdInfoEntity.setId(RandomUtil.uuId());
        synThirdInfoEntity.setThirdType(Integer.valueOf(THIRD_TYPE_QY));
        synThirdInfoEntity.setDataType(Integer.valueOf(DATA_TYPE_USER));
        synThirdInfoEntity.setThirdObjectId(thirdObjid);
        return synThirdInfoEntity;
    }

    /**
     * 获取UserEntity
     * 根据account或phone或email,都不存在返回空
     *
     * @param account 账户
     * @param phone   手机
     * @param email   邮箱
     * @return
     */
    public UserEntity getUserEntity(String account, String phone, String email) {
        if (StringUtils.isNotEmpty(account) && accountMap.containsKey(account)) {
            return accountMap.get(account);
        } else if (StringUtils.isNotEmpty(phone) && phoneMap.containsKey(phone)) {
            return phoneMap.get(phone);
        } else if (StringUtils.isNotEmpty(email) && emailMap.containsKey(email)) {
            return emailMap.get(email);
        }
        return null;
    }

    /**
     * 根据系统system_object_id获取UserEntity
     *
     * @param userid
     * @return
     */
    public UserEntity getUserEntityByUserid(String userid) {
        if (userIdMap.containsKey(userid)) {
            return userIdMap.get(userid);
        }
        return null;
    }

    /**
     * 判断成员是否新增.
     * ## 标准2:表base_user中不存在account即为新增,否则为更新;
     * ##   新增new一个UserEntity,更新则获取系统UserEntity.
     *
     * @param account
     * @return
     */
    public UserEntity getUserEntityByAccount(String account) {
        if (accountMap.containsKey(account)) {
            return accountMap.get(account);
        }
        return null;
    }

    /**
     * 工号信息处理
     * ## 企业微信返回的用户信息,扩展属性extattr或有工号信息,需解析出来.
     *
     * @param wxCpUser
     * @return
     */
    public String getCode(WxCpUser wxCpUser) {
        String value = null;
        // 企业微信返回的extattr扩展属性中寻找工号进行设置
        if (StringUtils.isNotEmpty(wxCpUser.getExtAttrs())) {
            List<WxCpUser.Attr> extAttrs = wxCpUser.getExtAttrs();
            if (extAttrs.size() != 0) {
                for (WxCpUser.Attr attr : extAttrs) {
                    String name = attr.getName();

                    if (StringUtils.isNotEmpty(name) && name.contains("工号")) {
                        value = attr.getTextValue();
                    }
                }
            }
        }
        return value;
    }

    /**
     * 合并去重组织信息
     * 将bEntity的organizeId合并到aEntity的organizeId
     *
     * @param aEntity
     * @param bEntity
     */
    public void mergeDistinctOrganize(UserEntity aEntity, UserEntity bEntity) {
        // 合并组织
        String orgId1 = aEntity.getOrganizeId();
        String[] arr1 = StringUtils.isNotEmpty(orgId1) ? orgId1.split(REGEX_CHAR) : new String[0];
        // userEntity
        String orgId2 = bEntity.getOrganizeId();
        String[] arr2 = StringUtils.isNotEmpty(orgId2) ? orgId2.split(REGEX_CHAR) : new String[0];
        // 合并,去重,set组织
        String[] arrAll = ArrayUtil.addAll(arr1, arr2);
        String[] distinct = ArrayUtil.distinct(arrAll);
        String orgId3 = ArrayUtil.join(distinct, REGEX_CHAR);
        aEntity.setOrganizeId(orgId3);
    }

    /**
     * 合并去重岗位信息
     * 将bEntity的positionId合并到aEntity的positionId
     *
     * @param aEntity
     * @param bEntity
     */
    public void mergeDistinctPosition(UserEntity aEntity, UserEntity bEntity) {
        // 合并岗位
        String posId1 = aEntity.getPositionId();
        String[] posArr1 = StringUtils.isNotEmpty(posId1) ? posId1.split(REGEX_CHAR) : new String[0];
        String posId2 = bEntity.getPositionId();
        // userEntity
        String[] posArr2 = StringUtils.isNotEmpty(posId2) ? posId2.split(REGEX_CHAR) : new String[0];
        // 合并,去重,set岗位
        String[] posArrAll = ArrayUtil.addAll(posArr1, posArr2);
        String[] distinctPos = ArrayUtil.distinct(posArrAll);
        String posId3 = ArrayUtil.join(distinctPos, REGEX_CHAR);
        aEntity.setPositionId(posId3);
    }

    /**
     * 系统poitionKey
     * 判断是否存在
     *
     * @param poitionKey
     * @return
     */
    public PositionEntity getPositionByUserid(String poitionKey) {
        if (positionIdMap.containsKey(poitionKey)) {
            return positionIdMap.get(poitionKey);
        }
        return null;
    }

    /**
     * 岗位名称转换positionId
     * <p>
     * 岗位信息持久化转换
     * ## 经过前边处理,[企业通讯录]部门成员接口返回的成员信息已全部处理完成,接下来将数据持久化到本地.
     * ## 新增new一个PositionEntity岗位存入数据库.
     * ## 取出positionId,用positionId与user建立关联关系.在base_user_relation用户关系表内.
     * 注意:系统存在要更新的用户,组织和岗位以企业微信为准,系统配置的会丢失.
     *
     * @param savePositionMap
     * @param userEntity
     */
    public void convertPosition(Map<String, PositionEntity> savePositionMap, UserEntity userEntity) {

        // 岗位名称转换positionId
        String positionId = userEntity.getPositionId();
        String[] posArr = StringUtils.isNotEmpty(positionId) ? positionId.split(REGEX_CHAR) : new String[0];
        int posLength = posArr.length;
        String[] positionIdArr = new String[posLength];
        for (int i = 0; i < posLength; i++) {
            String poitionKey = posArr[i];
            PositionEntity positionEntity = getPositionByUserid(poitionKey);
            if (ObjectUtil.isNotEmpty(positionEntity)) {
                // 存在
                positionIdArr[i] = positionEntity.getId();
            } else {
                if (savePositionMap.containsKey(poitionKey)) {
                    // 存在
                    positionIdArr[i] = savePositionMap.get(poitionKey).getId();
                } else {
                    // 不存在
                    if (poitionKey.contains(TARGET_CHAR)) {
                        String position = poitionKey.substring(0, poitionKey.lastIndexOf(TARGET_CHAR));
                        String deptId = poitionKey.substring(poitionKey.lastIndexOf(TARGET_CHAR) + 1);
                        if (StringUtils.isNotEmpty(position)) {
                            // 企业微信 岗位信息不为空
                            PositionEntity posEntity = new PositionEntity();
                            posEntity.setId(RandomUtil.uuId());
                            posEntity.setFullName(position);
                            posEntity.setOrganizeId(deptId);
                            posEntity.setSortCode(0L);
                            // 新增岗位信息加入待保存列表
                            savePositionMap.put(poitionKey, posEntity);
                            positionIdArr[i] = posEntity.getId();
                        }
                    }
                }
            }
        }
        String posId3 = ArrayUtil.join(positionIdArr, REGEX_CHAR);
        userEntity.setPositionId(posId3);
    }

    /**
     * 系统已存在做更新的用户
     * 保留用户关联信息
     *
     * @param userEntity
     */
    public void setUserRelation(UserEntity userEntity) {
        String organizeIds = ""; // 组织
        String positionIds = ""; // 岗位
        String roleIds = ""; // 角色
        String groupIds = ""; // 分组
        // 根据用户主键ID 获取用户关系表数据
        List<UserRelationEntity> relationList = userRelationService.getListByUserId(userEntity.getId());
        if (StringUtils.isNotEmpty(relationList)) {
            for (UserRelationEntity item : relationList) {
                switch (item.getObjectType()) {
//                    case ORGANIZE:
//                        organizeIds += item.getObjectId() + REGEX_CHAR;
//                        break;
//                    case POSITION:
//                        positionIds += item.getObjectId() + REGEX_CHAR;
//                        break;
                    case ROLE:
                        roleIds += item.getObjectId() + REGEX_CHAR;
                        break;
                    case GROUP:
                        groupIds += item.getObjectId() + REGEX_CHAR;
                        break;
                    default:
                        break;
                }
            }
        }
//        userEntity.setOrganizeId(organizeIds); // 注释原因:不保留系统配置,以企业微信为准
//        userEntity.setPositionId(positionIds); // 注释原因:不保留系统配置,以企业微信为准
        userEntity.setRoleId(roleIds); // 保留系统配置
        userEntity.setGroupId(groupIds); // 保留系统配置
    }

}
