package com.bringspring.system.msgcenter.controller;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.afterturn.easypoi.excel.entity.enmus.ExcelType;
import cn.afterturn.easypoi.excel.entity.params.ExcelExportEntity;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.DownloadVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.*;
import com.bringspring.common.util.enums.FileTypeEnum;
import com.bringspring.common.util.file.UploadUtil;
import com.bringspring.system.msgcenter.entity.McBusinessEntity;
import com.bringspring.system.msgcenter.entity.McBusinessKeywordEntity;
import com.bringspring.system.msgcenter.model.mcbusiness.*;
import com.bringspring.system.msgcenter.model.mcbusinesskeyword.McBusinessKeywordModel;
import com.bringspring.system.msgcenter.service.McBusinessKeywordService;
import com.bringspring.system.msgcenter.service.McBusinessService;
import com.bringspring.system.msgcenter.service.McMsgSendService;
import com.bringspring.system.msgcenter.service.McMsgTemplateService;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.Cleanup;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.bringspring.system.msgcenter.constant.CommonConsts.ENABLED_MARK_OFF;
import static com.bringspring.system.msgcenter.constant.CommonConsts.ENABLED_MARK_ON;
import static com.bringspring.system.msgcenter.constant.DictConsts.MESSAGE_TYPE;

/**
 * 消息业务场景信息
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2024-06-06
 */
@Slf4j
@RestController
@Api(tags = "消息业务场景信息", value = "mcBusiness")
@RequestMapping("/api/msgCenter/mcBusiness")
public class McBusinessController {

    @Autowired
    private BaseDataUtil baseDataUtil;

    @Autowired
    private ConfigValueUtil configValueUtil;

    @Autowired
    private UserProvider userProvider;

    @Autowired
    private McBusinessService mcBusinessService;

    @Autowired
    private McBusinessKeywordService mcBusinessKeywordService;

    @Autowired
    private McMsgTemplateService mcMsgTemplateService;

    @Autowired
    private McMsgSendService mcMsgSendService;

    /**
     * 列表
     *
     * @param mcBusinessPagination
     * @return
     */
    @PostMapping("/getList")
    public ActionResult list(@RequestBody McBusinessPagination mcBusinessPagination) throws IOException {
        List<McBusinessEntity> list = mcBusinessService.getList(mcBusinessPagination);
        List<McBusinessListVO> listVO = JsonUtil.getJsonToList(list, McBusinessListVO.class);

        //处理id字段转名称，若无需转或者为空可删除
        for (McBusinessListVO vo : listVO) {
            vo.setCategory(baseDataUtil.getDictName(vo.getCategory(), MESSAGE_TYPE));
//            vo.setEnabledMark(baseDataUtil.switchSelectValue(vo.getEnabledMark(), ENABLED_MARK_ON, ENABLED_MARK_OFF));
            vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
            vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));
            vo.setDeleteUserId(baseDataUtil.userSelectValue(vo.getDeleteUserId()));
            vo.setCompanyId(baseDataUtil.comSelectValues(vo.getCompanyId()));
            List<String> keywordNames = mcBusinessKeywordService.getKeywordNames(vo.getEnCode());
            vo.setKeywordNames(StringUtils.join(keywordNames, ','));
        }
        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        PaginationVO page = JsonUtil.getJsonToBean(mcBusinessPagination, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }


    /**
     * 创建
     *
     * @param mcBusinessCrForm
     * @return
     */
    @PostMapping
    public ActionResult create(@RequestBody @Valid McBusinessCrForm mcBusinessCrForm) throws DataException {
        // 校验编码是否存在重复
        if (!mcBusinessService.checkEnCodeUnique(mcBusinessCrForm.getEnCode(), null)) {
            return ActionResult.fail("编码已存在");
        }
        // 校验名称是否存在重复
        if (!mcBusinessService.checkFullNameUnique(mcBusinessCrForm.getFullName(), null)) {
            return ActionResult.fail("名称已存在");
        }
        // 校验关键词重复
        if (!mcBusinessKeywordService.checkEnCodeUnique(mcBusinessCrForm.getKeywordList())) {
            return ActionResult.fail("关键词重复");
        }
        mcBusinessService.create(mcBusinessCrForm);
        return ActionResult.success("创建成功");
    }

    /**
     * 信息
     *
     * @param id
     * @return
     */
    @GetMapping("/{id}")
    public ActionResult<McBusinessInfoVO> info(@PathVariable("id") String id) {
        McBusinessEntity entity = mcBusinessService.getInfo(id);

        McBusinessInfoVO vo = JsonUtil.getJsonToBean(entity, McBusinessInfoVO.class);

        // 子表数据转换
        List<McBusinessKeywordEntity> keywordList = mcBusinessService.getMcBusinessKeywordList(entity.getEnCode());
        List<McBusinessKeywordModel> keywordModelList = JsonUtil.getJsonToList(keywordList,
                McBusinessKeywordModel.class);
        if (CollectionUtil.isNotEmpty(keywordModelList)) {
            for (McBusinessKeywordModel keywordModel : keywordModelList) {
            }
        }
        vo.setKeywordList(keywordModelList);

        return ActionResult.success(vo);
    }

    /**
     * 表单信息(详情页)
     *
     * @param id
     * @return
     */
    @GetMapping("/detail/{id}")
    public ActionResult<McBusinessInfoVO> detailInfo(@PathVariable("id") String id) {
        McBusinessEntity entity = mcBusinessService.getInfo(id);

        McBusinessInfoVO vo = JsonUtil.getJsonToBean(entity, McBusinessInfoVO.class);
        //添加到详情表单对象中
        vo.setCategory(baseDataUtil.getDictName(vo.getCategory(), MESSAGE_TYPE));
        vo.setEnabledMark(baseDataUtil.switchSelectValue(vo.getEnabledMark(), ENABLED_MARK_ON, ENABLED_MARK_OFF));
        vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
        vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));
        vo.setDeleteUserId(baseDataUtil.userSelectValue(vo.getDeleteUserId()));
        vo.setCompanyId(baseDataUtil.comSelectValues(vo.getCompanyId()));

        //子表数据转换
        List<McBusinessKeywordEntity> keywordList = mcBusinessService.getMcBusinessKeywordList(entity.getEnCode());
        List<McBusinessKeywordModel> keywordModelList = JsonUtil.getJsonToList(keywordList,
                McBusinessKeywordModel.class);
        if (CollectionUtil.isNotEmpty(keywordModelList)) {
            for (McBusinessKeywordModel keywordModel : keywordModelList) {
            }
        }
        vo.setKeywordList(keywordModelList);

        return ActionResult.success(vo);
    }


    /**
     * 更新
     *
     * @param id
     * @return
     */
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid McBusinessUpForm mcBusinessUpForm) throws DataException {
        // 校验编码是否存在重复
        if (!mcBusinessService.checkEnCodeUnique(mcBusinessUpForm.getEnCode(), id)) {
            return ActionResult.fail("编码已存在");
        }
        // 校验名称是否存在重复
        if (!mcBusinessService.checkFullNameUnique(mcBusinessUpForm.getFullName(), id)) {
            return ActionResult.fail("名称已存在");
        }
        // 校验关键词重复
        if (!mcBusinessKeywordService.checkEnCodeUnique(mcBusinessUpForm.getKeywordList())) {
            return ActionResult.fail("关键词重复");
        }

        McBusinessEntity entity = mcBusinessService.getInfo(id);
        if (ObjectUtil.isNotNull(entity)) {
            mcBusinessService.update(id, mcBusinessUpForm);
            return ActionResult.success("更新成功");
        } else {
            return ActionResult.fail("更新失败，数据不存在");
        }
    }


    /**
     * 删除
     *
     * @param id
     * @return
     */
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        McBusinessEntity entity = mcBusinessService.getInfo(id);
        if (ObjectUtil.isNotNull(entity)) {
            if (mcMsgTemplateService.getCountByMessageSource(entity.getEnCode()) > 0) {
                return ActionResult.fail("消息模板已关联场景，不允许删除！");
            }
            if (mcMsgSendService.getCountByMessageSource(entity.getEnCode()) > 0) {
                return ActionResult.fail("发送配置已关联场景，不允许删除！");
            }
            mcBusinessService.delete(entity);
        }
        return ActionResult.success("删除成功");
    }


    /**
     * 消息场景下拉框
     *
     * @return
     */
    @ApiOperation("消息场景下拉框")
    @GetMapping("/Selector")
    public ActionResult<PageListVO<McBusinessSelector>> selector(McBusinessPagination pagination) {
        List<McBusinessEntity> list = mcBusinessService.getList(pagination);
        List<McBusinessSelector> listSelector = JsonUtil.getJsonToList(list, McBusinessSelector.class);

        if (CollectionUtil.isNotEmpty(listSelector)) {
            for (McBusinessSelector selector : listSelector) {
                //子表数据转换
                List<McBusinessKeywordEntity> keywordList =
                        mcBusinessService.getMcBusinessKeywordList(selector.getEnCode());
                List<McBusinessKeywordModel> keywordModelList = JsonUtil.getJsonToList(keywordList,
                        McBusinessKeywordModel.class);
                selector.setKeywordList(keywordModelList);
            }
        }

        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(listSelector, paginationVO);
    }

    /**
     * 消息场景关键词
     *
     * @return
     */
    @ApiOperation("消息场景关键词")
    @GetMapping("/keyword/Selector/{enCode}")
    public ActionResult selector(@PathVariable("enCode") String enCode) {
        List<McBusinessKeywordEntity> keywordList = mcBusinessService.getMcBusinessKeywordList(enCode);
        List<McBusinessKeywordModel> keywordModelList = JsonUtil.getJsonToList(keywordList,
                McBusinessKeywordModel.class);
        return ActionResult.success(keywordModelList);
    }

    /**
     * 模板下载
     *
     * @return
     */
    @ApiOperation("模板下载")
    @GetMapping("/templateDownload")
    public ActionResult<DownloadVO> TemplateDownload() {
        UserInfo userInfo = userProvider.get();
        DownloadVO vo = DownloadVO.builder().build();
        try {
            vo.setName("职员信息.xlsx");
            vo.setUrl(UploaderUtil.uploaderFile("/api/Common/DownloadModel?encryption=", userInfo.getId() + "#" +
                    "职员信息.xlsx" + "#" + "Temporary"));
        } catch (Exception e) {
            log.error("信息导出Excel错误:{}", e.getMessage());
        }
        return ActionResult.success(vo);
    }

    /**
     * 导出Excel
     *
     * @return
     */
    @ApiOperation("导出Excel")
    @GetMapping("/Actions/Export")
    public ActionResult Export(McBusinessPaginationExportModel mcBusinessPaginationExportModel) throws IOException {
        if (StringUtils.isEmpty(mcBusinessPaginationExportModel.getSelectKey())) {
            return ActionResult.fail("请选择导出字段");
        }
        McBusinessPagination mcBusinessPagination = JsonUtil.getJsonToBean(mcBusinessPaginationExportModel,
                McBusinessPagination.class);
        List<McBusinessEntity> list = mcBusinessService.getTypeList(mcBusinessPagination,
                mcBusinessPaginationExportModel.getDataType());
        List<McBusinessListVO> listVO = JsonUtil.getJsonToList(list, McBusinessListVO.class);
        //处理id字段转名称，若无需转或者为空可删除
        for (McBusinessListVO vo : listVO) {
            vo.setCategory(baseDataUtil.getDictName(vo.getCategory(), MESSAGE_TYPE));
            vo.setEnabledMark(baseDataUtil.switchSelectValue(vo.getEnabledMark(), ENABLED_MARK_ON, ENABLED_MARK_OFF));
            vo.setCreatorUserId(baseDataUtil.userSelectValue(vo.getCreatorUserId()));
            vo.setLastModifyUserId(baseDataUtil.userSelectValue(vo.getLastModifyUserId()));
            vo.setDeleteUserId(baseDataUtil.userSelectValue(vo.getDeleteUserId()));
            vo.setCompanyId(baseDataUtil.comSelectValues(vo.getCompanyId()));
        }

        //转换为map输出
        List<Map<String, Object>> mapList = JsonUtil.getJsonToListMap(JsonUtil.getObjectToStringDateFormat(listVO,
                "yyyy-MM-dd HH:mm:ss"));
        String[] keys = !StringUtils.isEmpty(mcBusinessPaginationExportModel.getSelectKey()) ?
                mcBusinessPaginationExportModel.getSelectKey().split(",") : new String[0];
        UserInfo userInfo = userProvider.get();
        DownloadVO vo = this.creatModelExcel(configValueUtil.getTemporaryFilePath(), mapList, keys, userInfo);
        return ActionResult.success(vo);
    }

    //导出表格
    public DownloadVO creatModelExcel(String path, List<Map<String, Object>> list, String[] keys, UserInfo userInfo) {
        DownloadVO vo = DownloadVO.builder().build();
        List<ExcelExportEntity> entitys = new ArrayList<>();
        if (keys.length > 0) {
            for (String key : keys) {
                switch (key) {
                    case "category":
                        entitys.add(new ExcelExportEntity("分类", "category"));
                        break;
                    case "fullName":
                        entitys.add(new ExcelExportEntity("名称", "fullName"));
                        break;
                    case "enCode":
                        entitys.add(new ExcelExportEntity("编码", "enCode"));
                        break;
                    case "enabledMark":
                        entitys.add(new ExcelExportEntity("状态", "enabledMark"));
                        break;
                    case "description":
                        entitys.add(new ExcelExportEntity("描述", "description"));
                        break;
                    case "creatorUserId":
                        entitys.add(new ExcelExportEntity("创建用户", "creatorUserId"));
                        break;
                    case "creatorTime":
                        entitys.add(new ExcelExportEntity("创建时间", "creatorTime"));
                        break;
                    case "lastModifyUserId":
                        entitys.add(new ExcelExportEntity("修改用户", "lastModifyUserId"));
                        break;
                    case "lastModifyTime":
                        entitys.add(new ExcelExportEntity("修改时间", "lastModifyTime"));
                        break;
                    case "deleteUserId":
                        entitys.add(new ExcelExportEntity("删除用户", "deleteUserId"));
                        break;
                    case "deleteTime":
                        entitys.add(new ExcelExportEntity("删除时间", "deleteTime"));
                        break;
                    case "deleteMark":
                        entitys.add(new ExcelExportEntity("有效标志", "deleteMark"));
                        break;
                    case "companyId":
                        entitys.add(new ExcelExportEntity("所属公司", "companyId"));
                        break;
                    default:
                        break;
                }
            }
        }

        ExportParams exportParams = new ExportParams(null, "表单信息");
        exportParams.setType(ExcelType.XSSF);
        try {
            @Cleanup Workbook workbook = new HSSFWorkbook();
            if (entitys.size() > 0) {
                workbook = ExcelExportUtil.exportExcel(exportParams, entitys, list);
            }
            String name = "表单信息" + DateUtil.dateNow("yyyyMMdd") + "_" + RandomUtil.uuId() + ".xlsx";

            String fileName = configValueUtil.getTemporaryFilePath() + name;
            @Cleanup FileOutputStream output = new FileOutputStream(XSSEscape.escapePath(fileName));
            workbook.write(output);
            //上传文件
            UploadUtil.uploadFile(configValueUtil.getFileType(), fileName, FileTypeEnum.TEMPORARY, name);

            vo.setName(name);
            vo.setUrl(UploaderUtil.uploaderFile(userInfo.getId() + "#" + name + "#" + "Temporary"));
        } catch (Exception e) {
            log.error("信息导出Excel错误:{}", e.getMessage());
            e.printStackTrace();
        }
        return vo;
    }

}
