package com.bringspring.system.msgcenter.controller;

import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.msgcenter.entity.McMsgSendEntity;
import com.bringspring.system.msgcenter.model.BusinessKeys;
import com.bringspring.system.msgcenter.model.TaskMsg;
import com.bringspring.system.msgcenter.service.McMsgSendService;
import com.bringspring.system.msgcenter.util.SendTaskMsgUtils;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;
import java.util.concurrent.Executor;

import static com.bringspring.system.msgcenter.constant.DictConsts.TASK_CODE;

@Slf4j
@RestController
@Api(tags = "业务消息发送接口服务", value = "messageService")
@RequestMapping("/api/outer/messageService")
public class MsgCenterService {
    /**
     * 取出线程池
     */
    @Autowired
    private Executor threadPoolExecutor;
    @Autowired
    private SendTaskMsgUtils sendTaskMsgUtils;

    @Autowired
    private McMsgSendService mcMsgSendService;
    @Autowired
    private BaseDataUtil baseDataUtil;

    /**
     * 业务消息发送
     *
     * @param taskMsg
     * @return
     */
    @PostMapping("/taskMsg/send")
    public ActionResult create(@RequestBody @Valid TaskMsg taskMsg) throws DataException {
        String errMsg = "业务消息发送";
        // 数据验证
        if (ObjectUtil.isNotEmpty(taskMsg)) {
            // 业务系统的code 验证
            String taskCode = taskMsg.getTaskCode();
            if (StringUtils.isEmpty(taskCode)) {
                errMsg += "，失败：taskCode不能为空";
                log.error(errMsg);
                return ActionResult.fail(errMsg);
            } else {
                if (StringUtils.isEmpty(baseDataUtil.getDictName(taskCode, TASK_CODE))) {
                    errMsg += "，失败：taskCode未授权";
                    log.error(errMsg);
                    return ActionResult.fail(errMsg);
                }
            }
            // 发送配置 验证
            String sendCode = taskMsg.getSendCode();
            if (StringUtils.isEmpty(sendCode)) {
                errMsg += "，失败：sendCode不能为空";
                log.error(errMsg);
                return ActionResult.fail(errMsg);
            }

            McMsgSendEntity msgSendEntity = mcMsgSendService.getValidInfo(sendCode);
            // 验证发送配置code有效性
            if (ObjectUtil.isEmpty(msgSendEntity)) {
                errMsg += "，失败：sendCode不存在或禁用，请确定发送配置编码";
                log.error(errMsg);
                return ActionResult.fail(errMsg);
            }
            BusinessKeys businessKey = taskMsg.getBusinessKey();
            if (ObjectUtil.isEmpty(businessKey)) {
                errMsg += "，失败：businessKey不能为空";
                log.error(errMsg);
                return ActionResult.fail(errMsg);
            }

            // 接收人或接收部门 验证
            String toUserIds = taskMsg.getToUserIds();
            String toDeptIds = taskMsg.getToDeptIds();
            if (StringUtils.isEmpty(toUserIds) && StringUtils.isEmpty(toDeptIds)) {
                errMsg += "，失败：无效的接收人";
                log.error(errMsg);
                return ActionResult.fail(errMsg);
            }

            // 异步处理 消息数据接收和发送
            threadPoolExecutor.execute(() -> {
                try {
                    sendTaskMsgUtils.sendTaskMsg(taskMsg);
                } catch (Exception e) {
                    e.printStackTrace();
                    log.error("失败: " + e.getMessage());
                }
            });

            return ActionResult.success("发送成功");
        } else {
            errMsg += "，失败：请求参数不能为空";
            log.error(errMsg);
            return ActionResult.fail(errMsg);
        }
    }
}
