package com.bringspring.system.msgcenter.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.util.*;
import com.bringspring.system.msgcenter.entity.McMsgSendEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.exception.MsgCenterException;
import com.bringspring.system.msgcenter.mapper.McTaskMsgMapper;
import com.bringspring.system.msgcenter.model.SendReceiveParam;
import com.bringspring.system.msgcenter.model.mcmsgsend.McMsgSendInfoVO;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.model.mctaskmsg.McTaskMsgCrForm;
import com.bringspring.system.msgcenter.model.mctaskmsg.McTaskMsgPagination;
import com.bringspring.system.msgcenter.model.mctaskmsg.McTaskMsgUpForm;
import com.bringspring.system.msgcenter.service.*;
import com.bringspring.system.msgcenter.service.strategy.AbstractMessageChannelStrategy;
import com.bringspring.system.msgcenter.service.strategy.factory.MessageChannelStrategyFactory;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.UserService;
import org.apache.commons.collections4.SetUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import static com.bringspring.system.msgcenter.constant.CommonConsts.*;

/**
 * 业务场景消息表
 * 版本： V1.0.0
 * 版权： 荣科科技股份有限公司
 * 作者： RKKJ开发平台组
 * 日期： 2024-06-14
 */
@Service
public class McTaskMsgServiceImpl extends ServiceImpl<McTaskMsgMapper, McTaskMsgEntity> implements McTaskMsgService {


    @Autowired
    private UserProvider userProvider;
    @Autowired
    private UserService userService;

    @Autowired
    private McMsgSendService mcMsgSendService;
    @Autowired
    private McBusinessService mcBusinessService;
    @Autowired
    private McMsgTemplateService mcMsgTemplateService;
    @Autowired
    private McTaskMsgContentService mcTaskMsgContentService;
    @Autowired
    private McTaskMsgReceiveService mcTaskMsgReceiveService;
    @Autowired
    private McTaskMsgMapper mcTaskMsgMapper;

    @Autowired
    private MessageChannelStrategyFactory strategyFactory;

    /**
     * 取出线程池
     */
    @Autowired
    private Executor threadPoolExecutor;


    @Override
    public List<McTaskMsgEntity> getList(McTaskMsgPagination pagination) {
        String userId = userProvider.get().getUserId();
        List<String> AllIdList = new ArrayList();
        int total = 0;
        int mcTaskMsgNum = 0;
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();

        if (StringUtils.isNotEmpty(pagination.getKeyword())) {
            queryWrapper.lambda().and(t -> t.like(McTaskMsgEntity::getId, pagination.getKeyword()).or().like(McTaskMsgEntity::getTaskCode, pagination.getKeyword()).or().like(McTaskMsgEntity::getSendCode, pagination.getKeyword()));
            mcTaskMsgNum++;
        }
        if (StringUtils.isNotEmpty(pagination.getTaskCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getTaskCode, pagination.getTaskCode());
        }

        if (StringUtils.isNotEmpty(pagination.getSendCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getSendCode, pagination.getSendCode());
        }
        if (StringUtils.isNotEmpty(pagination.getEnabledMark())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().eq(McTaskMsgEntity::getEnabledMark, Integer.parseInt(pagination.getEnabledMark()));
        }

        if (StringUtils.isNotEmpty(pagination.getToUserIds())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getToUserIds, pagination.getToUserIds());
        }

        if (AllIdList.size() > 0) {
            queryWrapper.lambda().in(McTaskMsgEntity::getId, AllIdList);
        }
        //排序
        if (StringUtils.isEmpty(pagination.getSidx())) {
            queryWrapper.lambda().orderByDesc(McTaskMsgEntity::getCreatorTime);
        } else {
            try {
                String sidx = pagination.getSidx();
                McTaskMsgEntity mcTaskMsgEntity = new McTaskMsgEntity();
                Field declaredField = mcTaskMsgEntity.getClass().getDeclaredField(sidx);
                declaredField.setAccessible(true);
                String value = declaredField.getAnnotation(TableField.class).value();
                queryWrapper = "asc".equals(pagination.getSort().toLowerCase()) ?
                        queryWrapper.orderByAsc(value) : queryWrapper.orderByDesc(value);
            } catch (NoSuchFieldException e) {
                e.printStackTrace();
            }
        }
        if ((total > 0 && AllIdList.size() > 0) || total == 0) {
            Page<McTaskMsgEntity> page = new Page<>(pagination.getCurrentPage(), pagination.getPageSize());
            IPage<McTaskMsgEntity> userIPage = this.page(page, queryWrapper);
            return pagination.setData(userIPage.getRecords(), userIPage.getTotal());
        } else {
            List<McTaskMsgEntity> list = new ArrayList();
            return pagination.setData(list, list.size());
        }
    }

    @Override
    public List<McTaskMsgEntity> getTypeList(McTaskMsgPagination pagination, String dataType) {
        String userId = userProvider.get().getUserId();
        List<String> AllIdList = new ArrayList();
        int total = 0;
        int mcTaskMsgNum = 0;
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(pagination.getTaskCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getTaskCode, pagination.getTaskCode());
        }

        if (StringUtils.isNotEmpty(pagination.getSendCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getSendCode, pagination.getSendCode());
        }

        if (StringUtils.isNotEmpty(pagination.getToUserIds())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getToUserIds, pagination.getToUserIds());
        }

        if (AllIdList.size() > 0) {
            queryWrapper.lambda().in(McTaskMsgEntity::getId, AllIdList);
        }
        //排序
        if (StringUtils.isEmpty(pagination.getSidx())) {
            queryWrapper.lambda().orderByDesc(McTaskMsgEntity::getCreatorTime);
        } else {
            try {
                String sidx = pagination.getSidx();
                McTaskMsgEntity mcTaskMsgEntity = new McTaskMsgEntity();
                Field declaredField = mcTaskMsgEntity.getClass().getDeclaredField(sidx);
                declaredField.setAccessible(true);
                String value = declaredField.getAnnotation(TableField.class).value();
                queryWrapper = "asc".equals(pagination.getSort().toLowerCase()) ?
                        queryWrapper.orderByAsc(value) : queryWrapper.orderByDesc(value);
            } catch (NoSuchFieldException e) {
                e.printStackTrace();
            }
        }
        if ("0".equals(dataType)) {
            if ((total > 0 && AllIdList.size() > 0) || total == 0) {
                Page<McTaskMsgEntity> page = new Page<>(pagination.getCurrentPage(), pagination.getPageSize());
                IPage<McTaskMsgEntity> userIPage = this.page(page, queryWrapper);
                return pagination.setData(userIPage.getRecords(), userIPage.getTotal());
            } else {
                List<McTaskMsgEntity> list = new ArrayList();
                return pagination.setData(list, list.size());
            }
        } else {
            return this.list(queryWrapper);
        }
    }

    @Override
    public McTaskMsgEntity getInfo(String id) {
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(McTaskMsgEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    @DSTransactional
    public void create(McTaskMsgCrForm mcTaskMsgCrForm) {
        String mainId = RandomUtil.uuId();
        McTaskMsgEntity entity = JsonUtil.getJsonToBean(mcTaskMsgCrForm, McTaskMsgEntity.class);
        entity.setId(mainId);
        this.save(entity);
    }

    @Override
    @DSTransactional
    public boolean update(String id, McTaskMsgUpForm mcTaskMsgUpForm) {
        McTaskMsgEntity entity = JsonUtil.getJsonToBean(mcTaskMsgUpForm, McTaskMsgEntity.class);
        entity.setId(id);
        return this.updateById(entity);
    }

    @Override
    @DSTransactional
    public void delete(McTaskMsgEntity entity) {
        if (entity != null) {
            this.removeById(entity.getId());
        }
    }


    /**
     * 创建业务消息并下发
     *
     * @param mcTaskMsgCrForm
     */
    @Override
    public void addAndSendMessage(McTaskMsgCrForm mcTaskMsgCrForm) throws Exception {
        // 创建消息
        McTaskMsgEntity entity = this.addTaskMsg(mcTaskMsgCrForm);

        if (!ObjectUtil.isEmpty(entity)) {
            // 下发消息
            this.sendTaskMsgReceive(entity);
        }
    }

    /**
     * 业务消息创建
     * 1.数据验证
     * 2.持久化保存
     *
     * @param mcTaskMsgCrForm
     */
    @Override
    public McTaskMsgEntity addTaskMsg(McTaskMsgCrForm mcTaskMsgCrForm) throws MsgCenterException {

        /**
         * 数据验证
         */
        if (ObjectUtil.isNotEmpty(mcTaskMsgCrForm)) {
            // 业务系统的code 验证
            String taskCode = mcTaskMsgCrForm.getTaskCode();
            if (StringUtils.isEmpty(taskCode)) throw new MsgCenterException("taskCode不能为空");
            // 发送配置 验证
            String sendCode = mcTaskMsgCrForm.getSendCode();
            if (StringUtils.isEmpty(sendCode)) throw new MsgCenterException("sendCode不能为空");

            McMsgSendEntity msgSendEntity = mcMsgSendService.getValidInfo(sendCode);
            // 验证发送配置code有效性
            if (ObjectUtil.isEmpty(msgSendEntity)) {
                throw new MsgCenterException("sendCode不存在或禁用，请确定发送配置编码");
            }

            if (StringUtils.isEmpty(mcTaskMsgCrForm.getBusinessKey())) {
                throw new MsgCenterException("使用模板，businessKey不能为空");
            } else if (!JsonUtil.isJSONString(mcTaskMsgCrForm.getBusinessKey())) {
                throw new MsgCenterException("parameterMap非json格式");
            }
            // 接收人或接收部门 验证
            final String toUserIds = mcTaskMsgCrForm.getToUserIds();
            final String toDeptIds = mcTaskMsgCrForm.getToDeptIds();
            if (StringUtils.isEmpty(toUserIds) && StringUtils.isEmpty(toDeptIds))
                throw new MsgCenterException("无效的接收人");

            mcTaskMsgCrForm.setEnabledMark("1"); // 待发送

            McTaskMsgEntity entity = JsonUtil.getJsonToBean(mcTaskMsgCrForm, McTaskMsgEntity.class);
            entity.setId(RandomUtil.uuId()); // 设置消息内容表主键id
            this.save(entity);
            return entity;
        } else {
            throw new MsgCenterException("请求参数不能为空");
        }
    }

    /**
     * 下发消息
     * <p>
     * 1.根据task_msg的发送配置，解释消息模板及消息内容，解释接收人：下发渠道+接收人=>task_msg_receive接收人记录
     * 接收人：user列表、部门列表或@ALL，@ALL时部门列表不生效
     * 2.根据task_msg_receive数据，渠道消息下发：首次全量下发
     * 根据消息内容主键Id,按渠道类型分别获取受接收人记录进行渠道消息封装下发。
     *
     * @param entity 业务消息
     * @return
     */
    @Override
    @Transactional
    public void sendTaskMsgReceive(McTaskMsgEntity entity) {
        // 发送配置
        String sendCode = entity.getSendCode();
        McMsgSendInfoVO sendInfoVO = mcMsgSendService.getInfoByEnCode(sendCode);
        List<McMsgSendTemplateModel> sendTemplateList = sendInfoVO.getSendTemplateList();
        if (CollectionUtil.isNotEmpty(sendTemplateList)) {
            /**
             * 解释消息模板及消息内容，解释接收人
             */
            this.saveContentAndReceive(entity, sendTemplateList);
            /**
             * 设置消息发送状态
             */
            entity.setEnabledMark(2); // 消息状态，1待发送、2已发送
            this.updateById(entity);

            SendReceiveParam sendReceiveParam = new SendReceiveParam();
            sendReceiveParam.setCurrentUserInfo(userProvider.get());
            sendReceiveParam.setTaskMsg(entity);
            sendReceiveParam.setSendTemplateList(sendTemplateList);
            sendReceiveParam.setEnabledMark(0);
            threadPoolExecutor.execute(() -> {
                try {
                    /**
                     * 根据task_msg_receive数据，渠道消息下发：首次全量下发
                     */
                    mcTaskMsgReceiveService.postSendReceive(sendReceiveParam);
                } catch (Exception e) {
                    log.error("消息中心-根据task_msg_receive数据，渠道消息下发失败，异常：" + e.getMessage());
                }
            });
        } else {
            entity.setDescription("发送配置-未配置渠道消息模板");
            this.updateById(entity);
        }
    }

    /**
     * 解释消息模板及消息内容，解释接收人
     *
     * @param taskMsg          原始消息
     * @param sendTemplateList 发送配置 消息模板组合列表
     */
    private void saveContentAndReceive(McTaskMsgEntity taskMsg, List<McMsgSendTemplateModel> sendTemplateList) {
        if (ObjectUtil.isEmpty(taskMsg) || CollectionUtil.isEmpty(sendTemplateList)) {
            return;
        }

        // 定义接收人set集合
        boolean allUsers = false; // 特殊情况：是否向全部成员发送的。
        Set<String> userSet = new HashSet<>();
        String toUserIds = taskMsg.getToUserIds();
        if (StringUtils.isNotEmpty(toUserIds)) {
            if (ALL_USERS.equals(toUserIds)) {
                // 触发特殊情况：指定为"@ALL"，则向全部成员发送
                allUsers = true;
                // 所有用户
                List<String> listId = userService.getListId();
                userSet.addAll(listId);
            } else {
                // 指定用户
                String[] toUserArr = toUserIds.split(PART_COMMA);
                Collections.addAll(userSet, toUserArr); // 通过Collections将数组转换为Set
            }
        }
        String toDeptIds = taskMsg.getToDeptIds();
        if (!allUsers && StringUtils.isNotEmpty(toDeptIds)) { // 当toUserIds为"@ALL"时忽略本参数
            // 指定部门
            List<String> orgIds = Arrays.asList(toDeptIds.split(PART_COMMA));
            // 根据部门ID批量获取用户列表
            List<UserEntity> userList = userService.getUserByOrgIds(orgIds);
            if (CollectionUtil.isNotEmpty(userList)) {
                final Set<String> deptUserSet = userList.stream().map(UserEntity::getId).collect(Collectors.toSet());
                userSet = SetUtils.union(userSet, deptUserSet);
            }
        }

        // 不同渠道消息接收人
        Map<String, List<McTaskMsgReceiveEntity>> receiveMap = new HashMap<>();
        // 不同应用消息内容
        Map<String, McTaskMsgContentEntity> contentMap = new HashMap<>();
        /**
         * 根据task_msg的发送配置，解释消息模板及消息内容，解释接收人：下发渠道+接收人=>task_msg_receive接收人记录
         * 接收人：user列表、部门列表或@ALL，@ALL时部门列表不生效
         */
        for (McMsgSendTemplateModel sendTemplate : sendTemplateList) {
            AbstractMessageChannelStrategy strategy = strategyFactory.getStrategy(sendTemplate.getMessageType());
            if (!ObjectUtil.isEmpty(strategy)) {
                // 解释消息模板及消息内容
                strategy.processMessage(taskMsg, sendTemplate, contentMap, receiveMap, userSet);
            }
        }

        // 接收人不为空
        List<McTaskMsgContentEntity> contentEntityList = new ArrayList<>();
        for (McTaskMsgContentEntity value : contentMap.values()) {
            contentEntityList.add(value);
        }
        List<McTaskMsgReceiveEntity> receiveEntityList = new ArrayList<>();
        for (List<McTaskMsgReceiveEntity> value : receiveMap.values()) {
            receiveEntityList.addAll(value);
        }
        // 设置 渠道消息未发送的尝试重发最大限制次数
        Integer autoAgainMaxNum = ObjectUtil.isEmpty(taskMsg.getAutoAgainMaxNum()) ? AUTO_AGAIN_MAX_NUM :
                taskMsg.getAutoAgainMaxNum();
        receiveEntityList.stream().forEach(item -> item.setAutoAgainMaxNum(autoAgainMaxNum));

        // 消息内容持久化
        mcTaskMsgContentService.saveBatch(contentEntityList);
        // 消息接收人持久化
        mcTaskMsgReceiveService.saveBatch(receiveEntityList);
    }

    @Override
    public List<Map<String, Object>> selectGroupByEnabledMark(McTaskMsgPagination pagination) {
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();
        // 此处根据menuId 开启数据权限
        // 创建时间
        if (StringUtils.isNotEmpty(pagination.getCreatorTime())) {
            List<String> SendTimeList = pagination.getCreatorTime();
            Long fir = Long.valueOf(SendTimeList.get(0));
            Long sec = Long.valueOf(SendTimeList.get(1));

            queryWrapper.lambda().ge(McTaskMsgEntity::getCreatorTime, new Date(fir))
                    .le(McTaskMsgEntity::getCreatorTime, DateUtil.stringToDate(DateUtil.daFormatYmd(sec) + " " +
                            "23:59:59"));
        }
        return mcTaskMsgMapper.selectGroupByEnabledMark(queryWrapper);
    }
}