package com.bringspring.system.msgcenter.service.strategy;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.msgcenter.entity.*;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.model.mcbusiness.McBusinessInfoVO;
import com.bringspring.system.msgcenter.service.McBusinessService;
import com.bringspring.system.msgcenter.service.McMsgAccountService;
import com.bringspring.system.msgcenter.service.McMsgTemplateService;
import com.bringspring.system.msgcenter.util.SensitiveWordUtil;
import com.bringspring.system.msgcenter.util.UrlUtils;
import com.bringspring.system.permission.entity.UserEntity;
import org.apache.commons.text.StringSubstitutor;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.stream.Collectors;

import static com.bringspring.system.msgcenter.constant.CommonConsts.MSG_ID;
import static com.bringspring.system.msgcenter.constant.CommonConsts.NOT_READ;
import static com.bringspring.system.msgcenter.enums.ChannelTypeEnum.getByCode;

/**
 * 定义抽象策略基类
 * 将公共方法（如getTaskMsgContent、packThirdReceive）提取到抽象类中，供具体策略复用
 */
public abstract class AbstractMessageChannelStrategy implements MessageChannelStrategy {
    @Autowired
    protected McBusinessService mcBusinessService;
    @Autowired
    private McMsgTemplateService mcMsgTemplateService;
    @Autowired
    protected McMsgAccountService mcMsgAccountService;
    @Autowired
    protected SynThirdInfoService synThirdInfoService;

    /**
     * 获取消息发送渠道类型
     *
     * @return
     */
    public abstract ChannelTypeEnum getChannelType();

    /**
     * 公共方法：生成消息内容实体
     * 模板占位符替换
     *
     * @param templateId
     * @param channelType
     * @param taskMsg
     * @param msgTemplate
     * @param account
     * @return
     */
    protected McTaskMsgContentEntity getTaskMsgContent(
            String templateId, String channelType, McTaskMsgEntity taskMsg,
            McMsgTemplateEntity msgTemplate, McMsgAccountEntity account
    ) {
        String taskMsgId = taskMsg.getId();
        // 消息场景
        McBusinessInfoVO businessInfoVO = mcBusinessService.getInfoByEnCode(msgTemplate.getMessageSource());
        // 业务消息关键词
        String businessKey = taskMsg.getBusinessKey();
        Map<String, Object> businessKeyMap = JsonUtil.stringToMap(businessKey);

        // 标题（带有占位符的）
        String title = msgTemplate.getTitle();
        // 内容（带有占位符的）
        String content = msgTemplate.getContent();
        // 摘要
        String description = null;
        // 消息链接
        String linkUrl = null;
        // 图片链接
        String picUrl = null;
        // 替换参数
        if (StringUtils.isNotEmpty(businessKeyMap)) {
            // 标题
            if (StringUtils.isNotEmpty(title)) {
                StringSubstitutor strSubstitutor = new StringSubstitutor(businessKeyMap, "{", "}");
                title = strSubstitutor.replace(title);
            } else {
                title = String.valueOf(businessKeyMap.get("title"));
            }
            // 内容
            if (StringUtils.isNotEmpty(content)) {
                StringSubstitutor strSubstitutor = new StringSubstitutor(businessKeyMap, "{", "}");
                content = strSubstitutor.replace(content);
            } else {
                content = String.valueOf(businessKeyMap.get("content"));
            }
            // 摘要
            if (businessKeyMap.containsKey("desc") && StringUtils.isNotEmpty(businessKeyMap.get("desc").toString())) {
                description = businessKeyMap.get("desc").toString();
            } else {
                description = StringUtils.isEmpty(content) ? null : StringUtils.substring(content, 0, 200);
            }
            // 消息链接
            linkUrl = this.packLinkUrl(channelType, businessKeyMap, account, taskMsgId);
            // 图片链接
            if (businessKeyMap.containsKey("picUrl") && ObjectUtil.isNotEmpty(businessKeyMap.get("picUrl"))) {
                picUrl = String.valueOf(businessKeyMap.get("picUrl"));
            }
        }

        // 封装业务消息内容实体
        McTaskMsgContentEntity taskMsgContentEntity = new McTaskMsgContentEntity();
        // 标题
        if (StringUtils.isNotEmpty(title)) {
//            title = StringUtils.stringFilter(title);
            title = SensitiveWordUtil.sensitiveFilter(title, true);
            title = StringUtils.substring(title, 0, 50);
            taskMsgContentEntity.setTitle(title);
        }
        // 内容
        if (StringUtils.isNotEmpty(content)) {
            content = SensitiveWordUtil.sensitiveFilter(content, true);
            taskMsgContentEntity.setContent(content);
        }
        // 摘要
        if (StringUtils.isNotEmpty(description)) {
            description = SensitiveWordUtil.sensitiveFilter(description, true);
            taskMsgContentEntity.setDescription(description);
        }
        // 消息链接
        taskMsgContentEntity.setLinkUrl(linkUrl);
        // 图片链接
        if (StringUtils.isNotEmpty(picUrl)) {
            taskMsgContentEntity.setPicUrl(picUrl);
        }

        taskMsgContentEntity.setId(RandomUtil.uuId());
        taskMsgContentEntity.setMsgId(taskMsgId);
        taskMsgContentEntity.setChannelType(channelType);
        taskMsgContentEntity.setTemplateId(templateId);
        taskMsgContentEntity.setMsgTemplateId(msgTemplate.getId());
        taskMsgContentEntity.setMsgAccountId(ObjectUtil.isEmpty(account) ? null : account.getId());
        taskMsgContentEntity.setContentType(businessInfoVO.getCategory());
        return taskMsgContentEntity;
    }


    /**
     * 公共方法：生成消息内容实体
     * 模板变量替换
     *
     * @param templateId
     * @param channelType
     * @param taskMsg
     * @param msgTemplate
     * @param account
     * @return
     */
    protected McTaskMsgContentEntity getTaskMsgContentKey(String templateId, String channelType,
                                                          McTaskMsgEntity taskMsg, McMsgTemplateEntity msgTemplate,
                                                          McMsgAccountEntity account) {
        String taskMsgId = taskMsg.getId();
        // 消息场景
        McBusinessInfoVO businessInfoVO = mcBusinessService.getInfoByEnCode(msgTemplate.getMessageSource());
        // 业务消息关键词
        String businessKey = taskMsg.getBusinessKey();
        Map<String, Object> businessKeyMap = JsonUtil.stringToMap(businessKey);

        McTaskMsgContentEntity taskMsgContentEntity = new McTaskMsgContentEntity();
        // 模板变量
        List<McMsgTemplateFieldEntity> fieldList = mcMsgTemplateService.getMcMsgTemplateFieldList(msgTemplate.getId());
        if (CollectionUtil.isNotEmpty(fieldList) && StringUtils.isNotEmpty(businessKeyMap)) {
            HashMap<Object, Object> fieldMap = new HashMap<>();
            for (McMsgTemplateFieldEntity field : fieldList) {
                fieldMap.put(field.getThirdField(), businessKeyMap.get(field.getKeywordCode()));
                if (Integer.valueOf(field.getIsTitle()) == 1) {
                    taskMsgContentEntity.setTitle(businessKeyMap.get(field.getKeywordCode()).toString());
                }
            }
            // 内容
            String content = JSONObject.toJSONString(fieldMap);
            if (StringUtils.isNotEmpty(content)) {
                content = SensitiveWordUtil.sensitiveFilter(content, true);
                taskMsgContentEntity.setContent(content);
            }

            // 消息链接
            taskMsgContentEntity.setLinkUrl(this.packLinkUrl(channelType, businessKeyMap, account, taskMsgId));
            // 图片链接
            if (businessKeyMap.containsKey("picUrl") && ObjectUtil.isNotEmpty(businessKeyMap.get("picUrl"))) {
                taskMsgContentEntity.setLinkUrl(String.valueOf(businessKeyMap.get("picUrl")));
            }
        }
        taskMsgContentEntity.setId(RandomUtil.uuId());
        taskMsgContentEntity.setMsgId(taskMsgId);
        taskMsgContentEntity.setChannelType(channelType);
        taskMsgContentEntity.setTemplateId(templateId);
        taskMsgContentEntity.setMsgTemplateId(msgTemplate.getId());
        taskMsgContentEntity.setMsgAccountId(ObjectUtil.isEmpty(account) ? null : account.getId());
        taskMsgContentEntity.setContentType(businessInfoVO.getCategory());
        return taskMsgContentEntity;
    }

    // 公共方法：生成第三方接收人
    protected List<McTaskMsgReceiveEntity> packThirdReceive(
            List<UserEntity> sysList,
            List<SynThirdInfoEntity> thirdList, String taskMsgId,
            String templateId, String channelType,
            String msgTemplateId, String msgAccountId, String contentId,
            List<McTaskMsgReceiveEntity> receiveList
    ) {
        if (CollectionUtil.isEmpty(receiveList)) {
            List<McTaskMsgReceiveEntity> receiveEntities = new ArrayList<>();
            for (UserEntity user : sysList) {
                McTaskMsgReceiveEntity thirdReceive = new McTaskMsgReceiveEntity();
                thirdReceive.setMsgId(taskMsgId);
                // 发送配置模板id
                thirdReceive.setTemplateId(templateId);
                // 消息模板Id
                thirdReceive.setMsgTemplateId(msgTemplateId);
                // 授权应用Id
                thirdReceive.setMsgAccountId(msgAccountId);
                // 消息内容Id
                thirdReceive.setContentId(contentId);
                // 消息接收人渠道
                thirdReceive.setChannelType(channelType);
                // 下发渠道接收人：base_user主键id
                thirdReceive.setSysUserId(user.getId());
                thirdReceive.setIsRead(NOT_READ); // 是否阅读，0否、1是
                List<SynThirdInfoEntity> tempList =
                        thirdList.stream().filter(s -> user.getId().equals(s.getSystemObjectId())).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(tempList)) {
                    tempList.stream().forEach(s -> {
                        McTaskMsgReceiveEntity thirdReceiveNew = new McTaskMsgReceiveEntity();
                        BeanUtils.copyProperties(thirdReceive, thirdReceiveNew);
                        // 下发渠道接收人：base_sync_third_info主键id
                        thirdReceiveNew.setReceiveUserId(s.getId());
                        thirdReceiveNew.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                        thirdReceiveNew.setDescription("创建，设置下发状态为‘待下发’");
                        thirdReceiveNew.setId(RandomUtil.uuId());
                        receiveEntities.add(thirdReceiveNew);
                    });
                } else {
                    thirdReceive.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                    thirdReceive.setDescription("创建，设置下发状态为‘待下发’");
                    thirdReceive.setId(RandomUtil.uuId());
                    receiveEntities.add(thirdReceive);
                }
            }
            return receiveEntities;
        } else {
            for (SynThirdInfoEntity thirdInfo : thirdList) {
                List<McTaskMsgReceiveEntity> rTempList =
                        receiveList.stream().filter(r -> thirdInfo.getSystemObjectId().equals(r.getSysUserId())).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(rTempList)) {
                    McTaskMsgReceiveEntity mcTaskMsgReceiveEntity = rTempList.get(0);
                    if (StringUtils.isEmpty(mcTaskMsgReceiveEntity.getReceiveUserId())) {
                        // 下发渠道接收人：base_sync_third_info主键id
                        mcTaskMsgReceiveEntity.setReceiveUserId(thirdInfo.getId());
                        // 发送配置模板id
                        mcTaskMsgReceiveEntity.setTemplateId(templateId);
                        // 消息模板Id
                        mcTaskMsgReceiveEntity.setMsgTemplateId(msgTemplateId);
                        // 授权应用Id
                        mcTaskMsgReceiveEntity.setMsgAccountId(msgAccountId);
                        // 消息内容Id
                        mcTaskMsgReceiveEntity.setContentId(contentId);
                    } else {
                        McTaskMsgReceiveEntity thirdReceiveNew = new McTaskMsgReceiveEntity();
                        BeanUtils.copyProperties(mcTaskMsgReceiveEntity, thirdReceiveNew);
                        thirdReceiveNew.setId(RandomUtil.uuId());
                        // 下发渠道接收人：base_sync_third_info主键id
                        thirdReceiveNew.setReceiveUserId(thirdInfo.getId());
                        // 发送配置模板id
                        thirdReceiveNew.setTemplateId(templateId);
                        // 消息模板Id
                        thirdReceiveNew.setMsgTemplateId(msgTemplateId);
                        // 授权应用Id
                        thirdReceiveNew.setMsgAccountId(msgAccountId);
                        // 消息内容Id
                        thirdReceiveNew.setContentId(contentId);
                        receiveList.add(thirdReceiveNew);
                    }
                }
            }
            return receiveList;
        }
    }

    // 公共方法：生成第三方接收人（非系统用户，第三方唯一标识直接发送）
    protected List<McTaskMsgReceiveEntity> packNonSystemReceive(Set<String> userSet, String taskMsgId,
                                                                String templateId, String channelType,
                                                                String msgTemplateId, String msgAccountId,
                                                                String contentId
    ) {
        List<McTaskMsgReceiveEntity> receiveEntities = new ArrayList<>();
        for (String openid : userSet) {
            McTaskMsgReceiveEntity thirdReceive = new McTaskMsgReceiveEntity();
            thirdReceive.setId(RandomUtil.uuId());
            thirdReceive.setMsgId(taskMsgId);
            // 发送配置模板id
            thirdReceive.setTemplateId(templateId);
            // 消息模板Id
            thirdReceive.setMsgTemplateId(msgTemplateId);
            // 授权应用Id
            thirdReceive.setMsgAccountId(msgAccountId);
            // 消息内容Id
            thirdReceive.setContentId(contentId);
            // 消息接收人渠道
            thirdReceive.setChannelType(channelType);
            // 下发渠道接收人：base_user主键id。非系统用户，第三方唯一标识直接发送，不用关联系统用户
            thirdReceive.setSysUserId(null);
            thirdReceive.setReceiveUserId(openid); // 第三方用户唯一标识
            thirdReceive.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
            thirdReceive.setIsRead(NOT_READ); // 是否阅读，0否、1是
            thirdReceive.setDescription("创建，设置下发状态为‘待下发’");
            receiveEntities.add(thirdReceive);
        }
        return receiveEntities;
    }

    protected String packLinkUrl(String channelType, Map<String, Object> businessKeyMap,
                                 McMsgAccountEntity account, String taskMsgId) {
        // 消息渠道
        ChannelTypeEnum typeEnum = getByCode(channelType);
        Object linkUrl = null;
        switch (typeEnum) {
            case SYS_MSG: // 站内消息
                if (businessKeyMap.containsKey("linkUrl")) linkUrl = businessKeyMap.get("linkUrl");
            case MAIL_MSG: // 邮件
                if (businessKeyMap.containsKey("linkUrl2")) linkUrl = businessKeyMap.get("linkUrl2");
            case SMS_MSG: // 发送短信
                if (businessKeyMap.containsKey("linkUrl3")) linkUrl = businessKeyMap.get("linkUrl3");
                break;
            case DING_MSG: // 钉钉
                if (businessKeyMap.containsKey("linkUrl4")) linkUrl = businessKeyMap.get("linkUrl4");
                break;
            case QY_MSG:
                if (businessKeyMap.containsKey("linkUrl5")) linkUrl = businessKeyMap.get("linkUrl5");
                break;
            case WX_MINIAPP:
                if (businessKeyMap.containsKey("linkUrl6")) linkUrl = businessKeyMap.get("linkUrl6");
                break;
            case WX_MP:
                if (businessKeyMap.containsKey("linkUrl7")) linkUrl = businessKeyMap.get("linkUrl7");
                break;
            case WEBHOOK:
                if (businessKeyMap.containsKey("linkUrl8")) linkUrl = businessKeyMap.get("linkUrl8");
                break;
            default:

                break;
        }

        if (ObjectUtil.isEmpty(linkUrl) && ObjectUtil.isNotEmpty(account)) {
            linkUrl = account.getAddress();
        }

        if (ObjectUtil.isNotEmpty(linkUrl)) {
            Map<String, Object> map = new HashMap<>();
            map.put(MSG_ID, taskMsgId);
            return UrlUtils.addParameter(String.valueOf(linkUrl), map);
        }
        return null;
    }
}
