package com.bringspring.system.msgcenter.service.strategy.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.external.config.mutil.DingTalkConfiguration;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.model.message.DingSentResult;
import com.bringspring.system.msgcenter.model.message.MyDingTalkModel;
import com.bringspring.system.msgcenter.service.McMsgAccountService;
import com.bringspring.system.msgcenter.service.McTaskMsgReceiveService;
import com.bringspring.system.msgcenter.service.strategy.AbstractMessageSender;
import com.bringspring.system.msgcenter.util.BlacklistUtil;
import com.bringspring.system.msgcenter.util.DingTalkUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

import static com.bringspring.system.external.constant.DingTalkConfigConsts.TO_ALL;
import static com.bringspring.system.msgcenter.constant.CommonConsts.PART_COMMA;

/**
 * 消息中心-钉钉消息下发工具类
 * dingtalk
 */
@Component
@Slf4j
public class MsgToDingService extends AbstractMessageSender {

    private final DingTalkConfiguration dingTalkConfiguration;
    private final SynThirdInfoService synThirdInfoService;
    private final McTaskMsgReceiveService mcTaskMsgReceiveService;
    private final McMsgAccountService mcMsgAccountService;

    public MsgToDingService(DingTalkConfiguration dingTalkConfiguration, SynThirdInfoService synThirdInfoService,
                            McTaskMsgReceiveService mcTaskMsgReceiveService, McMsgAccountService mcMsgAccountService) {
        this.dingTalkConfiguration = dingTalkConfiguration;
        this.synThirdInfoService = synThirdInfoService;
        this.mcTaskMsgReceiveService = mcTaskMsgReceiveService;
        this.mcMsgAccountService = mcMsgAccountService;
    }

    @Override
    protected int getBatchSize() {
        return 1000;
    }

    @Override
    public ChannelTypeEnum getChannelType() {
        return ChannelTypeEnum.DING_MSG;
    }

    @Override
    protected void sendBatchNonSystemUser(boolean isAgain,
                                          McMsgSendTemplateModel sendTemplate,
                                          McTaskMsgContentEntity taskMsgContent,
                                          List<McTaskMsgReceiveEntity> receiveListS,
                                          UserInfo currentUserInfo) {
        if (CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 钉钉消息【非系统用户】下发，接收人列表为空~·~·~·~");
        }

        List<McTaskMsgReceiveEntity> receiveList = JsonUtil.getJsonToList(receiveListS, McTaskMsgReceiveEntity.class);
        if (CollectionUtil.isNotEmpty(receiveList)) {
            /**
             * 指定接收消息的成员，成员ID列表（接收者的userid列表，最大用户列表长度100）
             */
            Set<String> touserList = receiveList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toSet());
            if (CollectionUtil.isNotEmpty(touserList)) {
                /**
                 * 调用企业API 发送消息
                 */
                McMsgAccountEntity info = mcMsgAccountService.getInfo(sendTemplate.getAccountConfigId());
                MyDingTalkModel dingTalkModel = dingTalkConfiguration.getDingTalkModel(info.getEnterpriseId());
                String appKey = dingTalkModel.getClientId();
                String appSecret = dingTalkModel.getClientSecrent();
                String agentId = dingTalkModel.getAgentId();
                String corpId = dingTalkModel.getCorpId();
                String touser = StringUtils.join(touserList, PART_COMMA);
                DingSentResult result =
                        DingTalkUtil.sendDingMessage(appKey, appSecret, agentId, touser, taskMsgContent);

                /**
                 * 处理发送结果
                 * 将‘待下发’临时处理为‘下发成功’
                 */
                String errMsg = isAgain ? "重发成功：" : "下发成功：";
                if (CollectionUtil.isNotEmpty(receiveList)) {
                    mcTaskMsgReceiveService.updateByList(isAgain, receiveList, 1, errMsg, currentUserInfo);
                }
                if (ObjectUtil.isNotEmpty(result) && StringUtils.isNotEmpty(result.getFailUser())) {
                    List<String> failUserList = Arrays.asList(result.getFailUser().split(PART_COMMA)); // 失败的userid
                    List<McTaskMsgReceiveEntity> list =
                            receiveList.stream().filter(r -> failUserList.contains(r.getReceiveUserId())).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(list)) {
                        errMsg += result.getErrMsg();
                        mcTaskMsgReceiveService.updateByList(isAgain, list, 2, errMsg, currentUserInfo);
                    }
                }
            }
        }
    }


    /**
     * 钉钉消息下发
     * 发送应用消息
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 消息
     * @param receiveListS   接收人
     */
    @Override
    protected void sendBatch(boolean isAgain,
                             McMsgSendTemplateModel sendTemplate,
                             McTaskMsgContentEntity taskMsgContent,
                             List<McTaskMsgReceiveEntity> receiveListS,
                             UserInfo currentUserInfo) {
        if (CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 钉钉消息下发，接收人列表为空~·~·~·~");
        }
        List<McTaskMsgReceiveEntity> receiveList = JsonUtil.getJsonToList(receiveListS, McTaskMsgReceiveEntity.class);

        /**
         * 判断是否全员发送
         */
        Boolean toUserIdAll = false; // true为全部成员，false为非全部成员
        // 黑名单成员
        List<McTaskMsgReceiveEntity> receiveBlacklist = BlacklistUtil.receiveListFilter(receiveList);

        if (CollectionUtil.isNotEmpty(receiveList)) {
            // 无third绑定数据
            List<McTaskMsgReceiveEntity> thirdNullList =
                    receiveList.stream().filter(r -> StringUtils.isEmpty(r.getReceiveUserId())).collect(Collectors.toList());
            // 有third绑定数据
            List<McTaskMsgReceiveEntity> thirdNotNullList =
                    receiveList.stream().filter(r -> StringUtils.isNotEmpty(r.getReceiveUserId())).collect(Collectors.toList());
            List<String> thirdInfoIds =
                    thirdNotNullList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(thirdInfoIds)) {
                List<SynThirdInfoEntity> thirdInfoList = synThirdInfoService.listByIds(thirdInfoIds);
                // third无userid
                List<SynThirdInfoEntity> thirdUseridNullList =
                        thirdInfoList.stream().filter(r -> StringUtils.isEmpty(r.getThirdObjectId())).collect(Collectors.toList());
                List<McTaskMsgReceiveEntity> useridNullList =
                        thirdNotNullList.stream().filter(r -> thirdUseridNullList.stream().map(e ->
                                e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());
                // third有userid
                List<SynThirdInfoEntity> thirdUseridNotNullList =
                        thirdInfoList.stream().filter(r -> StringUtils.isNotEmpty(r.getThirdObjectId())).collect(Collectors.toList());
                List<McTaskMsgReceiveEntity> useridNotNullList =
                        thirdNotNullList.stream().filter(r -> thirdUseridNotNullList.stream().map(e ->
                                e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());

                // 多例企业配置
                Map<String, DingSentResult> resultMap = new HashMap<>();
                if (!toUserIdAll) {
                    /**
                     * 指定接收消息的成员，成员ID列表（接收者的userid列表，最大用户列表长度100）
                     */
                    Map<String, List<String>> corpUserMap = new HashMap<>();
                    if (CollectionUtil.isNotEmpty(thirdUseridNotNullList)) {
                        for (SynThirdInfoEntity stiEntity : thirdUseridNotNullList) {
                            if (StringUtils.isNotEmpty(stiEntity.getThirdObjectId())) {
                                final String corpid = stiEntity.getCropId();
                                final String userid = stiEntity.getThirdObjectId();
                                if (corpUserMap.containsKey(corpid)) {
                                    corpUserMap.get(corpid).add(userid);
                                } else {
                                    List<String> useridList = new ArrayList<>();
                                    useridList.add(userid);
                                    corpUserMap.put(corpid, useridList);
                                }
                            }
                        }
                    }

                    if (CollectionUtil.isNotEmpty(corpUserMap)) {
                        /**
                         * 调用企业API 发送消息
                         */
                        for (Map.Entry<String, List<String>> entry : corpUserMap.entrySet()) {
                            MyDingTalkModel dingTalkModel = dingTalkConfiguration.getDingTalkModel(entry.getKey());
                            String appKey = dingTalkModel.getClientId();
                            String appSecret = dingTalkModel.getClientSecrent();
                            String agentId = dingTalkModel.getAgentId();
                            String corpId = dingTalkModel.getCorpId();
                            String touser = StringUtils.join(entry.getValue(), PART_COMMA);
                            DingSentResult result =
                                    DingTalkUtil.sendDingMessage(appKey, appSecret, agentId, touser, taskMsgContent);
                            System.out.println("~·~发送消息：" + corpId + result);
                            resultMap.put(corpId, result);// 不合法的userid
                        }
                    }
                } else {
                    /**
                     * 特殊情况：指定为"@ALL"，则向企业应用的全部成员发送
                     */
                    Map<String, MyDingTalkModel> dingTalkModels = dingTalkConfiguration.getDingTalkModels();
                    for (MyDingTalkModel dingTalkModel : dingTalkModels.values()) {
                        String appKey = dingTalkModel.getClientId();
                        String appSecret = dingTalkModel.getClientSecrent();
                        String agentId = dingTalkModel.getAgentId();
                        String corpId = dingTalkModel.getCorpId();
                        /**
                         * 调用企业API 发送消息
                         */
                        DingSentResult result =
                                DingTalkUtil.sendDingMessage(appKey, appSecret, agentId, TO_ALL, taskMsgContent);

                        System.out.println("~·~发送消息@all：" + corpId + result);
                        resultMap.put(corpId, result);// 不合法的userid
                    }
                }

                /**
                 * 处理发送结果
                 */
                if (CollectionUtil.isNotEmpty(resultMap)) {
                    this.dingSendMessageResult(isAgain, taskMsgContent, useridNotNullList, thirdUseridNotNullList,
                            resultMap, currentUserInfo);
                }

                if (CollectionUtil.isNotEmpty(useridNullList)) {
                    String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "third无userid";
                    mcTaskMsgReceiveService.updateByList(isAgain, useridNullList, 2, errMsg, currentUserInfo);
                }
            }
            if (CollectionUtil.isNotEmpty(thirdNullList)) {
                String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "无third绑定数据";
                mcTaskMsgReceiveService.updateByList(isAgain, thirdNullList, 2, errMsg, currentUserInfo);
            }
        }
        if (CollectionUtil.isNotEmpty(receiveBlacklist)) {
            // 黑名单
            mcTaskMsgReceiveService.updateBlacklist(receiveBlacklist, currentUserInfo);
        }
    }

    /**
     * 执行 钉钉应用消息post下发 结果处理
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 消息内容表信息
     * @param receiveList    钉钉当前处理接收人集合
     * @param resultMap      钉钉消息发送结果集合
     */
    public void dingSendMessageResult(boolean isAgain, McTaskMsgContentEntity taskMsgContent,
                                      List<McTaskMsgReceiveEntity> receiveList,
                                      List<SynThirdInfoEntity> thirdInfoList,
                                      Map<String, DingSentResult> resultMap, UserInfo currentUserInfo) {
        /**
         * 将‘待下发’临时处理为‘下发成功’
         */
        if (CollectionUtil.isNotEmpty(receiveList)) {
            String errMsg = isAgain ? "重发成功：" : "下发成功：";
            mcTaskMsgReceiveService.updateByList(isAgain, receiveList, 1, errMsg, currentUserInfo);
        }

        /**
         * 处理钉钉下发接口返回的 不合法的userid
         */
        if (CollectionUtil.isNotEmpty(resultMap)) {
            for (Map.Entry<String, DingSentResult> entry : resultMap.entrySet()) {
                DingSentResult result = entry.getValue();
                if (ObjectUtil.isNotEmpty(result) && StringUtils.isNotEmpty(result.getFailUser())) {
                    List<String> failUserList = Arrays.asList(result.getFailUser().split(PART_COMMA)); // 失败的userid
                    List<SynThirdInfoEntity> failThirdList =
                            thirdInfoList.stream().filter(r -> failUserList.contains(r.getThirdObjectId())).collect(Collectors.toList());
                    List<McTaskMsgReceiveEntity> list =
                            receiveList.stream().filter(r -> failThirdList.stream().map(e ->
                                    e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(list)) {
                        String errMsg = isAgain ? "重发失败：" : "下发失败：" + result.getErrMsg();
                        mcTaskMsgReceiveService.updateByList(isAgain, list, 2, errMsg, currentUserInfo);
                    }
                }
            }
        }

    }

}
