package com.bringspring.system.permission.aop;

import com.bringspring.common.util.UserProvider;
import com.bringspring.system.permission.constant.PermissionConstant;
import com.bringspring.system.permission.constant.PermissionConst;
import com.bringspring.system.permission.entity.OrganizeRelationEntity;
import com.bringspring.system.permission.model.user.form.UserCrForm;
import com.bringspring.system.permission.model.user.form.UserUpForm;
import com.bringspring.system.permission.model.userrelation.UserRelationForm;
import com.bringspring.system.permission.service.OrganizeRelationService;
import com.bringspring.system.permission.service.PositionService;
import com.bringspring.system.permission.util.PermissionAspectUtil;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户分级管理
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/15 17:12
 */
@Slf4j
@Aspect
@Component
public class PermissionUserAspect implements PermissionAdminBase{

    @Autowired
    private UserProvider userProvider;
    @Autowired
    private OrganizeRelationService organizeRelationService;
    @Autowired
    private PositionService positionService;

    /**
     * 分级管理切点
     */
    @Pointcut("@annotation(com.bringspring.common.annotation.UserPermission)")
    public void pointcut() {
    }

    /**
     * 分级管理切点
     *
     * @param pjp
     * @return
     * @throws Throwable
     */
    @Around("pointcut()")
    public Object around(ProceedingJoinPoint pjp) throws Throwable {
        return PermissionAdminBase.permissionCommon(pjp, userProvider, this);
    }

    @Override
    public Boolean detailPermission(ProceedingJoinPoint pjp, String operatorUserId, String methodName) {
        switch (methodName){
            case PermissionConstant.METHOD_CREATE:
                UserCrForm userCrForm = (UserCrForm) pjp.getArgs()[0];
                return PermissionAspectUtil.getPermitByOrgIds(
                                        // 操作目标对象组织ID集合
                                        userCrForm.getOrganizeId(),
                                        operatorUserId,
                                        PermissionConstant.METHOD_CREATE);
            case PermissionConstant.METHOD_UPDATE:
                return PermissionAspectUtil.getPermitByUserId(
                                        // 操作目标对象的ID
                                        String.valueOf(pjp.getArgs()[0]),
                                        operatorUserId,
                                        PermissionConstant.METHOD_UPDATE) ||
                                        PermissionAspectUtil.getPermitByOrgIds(
                                        // 操作目标对象组织ID集合
                                        ((UserUpForm) pjp.getArgs()[1]).getOrganizeId(),
                                        operatorUserId,
                                        PermissionConstant.METHOD_UPDATE);
            case PermissionConstant.METHOD_MODIFY_PW:
                return PermissionAspectUtil.getPermitByUserId(
                                        // 操作目标对象的ID
                                        String.valueOf(pjp.getArgs()[0]),
                                        operatorUserId,
                                        PermissionConstant.METHOD_UPDATE);
            case PermissionConstant.METHOD_DELETE:
                return PermissionAspectUtil.getPermitByUserId(
                                        // 操作目标对象的ID
                                        pjp.getArgs()[0].toString(),
                                        operatorUserId,
                                        PermissionConstant.METHOD_DELETE);
            case PermissionConstant.METHOD_SAVE:
                String objId = pjp.getArgs()[0].toString();
                UserRelationForm userRelationForm = (UserRelationForm)pjp.getArgs()[1];

                List<String> orgIds = new ArrayList<>();
                if(userRelationForm.getObjectType().equals(PermissionConst.ROLE)){
                    // 角色目前修改为只有超管才能够修改
                    if(userProvider.get().getIsAdministrator()){
                        return true;
                    }
                    orgIds.addAll(organizeRelationService.getRelationListByRoleId(objId).stream().map(OrganizeRelationEntity::getOrganizeId).collect(Collectors.toList()));
                    return PermissionAspectUtil.getPermitByOrgIds(
                            // 操作目标对象组织ID集合
                            String.join(",", orgIds),
                            operatorUserId,
                            PermissionConstant.METHOD_UPDATE);
                }else {
                    orgIds.add(positionService.getInfo(objId).getOrganizeId());
                    return PermissionAspectUtil.getPermitByOrgIds(
                            String.join(",", orgIds),
                            operatorUserId,
                            PermissionConstant.METHOD_UPDATE);
                }
            default:
                return false;
        }
    }





}
