package com.bringspring.system.permission.service.strategy;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.system.permission.model.authorize.ConditionModel;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;


/**
 * 具体策略类:
 * 定义：实现策略接口的具体算法类。
 * 角色：提供策略接口中定义的算法的具体实现。
 * 职责：根据条件模型ConditionModel中的条件信息，生成QueryWrapper的查询条件。
 * 适用性：该策略类适用于text类型的条件。
 * 依赖：该策略类依赖于UserProvider，用于获取当前用户信息。
 * 关联：该策略类与ConditionStrategy接口关联，实现ConditionStrategy接口的generateCondition方法。
 * 协作：该策略类与ConditionModel、QueryWrapper、UserProvider等类协作，完成具体的查询条件生成。
 * <p>
 * text类型的条件策略
 */
@Component
public class TextConditionStrategy implements ConditionStrategy {

    @Autowired
    private ConditionStrategyFactory conditionStrategyFactory;

    @Autowired
    private ContentStringInputText contentStringInputText;

    @Override
    public void generateCondition(QueryWrapper<T> tw, String logic, ConditionModel.ConditionItemModel fieldItem) {
        String fieldSymbol = fieldItem.getOp();
        String fieldContent = fieldItem.getValue();
        /**
         * 字段类型为text时，处理方式如下：
         * 字段内容：input任意文本、@userId当前用户、@organizeId当前组织、@companyId当前公司
         * 符号：==等于、<>不等于、like包含、notLike不包含、null为空、notNull不为空
         * 处理方式：
         * 1. 字段内容处理
         *    a. input任意文本 -> 不为空
         *    b. @userId当前用户 -> userProvider.get().getUserId()
         *    c. @organizeId当前组织 -> userProvider.get().getDepartmentId() || userProvider.get().getOrganizeId()
         *    d. @companyId当前公司 -> userProvider.get().getOrganizeId()
         * 2. 根据符号设置tw的查询条件
         */
        // 1. 字段内容处理
        Object value;
        if (fieldContent.startsWith("@")) {
            value = conditionStrategyFactory.getFieldContentText(fieldContent, fieldSymbol);
        } else {
            // 处理input任意文本
            value = contentStringInputText.convertFieldContent(fieldContent, fieldSymbol);
        }

        // 2. 根据符号设置tw的查询条件
        SymbolCommon.compare(logic, fieldItem.getField(), fieldSymbol, value, tw);
    }

    @Override
    public Integer key() {
        return 5;
    }

}
