package com.bringspring.visualdev.portal.controller;


import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.DownloadVO;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.*;
import com.bringspring.common.util.enums.DictionaryDataEnum;
import com.bringspring.common.util.enums.ExportModelTypeEnum;
import com.bringspring.common.util.enums.ModuleTypeEnum;
import com.bringspring.common.util.file.FileExport;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.DictionaryDataEntity;
import com.bringspring.system.base.entity.ModuleEntity;
import com.bringspring.system.base.service.DictionaryDataService;
import com.bringspring.system.base.service.ModuleService;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.system.permission.entity.AuthorizeEntity;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.AuthorizeService;
import com.bringspring.system.permission.service.UserService;
import com.bringspring.visualdev.base.model.VisualFunctionModel;
import com.bringspring.visualdev.portal.entity.PortalEntity;
import com.bringspring.visualdev.portal.model.*;
import com.bringspring.visualdev.portal.service.PortalService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 可视化门户
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Slf4j
@RestController
@Api(tags = "可视化门户", value = "Portal")
@RequestMapping("/api/visualdev/Portal")
public class PortalController {

    @Autowired
    private UserService userService;
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private PortalService portalService;
    @Autowired
    private AuthorizeService authorizeService;
    @Autowired
    private DictionaryDataService dictionaryDataService;
    @Autowired
    private CacheKeyUtil cacheKeyUtil;
    @Autowired
    private CacheUtil cacheUtil;
    @Autowired
    private FileExport fileExport;
    @Autowired
    private ConfigValueUtil configValueUtil;
    @Autowired
    private ModuleService moduleService;


    @GetMapping
    public ActionResult list(PortalPagination portalPagination) {
        List<PortalEntity> data = portalService.getList(portalPagination);
        List<String> userId = data.stream().map(t -> t.getCreatorUserId()).collect(Collectors.toList());
        List<String> lastUserId = data.stream().map(t -> t.getLastModifyUserId()).collect(Collectors.toList());
        List<UserEntity> userEntities = userService.getUserName(userId);
        List<UserEntity> lastUserIdEntities = userService.getUserName(lastUserId);
        List<DictionaryDataEntity> dictionList = dictionaryDataService.getList(DictionaryDataEnum.VISUALDEV_PORTAL.getDictionaryTypeId());
        List<VisualFunctionModel> modelAll = new LinkedList<>();
        for (PortalEntity entity : data) {
            VisualFunctionModel model = JsonUtil.getJsonToBean(entity, VisualFunctionModel.class);
            DictionaryDataEntity dataEntity = dictionList.stream().filter(t -> t.getId().equals(entity.getCategory())).findFirst().orElse(null);
            if (dataEntity != null) {
                model.setCategory(dataEntity.getFullName());
                UserEntity creatorUser = userEntities.stream().filter(t -> t.getId().equals(model.getCreatorUserId())).findFirst().orElse(null);
                if (creatorUser != null) {
                    model.setCreatorUserId(creatorUser.getRealName());
                } else {
                    model.setCreatorUserId("");
                }
                UserEntity lastmodifyuser = lastUserIdEntities.stream().filter(t -> t.getId().equals(model.getLastModifyUserId())).findFirst().orElse(null);
                if (lastmodifyuser != null) {
                    model.setLastModifyUserId(lastmodifyuser.getRealName());
                } else {
                    model.setLastModifyUserId("");
                }
                if (Objects.isNull(model.getSortCode())) {
                    model.setSortCode(0L);
                }
                modelAll.add(model);
            }
        }
        modelAll = modelAll.stream().sorted(Comparator.comparing(VisualFunctionModel::getSortCode)).collect(Collectors.toList());
        PaginationVO paginationVO = JsonUtil.getJsonToBean(portalPagination, PaginationVO.class);
        return ActionResult.page(modelAll, paginationVO);
    }

    @GetMapping("/Selector")
    public ActionResult listSelect(String type) {
        List<PortalSelectModel> modelList = portalService.getList(type);
        List<SumTree<PortalSelectModel>> sumTrees = TreeDotUtils.convertListToTreeDot(modelList);
        List<PortalSelectVO> listVO = JsonUtil.getJsonToList(sumTrees, PortalSelectVO.class);
        ListVO<PortalSelectVO> treeVo = new ListVO<>();
        treeVo.setList(listVO);
        return ActionResult.success(treeVo);
    }

    @GetMapping("/{id}")
    public ActionResult<PortalInfoVO> info(@PathVariable("id") String id) {
        PortalEntity entity = portalService.getInfo(id);
        PortalInfoVO vo = JsonUtil.getJsonToBean(JsonUtilEx.getObjectToStringDateFormat(entity, "yyyy-MM-dd HH:mm:ss"), PortalInfoVO.class);

        return ActionResult.success(vo);
    }

    @GetMapping("/{id}/auth")
    public ActionResult<PortalInfoAuthVO> infoAuth(@PathVariable("id") String id) {
        UserInfo userInfo = userProvider.get();
        UserEntity info = userService.getInfo(userInfo.getUserId());
        List<String> strings = Arrays.asList(info.getRoleId().split(","));
        if ((userInfo != null && strings != null)) {
            for (String roleId : strings) {
                List<AuthorizeEntity> authByObjectId = authorizeService.getListByObjectId(roleId);
                List<AuthorizeEntity> authorizeEntityList = authByObjectId.stream().filter(t -> "portal".equals(t.getItemType())).collect(Collectors.toList());
                for (AuthorizeEntity authorizeEntity : authorizeEntityList) {
                    if (id.equals(authorizeEntity.getItemId())) {
                        PortalEntity entity = portalService.getInfo(id);
                        PortalInfoAuthVO vo = JsonUtil.getJsonToBean(JsonUtilEx.getObjectToStringDateFormat(entity, "yyyy-MM-dd HH:mm:ss"), PortalInfoAuthVO.class);
                        return ActionResult.success(vo);
                    }
                }

                // 根据门户id查询菜单，判断用户角色是否有菜单权限
                List<String> moduleListByProperty = moduleService.getModuleListByProperty(id).stream().map(ModuleEntity::getId).collect(Collectors.toList());
                List<String> roleAuth = authByObjectId.stream().map(AuthorizeEntity::getItemId).collect(Collectors.toList());
                roleAuth.retainAll(moduleListByProperty);
                if (ObjectUtil.isNotEmpty(roleAuth)) {
                    PortalEntity entity = portalService.getInfo(id);
                    PortalInfoAuthVO vo = JsonUtil.getJsonToBean(JsonUtilEx.getObjectToStringDateFormat(entity, "yyyy-MM-dd HH:mm:ss"), PortalInfoAuthVO.class);
                    return ActionResult.success(vo);
                }

            }
            if (userInfo.getIsAdministrator()) {
                PortalEntity entity = portalService.getInfo(id);
                PortalInfoAuthVO vo = JsonUtil.getJsonToBean(JsonUtilEx.getObjectToStringDateFormat(entity, "yyyy-MM-dd HH:mm:ss"), PortalInfoAuthVO.class);
                return ActionResult.success(vo);
            }
        }

        return ActionResult.fail("您没有此门户使用权限，请重新设置");
    }


    @DeleteMapping("/{id}")
    @DSTransactional
    public ActionResult delete(@PathVariable("id") String id) {
        PortalEntity entity = portalService.getInfo(id);
        if (entity != null) {
            portalService.delete(entity);
            QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper();
            queryWrapper.lambda().eq(AuthorizeEntity::getItemId, id);
            authorizeService.remove(queryWrapper);
        }
        return ActionResult.success(MsgCode.SU003.get());
    }

    @PostMapping()
    @DSTransactional
    public ActionResult create(@RequestBody @Valid PortalCrForm portalCrForm) {
        PortalEntity entity = JsonUtil.getJsonToBean(portalCrForm, PortalEntity.class);
        entity.setId(RandomUtil.uuId());
        portalService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 复制功能
     *
     * @param id
     * @return
     */
    @ApiOperation("复制功能")
    @PostMapping("/{id}/Actions/Copy")
    public ActionResult copyInfo(@PathVariable("id") String id) {
        PortalEntity entity = portalService.getInfo(id);
        entity.setEnabledMark(0);
        String copyNum = UUID.randomUUID().toString().substring(0, 5);
        entity.setFullName(entity.getFullName() + ".副本" + copyNum);
        entity.setLastModifyTime(null);
        entity.setLastModifyUserId(null);
        entity.setId(RandomUtil.uuId());
        entity.setEnCode(entity.getEnCode() + copyNum);
        PortalEntity entity1 = JsonUtil.getJsonToBean(entity, PortalEntity.class);
        if (entity1.getEnCode().length() > 50 || entity1.getFullName().length() > 50) {
            return ActionResult.fail("已到达该模板复制上限，请复制源模板");
        }
        portalService.create(entity1);
        return ActionResult.success(MsgCode.SU007.get());
    }


    @PutMapping("/{id}")
    @DSTransactional
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid PortalUpForm portalUpForm) {
        PortalEntity entity = JsonUtil.getJsonToBean(portalUpForm, PortalEntity.class);
        boolean flag = portalService.update(id, entity);
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());

    }


    /**
     * 门户权限列表
     *
     * @param id 对象主键
     * @return
     */
    @DSTransactional
    @ApiOperation("设置默认门户")
    @PutMapping("/{id}/Actions/SetDefault")
    public ActionResult setDefault(@PathVariable("id") String id) {
        UserEntity userEntity = userService.getInfo(userProvider.get().getUserId());
        if (userEntity != null) {
            userEntity.setPortalId(id);
            userService.update(userEntity);
            String catchKey = cacheKeyUtil.getAllUser();
            if (cacheUtil.exists(catchKey)) {
                cacheUtil.remove(catchKey);
            }
        } else {
            return ActionResult.fail("设置失败，用户不存在");
        }
        return ActionResult.success(MsgCode.SU016.get());
    }

    @ApiOperation("门户导出")
    @PostMapping("/{modelId}/Actions/ExportData")
    public ActionResult exportFunction(@PathVariable("modelId") String modelId) {
        PortalEntity entity = portalService.getInfo(modelId);
        if (entity != null) {
            PortalExportDataVo vo = new PortalExportDataVo();
            BeanUtils.copyProperties(entity, vo);
            vo.setModelType(ExportModelTypeEnum.Portal.getMessage());
            DownloadVO downloadVO = fileExport.exportFile(vo, configValueUtil.getTemporaryFilePath(), entity.getFullName(), ModuleTypeEnum.VISUAL_PORTAL.getTableName());
            return ActionResult.success(downloadVO);
        } else {
            return ActionResult.success("并无该条数据");
        }
    }

    @ApiOperation("门户导入")
    @PostMapping(value = "/Model/Actions/ImportData", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public ActionResult importFunction(@RequestPart("file") MultipartFile multipartFile) {
        //判断是否为.json结尾
        if (FileUtil.existsSuffix(multipartFile, ModuleTypeEnum.VISUAL_PORTAL.getTableName())) {
            return ActionResult.fail(MsgCode.IMP002.get());
        }
        //获取文件内容
        String fileContent = FileUtil.getFileContent(multipartFile, configValueUtil.getTemporaryFilePath());
        PortalExportDataVo vo = JsonUtil.getJsonToBean(fileContent, PortalExportDataVo.class);
        if (vo.getModelType() == null || !vo.getModelType().equals(ExportModelTypeEnum.Portal.getMessage())) {
            return ActionResult.fail("请导入对应功能的json文件");
        }
        String modelId = vo.getId();
        if (StringUtils.isNotEmpty(modelId)) {
            PortalEntity portalEntity = portalService.getInfo(modelId);
            if (portalEntity != null) {
                return ActionResult.fail("已存在相同功能");
            }
        }
        PortalEntity entity = JsonUtil.getJsonToBean(fileContent, PortalEntity.class);
        portalService.create(entity);
        return ActionResult.success(MsgCode.IMP001.get());
    }
}
