package com.bringspring.workflow.engine.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.DownloadVO;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.model.FormAllModel;
import com.bringspring.common.model.FormEnum;
import com.bringspring.common.model.task.LocalTaskModel;
import com.bringspring.common.model.visiual.ComponentKeyConsts;
import com.bringspring.common.model.visiual.FormCloumnUtil;
import com.bringspring.common.model.visiual.FormDataField;
import com.bringspring.common.model.visiual.FormDataModel;
import com.bringspring.common.model.visiual.RecursionForm;
import com.bringspring.common.model.visiual.TableModel;
import com.bringspring.common.model.visiual.fields.FieLdsModel;
import com.bringspring.common.util.FileUtil;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.util.context.SpringContext;
import com.bringspring.common.util.enums.ModuleTypeEnum;
import com.bringspring.common.util.file.fileinfo.DataFileExport;
import com.bringspring.system.base.entity.DictionaryDataEntity;
import com.bringspring.system.base.exception.WorkFlowException;
import com.bringspring.system.scheduletask.model.TaskMethodsVO;
import com.bringspring.workflow.engine.annotation.FlowCallBackMethod;
import com.bringspring.workflow.engine.entity.FlowEngineEntity;
import com.bringspring.workflow.engine.entity.FlowTaskEntity;
import com.bringspring.workflow.engine.enums.FlowTaskStatusEnum;
import com.bringspring.workflow.engine.model.flowengine.FlowEngineCrForm;
import com.bringspring.workflow.engine.model.flowengine.FlowEngineInfoVO;
import com.bringspring.workflow.engine.model.flowengine.FlowEngineListVO;
import com.bringspring.workflow.engine.model.flowengine.FlowEngineModel;
import com.bringspring.workflow.engine.model.flowengine.FlowEngineSelectVO;
import com.bringspring.workflow.engine.model.flowengine.FlowEngineUpForm;
import com.bringspring.workflow.engine.model.flowengine.FlowExportModel;
import com.bringspring.workflow.engine.model.flowengine.FlowPageListVO;
import com.bringspring.workflow.engine.model.flowengine.FlowPagination;
import com.bringspring.workflow.engine.model.flowengine.PaginationFlowEngine;
import com.bringspring.workflow.engine.service.FlowEngineService;
import com.bringspring.workflow.engine.service.FlowTaskService;
import com.bringspring.workflow.engine.util.ServiceAllUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * 流程设计
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Api(tags = "流程引擎", value = "FlowEngine")
@RestController
@RequestMapping("/api/workflow/Engine/FlowEngine")
public class FlowEngineController {

    @Autowired
    private FlowEngineService flowEngineService;
    @Autowired
    private FlowTaskService flowTaskService;
    @Autowired
    private DataFileExport fileExport;
    @Autowired
    private ServiceAllUtil serviceUtil;
    @Autowired
    private ConfigValueUtil configValueUtil;

    @Autowired
    private ApplicationContext applicationContext;

    /**
     * 获取流程设计列表
     *
     * @return
     */
    @ApiOperation("获取流程引擎列表")
    @GetMapping
    public ActionResult list(FlowPagination pagination) {
        List<FlowEngineEntity> list = flowEngineService.getPageList(pagination);
        List<DictionaryDataEntity> dictionList = serviceUtil.getDictionName(list.stream().map(t -> t.getCategory()).collect(Collectors.toList()));
        for (FlowEngineEntity entity : list) {
            DictionaryDataEntity dataEntity = dictionList.stream().filter(t -> t.getEnCode().equals(entity.getCategory())).findFirst().orElse(null);
            entity.setCategory(dataEntity != null ? dataEntity.getFullName() : "");
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        List<FlowPageListVO> listVO = JsonUtil.getJsonToList(list, FlowPageListVO.class);
        return ActionResult.page(listVO, paginationVO);
    }

    /**
     * 获取流程设计列表
     *
     * @return
     */
    @ApiOperation("流程引擎下拉框")
    @GetMapping("/Selector")
    public ActionResult<ListVO<FlowEngineListVO>> listSelect(Integer type) {
        PaginationFlowEngine pagination = new PaginationFlowEngine();
        pagination.setFormType(type);
        pagination.setEnabledMark(1);
        pagination.setType(0);
        List<FlowEngineListVO> treeList = flowEngineService.getTreeList(pagination, true);
        ListVO vo = new ListVO();
        vo.setList(treeList);
        return ActionResult.success(vo);
    }

    /**
     * 主表属性
     *
     * @return
     */
    @ApiOperation("表单主表属性")
    @GetMapping("/{id}/FormDataFields")
    public ActionResult<ListVO<FormDataField>> getFormDataField(@PathVariable("id") String id) throws WorkFlowException {
        FlowEngineEntity entity = flowEngineService.getInfo(id);
        List<FormDataField> formDataFieldList = new ArrayList<>();
        if (entity.getFormType() == 1) {
            List<FlowEngineModel> list = JsonUtil.getJsonToList(entity.getFormData(), FlowEngineModel.class);
            for (FlowEngineModel model : list) {
                FormDataField formDataField = new FormDataField();
                formDataField.setLabel(model.getFiledName());
                formDataField.setVModel(model.getFiledId());

                formDataFieldList.add(formDataField);

            }
        } else {
            //formTempJson
            FormDataModel formData = JsonUtil.getJsonToBean(entity.getFormData(), FormDataModel.class);
            List<FieLdsModel> list = JsonUtil.getJsonToList(formData.getFields(), FieLdsModel.class);
            List<TableModel> tableModelList = JsonUtil.getJsonToList(entity.getFlowTables(), TableModel.class);
            List<FormAllModel> formAllModel = new ArrayList<>();
            RecursionForm recursionForm = new RecursionForm(list, tableModelList);
            FormCloumnUtil.recursionForm(recursionForm, formAllModel);
            //主表数据
            List<FormAllModel> mast = formAllModel.stream().filter(t -> FormEnum.mast.getMessage().equals(t.getKeyName())).collect(Collectors.toList());
            for (FormAllModel model : mast) {
                FieLdsModel fieLdsModel = model.getFormColumnModel().getFieLdsModel();
                String vmodel = fieLdsModel.getVModel();
                String keyName = fieLdsModel.getConfig().getKeyName();
                if (StringUtils.isNotEmpty(vmodel) && !ComponentKeyConsts.RELATIONFORM.equals(keyName) && !ComponentKeyConsts.RELATIONFLOW.equals(keyName)) {
                    FormDataField formDataField = new FormDataField();
                    formDataField.setLabel(fieLdsModel.getConfig().getLabel());
                    formDataField.setVModel(fieLdsModel.getVModel());
                    formDataFieldList.add(formDataField);
                }
            }
        }
        ListVO<FormDataField> listVO = new ListVO();
        listVO.setList(formDataFieldList);
        return ActionResult.success(listVO);
    }

    /**
     * 列表
     *
     * @return
     */
    @ApiOperation("表单列表")
    @GetMapping("/{id}/FieldDataSelect")
    public ActionResult<ListVO<FlowEngineSelectVO>> getFormData(@PathVariable("id") String id) {
        List<FlowTaskEntity> flowTaskList = flowTaskService.getTaskList(id, FlowTaskEntity::getId, FlowTaskEntity::getFullName, FlowTaskEntity::getEnCode).stream().filter(t -> FlowTaskStatusEnum.Adopt.getCode().equals(t.getStatus())).collect(Collectors.toList());
        List<FlowEngineSelectVO> vo = new ArrayList<>();
        for (FlowTaskEntity taskEntity : flowTaskList) {
            FlowEngineSelectVO selectVO = JsonUtil.getJsonToBean(taskEntity, FlowEngineSelectVO.class);
            selectVO.setFullName(taskEntity.getFullName() + "/" + taskEntity.getEnCode());
            vo.add(selectVO);
        }
        ListVO listVO = new ListVO();
        listVO.setList(vo);
        return ActionResult.success(listVO);
    }

    /**
     * 可见引擎下拉框
     *
     * @return
     */
    @ApiOperation("可见引擎下拉框")
    @GetMapping("/ListAll")
    public ActionResult<ListVO<FlowEngineListVO>> listAll() {
        PaginationFlowEngine pagination = new PaginationFlowEngine();
        List<FlowEngineListVO> treeList = flowEngineService.getTreeList(pagination, false);
        ListVO vo = new ListVO();
        vo.setList(treeList);
        return ActionResult.success(vo);
    }

    /**
     * 可见的流程引擎列表
     *
     * @return
     */
    @ApiOperation("可见的流程引擎列表")
    @GetMapping("/PageListAll")
    public ActionResult<PageListVO<FlowPageListVO>> listAll(FlowPagination pagination) {
        List<FlowEngineEntity> list = flowEngineService.getListAll(pagination, true);
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        List<FlowPageListVO> listVO = JsonUtil.getJsonToList(list, FlowPageListVO.class);
        return ActionResult.page(listVO, paginationVO);
    }

    @ApiOperation("可见的流程引擎列表")
    @GetMapping("/PageListAllGroup")
    public ActionResult listAllGroup(FlowPagination pagination) {
        List<FlowEngineEntity> list = flowEngineService.getListAll(pagination, false);
        Map<String, List<FlowEngineEntity>> collect = list.stream().collect(Collectors.groupingBy(FlowEngineEntity::getCategory));
        List returnList = CollectionUtil.list(false);
        for (String key : collect.keySet()) {
            Map map = MapUtil.newHashMap();
            map.put("enCode", key);
            map.put("data", collect.get(key));
            returnList.add(map);
        }
        return ActionResult.success(returnList);
    }

    /**
     * 获取流程引擎信息
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("获取流程引擎信息")
    @GetMapping("/{id}")
    public ActionResult<FlowEngineInfoVO> info(@PathVariable("id") String id) throws WorkFlowException {
        FlowEngineEntity flowEntity = flowEngineService.getInfo(id);
        FlowEngineInfoVO vo = JsonUtil.getJsonToBean(flowEntity, FlowEngineInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 新建流程设计
     *
     * @return
     */
    @ApiOperation("新建流程引擎")
    @PostMapping
    public ActionResult create(@RequestBody @Valid FlowEngineCrForm flowEngineCrForm) throws WorkFlowException {
        FlowEngineEntity flowEngineEntity = JsonUtil.getJsonToBean(flowEngineCrForm, FlowEngineEntity.class);
        if (flowEngineService.isExistByFullName(flowEngineEntity.getFullName(), flowEngineEntity.getId())) {
            return ActionResult.fail("流程名称不能重复");
        }
        if (flowEngineService.isExistByEnCode(flowEngineEntity.getEnCode(), flowEngineEntity.getId())) {
            return ActionResult.fail("流程编码不能重复");
        }
        flowEngineService.create(flowEngineEntity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新流程设计
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("更新流程引擎")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid FlowEngineUpForm flowEngineUpForm) throws WorkFlowException {
        FlowEngineEntity flowEngineEntity = JsonUtil.getJsonToBean(flowEngineUpForm, FlowEngineEntity.class);
        if (flowEngineService.isExistByFullName(flowEngineUpForm.getFullName(), id)) {
            return ActionResult.fail("流程名称不能重复");
        }
        if (flowEngineService.isExistByEnCode(flowEngineUpForm.getEnCode(), id)) {
            return ActionResult.fail("流程编码不能重复");
        }
        boolean flag = flowEngineService.updateVisible(id, flowEngineEntity);
        if (flag == false) {
            return ActionResult.success(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.getDesc());
    }

    /**
     * 删除流程设计
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("删除流程引擎")
    @DeleteMapping("/{id}")
    public ActionResult<String> delete(@PathVariable("id") String id) throws WorkFlowException {
        FlowEngineEntity entity = flowEngineService.getInfo(id);
        flowEngineService.delete(entity);
        return ActionResult.success(MsgCode.SU003.get());
    }

    /**
     * 复制流程表单
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("复制流程表单")
    @PostMapping("/{id}/Actions/Copy")
    public ActionResult copy(@PathVariable("id") String id) throws WorkFlowException {
        FlowEngineEntity flowEngineEntity = flowEngineService.getInfo(id);
        if (flowEngineEntity != null) {
            String copyNum = UUID.randomUUID().toString().substring(0, 5);
            flowEngineEntity.setFullName(flowEngineEntity.getFullName() + ".副本" + copyNum);
            flowEngineEntity.setEnCode(flowEngineEntity.getEnCode() + copyNum);
            flowEngineEntity.setCreatorTime(new Date());
            flowEngineEntity.setId(null);
            if (flowEngineEntity.getFormType() != 1) {
                List<TableModel> tableModelList = JsonUtil.getJsonToList(flowEngineEntity.getFlowTables(), TableModel.class);
                if (tableModelList.size() == 0) {
                    throw new WorkFlowException(MsgCode.WF008.get());
                }
            }
            flowEngineService.copy(flowEngineEntity);
            return ActionResult.success(MsgCode.SU007.get());
        }
        return ActionResult.fail(MsgCode.FA004.get());
    }

    /**
     * 流程表单状态
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("更新流程表单状态")
    @PutMapping("/{id}/Actions/State")
    public ActionResult state(@PathVariable("id") String id) throws WorkFlowException {
        FlowEngineEntity entity = flowEngineService.getInfo(id);
        if (entity != null) {
            entity.setEnabledMark("1".equals(String.valueOf(entity.getEnabledMark())) ? 0 : 1);
            flowEngineService.update(id, entity);
            return ActionResult.success("更新表单成功");
        }
        return ActionResult.fail(MsgCode.FA002.get());
    }

    /**
     * 发布流程引擎
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("发布流程设计")
    @PostMapping("/Release/{id}")
    public ActionResult release(@PathVariable("id") String id) throws WorkFlowException {
        FlowEngineEntity entity = flowEngineService.getInfo(id);
        if (entity != null) {
            entity.setEnabledMark(1);
            flowEngineService.update(id, entity);
            return ActionResult.success(MsgCode.SU011.get());
        }
        return ActionResult.fail(MsgCode.FA011.get());
    }

    /**
     * 停止流程引擎
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("停止流程设计")
    @PostMapping("/Stop/{id}")
    public ActionResult stop(@PathVariable("id") String id) throws WorkFlowException {
        FlowEngineEntity entity = flowEngineService.getInfo(id);
        if (entity != null) {
            entity.setEnabledMark(0);
            flowEngineService.update(id, entity);
            return ActionResult.success(MsgCode.SU008.get());
        }
        return ActionResult.fail(MsgCode.FA008.get());
    }

    /**
     * 工作流导出
     *
     * @param id 主键值
     * @return
     * @throws WorkFlowException
     */
    @ApiOperation("工作流导出")
    @GetMapping("/{id}/Actions/ExportData")
    public ActionResult exportData(@PathVariable("id") String id) throws WorkFlowException {
        FlowExportModel model = flowEngineService.exportData(id);
        DownloadVO downloadVO = fileExport.exportFile(model, configValueUtil.getTemporaryFilePath(), model.getFlowEngine().getFullName(), ModuleTypeEnum.FLOW_FLOWENGINE.getTableName());
        return ActionResult.success(downloadVO);
    }

    /**
     * 工作流导入
     *
     * @param multipartFile 文件
     * @return
     * @throws WorkFlowException
     */
    @ApiOperation("工作流导入")
    @PostMapping(value = "/Actions/ImportData", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public ActionResult ImportData(@RequestPart("file") MultipartFile multipartFile) throws WorkFlowException {
        //判断是否为.json结尾
        if (FileUtil.existsSuffix(multipartFile, ModuleTypeEnum.FLOW_FLOWENGINE.getTableName())) {
            return ActionResult.fail(MsgCode.IMP002.get());
        }
        //获取文件内容
        String fileContent = FileUtil.getFileContent(multipartFile, configValueUtil.getTemporaryFilePath());
        FlowExportModel vo = JsonUtil.getJsonToBean(fileContent, FlowExportModel.class);
        return flowEngineService.ImportData(vo.getFlowEngine(), vo.getVisibleList());
    }

    /**
     * 获取流程的本地任务列表
     */
    @ApiOperation("获取流程任务调度列表")
    @GetMapping("/FlowTaskMethods")
    public ActionResult<List<TaskMethodsVO>> FlowTaskMethods() {
        List<TaskMethodsVO> list = CollectionUtil.list(false);
        String[] beanNames = applicationContext.getBeanDefinitionNames();
        for (String beanName : beanNames) {
            // 得到Bean信息
            Object obj = applicationContext.getBean(beanName);
            Class clz = obj.getClass();
            if (AopUtils.isAopProxy(obj)) {
                if (ObjectUtil.isNotNull(AopProxyUtils.getSingletonTarget(obj))) {
                    clz = AopProxyUtils.getSingletonTarget(obj).getClass();
                }
            }
            // 得到类中的所有方法
            Method[] methods = clz.getDeclaredMethods();
            for (Method method : methods) {
                FlowCallBackMethod FlowAnnotation = method.getAnnotation(FlowCallBackMethod.class);
                if (FlowAnnotation != null) {
                    // 默认id生成规则
                    String ids = clz.getName() + "." + method.getName();
                    String id = StringUtils.isNotEmpty(FlowAnnotation.id()) ? FlowAnnotation.id() : ids;
                    String name = FlowAnnotation.fullName();
                    String description = FlowAnnotation.description();
                    // 封装调用对象
                    LocalTaskModel model = new LocalTaskModel();
                    model.setFullName(name);
                    model.setClz(obj);
                    model.setMethodName(method.getName());
                    model.setDescription(description);
                    model.setId(id);
                    model.setJobGroupName("jsbos");
                    TaskMethodsVO taskMethodsVO = new TaskMethodsVO();
                    taskMethodsVO.setId(model.getId());
                    taskMethodsVO.setFullName(model.getFullName());
                    taskMethodsVO.setDescription(model.getDescription());
                    list.add(taskMethodsVO);
                }
            }
        }
        return ActionResult.success(list);

    }

}