package com.bringspring.workflow.engine.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.UserProvider;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.workflow.engine.entity.FlowDelegateEntity;
import com.bringspring.workflow.engine.model.flowdelegate.FlowDelegatListVO;
import com.bringspring.workflow.engine.model.flowdelegate.FlowDelegateCrForm;
import com.bringspring.workflow.engine.model.flowdelegate.FlowDelegateInfoVO;
import com.bringspring.workflow.engine.model.flowdelegate.FlowDelegateUpForm;
import com.bringspring.workflow.engine.service.FlowDelegateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;
import java.util.List;

/**
 * 流程委托
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Api(tags = "流程委托", value = "FlowDelegate")
@RestController
@RequestMapping("/api/workflow/Engine/FlowDelegate")
public class FlowDelegateController {

    @Autowired
    private FlowDelegateService flowDelegateService;
    @Autowired
    private UserProvider userProvider;

    /**
     * 获取流程委托列表
     *
     * @param pagination
     * @return
     */
    @ApiOperation("获取流程委托列表")
    @GetMapping
    public ActionResult<PageListVO<FlowDelegatListVO>> list(Pagination pagination) {
        List<FlowDelegateEntity> list = flowDelegateService.getList(pagination);
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        List<FlowDelegatListVO> listVO = JsonUtil.getJsonToList(list, FlowDelegatListVO.class);
        return ActionResult.page(listVO, paginationVO);
    }

    /**
     * 获取流程委托信息
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("获取流程委托信息")
    @GetMapping("/{id}")
    public ActionResult<FlowDelegateInfoVO> info(@PathVariable("id") String id) throws DataException {
        FlowDelegateEntity entity = flowDelegateService.getInfo(id);
        FlowDelegateInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, FlowDelegateInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 新建流程委托
     *
     * @param flowDelegateCrForm 实体对象
     * @return
     */
    @ApiOperation("新建流程委托")
    @PostMapping
    public ActionResult create(@RequestBody @Valid FlowDelegateCrForm flowDelegateCrForm) {
        FlowDelegateEntity entity = JsonUtil.getJsonToBean(flowDelegateCrForm, FlowDelegateEntity.class);
        UserInfo userInfo = userProvider.get();
        if(userInfo.getUserId().equals(entity.getToUserid())){
            return ActionResult.fail("委托人为自己，委托失败");
        }
        flowDelegateService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新流程委托
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("更新流程委托")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid FlowDelegateUpForm flowDelegateUpForm) {
        FlowDelegateEntity entity = JsonUtil.getJsonToBean(flowDelegateUpForm, FlowDelegateEntity.class);
        UserInfo userInfo = userProvider.get();
        if(userInfo.getUserId().equals(entity.getToUserid())){
            return ActionResult.fail("委托人为自己，委托失败");
        }
        boolean flag = flowDelegateService.update(id, entity);
        if (flag == false) {
            return ActionResult.success(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除流程委托
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("删除流程委托")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        FlowDelegateEntity entity = flowDelegateService.getInfo(id);
        if (entity != null) {
            flowDelegateService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }
}
